package io.gamedock.sdk.gamedata.packages;

import android.content.Context;
import android.content.SharedPreferences;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.reflect.TypeToken;

import org.json.JSONArray;
import org.json.JSONException;

import java.lang.reflect.Type;
import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.models.gamedata.packages.Package;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class that handles all the logic regarding IAP Packages information of the SDK.
 * It processes the IAP Packages from a JSON format and saves it in the shared preferences for later retrieval via Unity/Native App.
 */
public class PackageManager {

    private Type packagesTypeToken = new TypeToken<ArrayList<Package>>() {
    }.getType();

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "virtualGoods";

    private static volatile PackageManager mInstance = null;
    private Context context;

    private Gson gson;
    private StorageUtil storageUtil;

    private PackageManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
        storageUtil = GamedockSDK.getInstance(context).getStorageUtil();

    }

    public static PackageManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new PackageManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to process the packages configuration response from the backend.
     *
     * @param packagesArrayJSON The packages configuration JSON.
     */
    public void processPackageJSON(JSONArray packagesArrayJSON) {
        if (packagesArrayJSON != null) {
            try {
                ArrayList<Package> packages = new ArrayList<>();
                JsonParser jsonParser = new JsonParser();
                for (int i = 0; i < packagesArrayJSON.length(); i++) {
                    Package packageObj = gson.fromJson(packagesArrayJSON.getJSONObject(i).toString(), Package.class);
                    if (packagesArrayJSON.getJSONObject(i).has("properties")) {
                        JsonObject properties = (JsonObject) jsonParser.parse(packagesArrayJSON.getJSONObject(i).getJSONObject("properties").toString());
                        packageObj.setProperties(properties);
                    }
                    packages.add(packageObj);
                }

                storageUtil.putString(StorageUtil.Keys.GamePackages, gson.toJson(packages));

                GamedockSDK.getInstance(context).getPackagesCallbacks().packagesAvailable(gson.toJson(packages));
            } catch (JSONException e) {
                e.printStackTrace();
                GamedockSDK.getInstance(context).getPackagesCallbacks().packagesNotAvailable();
            }
        } else {
            GamedockSDK.getInstance(context).getPackagesCallbacks().packagesNotAvailable();
        }
    }

    /**
     * Method that retrieves IAP Package information given a Package Id.
     * The information is stored in the {@link SharedPreferences}.
     *
     * @param packageId The id of the package as defined in the Gamedock backend that needs to be retrieved.
     * @return Returns a JSON {@link String} representation of the requested IAP Package.
     */
    public String getPackage(String packageId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        String packagesString = storageUtil.getString(StorageUtil.Keys.GamePackages, null);

        if (packagesString == null) {
            GamedockSDK.getInstance(context).getPackagesCallbacks().packagesNotAvailable();
            return null;
        }

        ArrayList<Package> packages = gson.fromJson(packagesString, packagesTypeToken);

        for (Package packageEntry : packages) {
            if (packageEntry.getPackageId().equals(packageId)) {
                return gson.toJson(packageEntry);
            }
        }

        return null;
    }

    /**
     * Method used to retrieve a Package object base on a supplied package id (Google Id).
     *
     * @param packageId The Package Google id.
     * @return The object that was requested or null if no package is found.
     */
    public Package getPackageObject(String packageId) {
        return gson.fromJson(getPackage(packageId), Package.class);
    }

    /**
     * Method use to retrieve a Package JSON string base on a supplied id.
     *
     * @param id The Package id.
     * @return The JSON string that was requested or null if no package is found.
     */
    public String getPackage(int id) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        String packagesString = storageUtil.getString(StorageUtil.Keys.GamePackages, null);

        if (packagesString == null) {
            GamedockSDK.getInstance(context).getPackagesCallbacks().packagesNotAvailable();
            return null;
        }

        ArrayList<Package> packages = gson.fromJson(packagesString, packagesTypeToken);

        for (Package packageEntry : packages) {
            if (packageEntry.getId() == id) {
                return gson.toJson(packageEntry);
            }
        }

        return null;
    }

    /**
     * Method used to retrieve a Package object base on a supplied package id (Gamedock Id).
     *
     * @param id The Package Gamedock backend id.
     * @return The object that was requested or null if no package is found.
     */
    public Package getPackageObject(int id) {
        return gson.fromJson(getPackage(id), Package.class);
    }

    /**
     * Method that retrieves all the IAP Packages that have been received from the Gamedock backend.
     * The information is stored in the {@link SharedPreferences}.
     *
     * @return Returns a JSON {@link String} representation of all the IAP Package information.
     */
    public String getAllPackages() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return "[]";
        }

        String packagesString = storageUtil.getString(StorageUtil.Keys.GamePackages, "[]");

        if (packagesString == null) {
            GamedockSDK.getInstance(context).getPackagesCallbacks().packagesNotAvailable();
            return "[]";
        }

        return packagesString;
    }

    public void resetPackages() {
        GamedockSDK.getInstance(context).getStorageUtil().remove(StorageUtil.Keys.GamePackages);
    }
}
