package io.gamedock.sdk.gamedata;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import com.google.gson.Gson;

import org.json.JSONArray;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.GameDataEvent;
import io.gamedock.sdk.gamedata.packages.PackageManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.bundles.Bundle;
import io.gamedock.sdk.models.gamedata.currencies.Currency;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.gamedata.shop.Tab;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class that handles all the logic regarding game data information of the SDK.
 * It processes the game data from a JSON format and saves it in the shared preferences for later retrieval via Unity/Native App.
 * Loads a default game data information file stored in the assets folder.
 */
public class GamedockGameDataManager {
    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "virtualGoods";

    private static volatile GamedockGameDataManager mInstance = null;
    private Context context;

    private GamedockGameData gamedockGameData = null;

    private final Gson gson;
    private final StorageUtil storageUtil;
    private final FileAssetsUtil fileAssetsUtil;


    public boolean loadFailedFired;

    private GamedockGameDataManager(Context context, Gson gson, StorageUtil storageUtil, FileAssetsUtil fileAssetsUtil) {
        this.context = context;
        this.gson = gson;
        this.storageUtil = storageUtil;
        this.fileAssetsUtil = fileAssetsUtil;
    }

    public static GamedockGameDataManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new GamedockGameDataManager(context, GamedockSDK.getInstance(context).getGson(), GamedockSDK.getInstance(context).getStorageUtil(), GamedockSDK.getFileAssetsUtil());
                }
            }
        }
        return mInstance;
    }

    /**
     * Method that requests the game data from the Gamedock backend.
     */
    public void requestGameData() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        GameDataEvent gameDataEvent = new GameDataEvent(context);
        gameDataEvent.setRequestGameData();

        GamedockSDK.getInstance(context).trackEvent(gameDataEvent, null);
    }

    /**
     * Method used to process the information received from the server for the Gamedock Game Data (currencies, items, bundles, shop and promotions).
     *
     * @param currencies A list of all the currencies present in the game as received from the Gamedock backend.
     * @param items      A list of all the items present in the game as received from the Gamedock backend.
     * @param bundles    A list of all the bundles present in the game as received from the Gamedock backend.
     * @param shop       A list of all the shop configurations present in the game as received from the Gamedock backend.
     */
    public void processGameData(HashMap<Integer, Currency> currencies, HashMap<Integer, Item> items, HashMap<Integer, Bundle> bundles, ArrayList<Tab> shop, JSONArray packages) {

        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            gamedockGameData.setCurrenciesMap(currencies);
            gamedockGameData.setItemsMap(items);
            gamedockGameData.setBundlesMap(bundles);
            gamedockGameData.setShop(shop);

            try {
                // Check if currencies have changed in Gamedock, add/remove updated currencies from the wallet.
                // Add new currencies
                Wallet wallet = UserDataManager.getInstance(context).getUserData().getWallet();
                for (Map.Entry<Integer, Currency> gameDataCurrency : gamedockGameData.getCurrenciesMap().entrySet()) {
                    if (!wallet.getCurrenciesMap().containsKey(gameDataCurrency.getKey())) {
                        wallet.getCurrenciesMap().put(gameDataCurrency.getValue().getId(), new PlayerCurrency(gameDataCurrency.getValue()));
                    }
                }

                // Check if currencies in wallet should be removed
                ArrayList<Integer> currenciesToRemove = new ArrayList<>();

                for (Map.Entry<Integer, PlayerCurrency> walletCurrency : wallet.getCurrenciesMap().entrySet()) {
                    if (!gamedockGameData.getCurrenciesMap().containsKey(walletCurrency.getKey())) {
                        currenciesToRemove.add(walletCurrency.getKey());
                    }
                }

                for (Integer currencyToRemove : currenciesToRemove) {
                    wallet.getCurrenciesMap().remove(currencyToRemove);
                }
            } catch (ClassCastException e) {
                e.printStackTrace();
            }

            updateGameData(gamedockGameData);
            this.gamedockGameData = gamedockGameData;

            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataAvailable(getGameDataJSON());
        }

        PackageManager.getInstance(context).processPackageJSON(packages);
    }

    /**
     * Retrieves the game data from either the shared prefs or the local JSON file.
     *
     * @return Returns the {@link GamedockGameData} object containing all the game data information.
     */
    public GamedockGameData getGameData() {
        if (gamedockGameData != null) {
            return gamedockGameData;
        } else {
            String gameObjects = storageUtil.getString(StorageUtil.Keys.SpilGameData, null);
            if (gameObjects != null) {
                gamedockGameData = gson.fromJson(gameObjects, GamedockGameData.class);
                return gamedockGameData;
            } else {
                gameObjects = loadGameDataFromAssets(context, fileAssetsUtil);

                if (gameObjects != null && gameObjects.length() > 0) {
                    gamedockGameData = gson.fromJson(gameObjects, GamedockGameData.class);
                    gamedockGameData.Build();
                    storageUtil.putString(StorageUtil.Keys.SpilGameData, gson.toJson(gamedockGameData));
                    return gamedockGameData;
                } else {
                    if (!loadFailedFired) {
                        GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
                        loadFailedFired = true;
                    }
                    return null;
                }
            }
        }
    }

    /**
     * Retrieves the game data in string format from either the shared prefs or the local JSON file.
     *
     * @return Returns the {@link GamedockGameData} as a {@link String} JSON.
     */
    public String getGameDataJSON() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        if (gamedockGameData != null) {
            return gamedockGameData.Build(getGameData());
        }

        return null;
    }

    /**
     * Retrieves the currency based on id.
     *
     * @param currencyId The currency id for which the {@link Currency} should be retrieved.
     * @return Return a {@link Currency} object with the requested id.
     */
    public Currency getCurrency(int currencyId) {

        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            return gamedockGameData.getCurrenciesMap().get(currencyId);
        } else {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.CurrencyNotFound);
            return null;
        }
    }

    /**
     * Retrieves an item based on a specified id.
     *
     * @param itemId The item id for which the {@link Item} should be retrieved.
     * @return Return a {@link Item} object with the requested id.
     */
    public Item getItem(int itemId) {

        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            return gamedockGameData.getItemsMap().get(itemId);
        } else {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.ItemNotFound);
            return null;
        }
    }

    /**
     * Retrieves a gacha item based on a specific id.
     *
     * @param gachaId The gacha id for which the Gacha {@link Item} should be retrieved.
     * @return Return a Gacha {@link Item} object with the requested id.
     */
    public Item getGacha(int gachaId) {
        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            Item item = gamedockGameData.getItemsMap().get(gachaId);

            if (item != null && item.isGacha()) {
                return item;
            } else {
                GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.GachaNotFound);
                return null;
            }
        } else {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.GachaNotFound);
            return null;
        }
    }

    /**
     * Method used to retrieve all gachas.
     *
     * @return The list of gacha items.
     */
    public ArrayList<Item> getAllGachas() {
        ArrayList<Item> gachas = new ArrayList<>();
        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            for (Item item : gamedockGameData.getItemsMap().values()) {
                if (item.isGacha()) {
                    gachas.add(item);
                }
            }
            return gachas;
        } else {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return null;
        }
    }

    /**
     * Retrieves a bundle based on a specified id.
     *
     * @param bundleId The bundle id for which the {@link Bundle} should be retrieved.
     * @return Return a {@link Bundle} object with the requested id.
     */
    public Bundle getBundle(int bundleId) {

        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData != null) {
            return gamedockGameData.getBundlesMap().get(bundleId);
        } else {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.BundleNotFound);
            return null;
        }
    }

    /**
     * Retrieves the shop information from the game data.
     *
     * @return The JSON {@link String} for the Shop configuration.
     */
    public String getShop() {

        GamedockGameData gamedockGameData = getGameData();

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return null;
        }

        for (int i = 0; i < gamedockGameData.getShop().size(); i++) {
            Tab tab = gamedockGameData.getShop().get(i);

            for (int j = 0; j < tab.getEntries().size(); j++) {
                if (!gamedockGameData.getBundlesMap().containsKey(tab.getEntries().get(j).getId())) {
                    GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
                    requestGameData();
                    return null;
                }
            }
        }

        return gson.toJson(gamedockGameData.getShop());
    }

    /**
     * Method used to set a limit on a certain currency. If the limit is reached an overflow value will be sent to the backend.
     *
     * @param currencyId The id of the currency for which the limit is set.
     * @param limit      The limit value.
     */
    public void setCurrencyLimit(int currencyId, int limit) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        GamedockGameData gamedockGameData = getGameData();
        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.WalletNotFound);
            return;
        }

        if (!gamedockGameData.getCurrenciesMap().containsKey(currencyId)) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.CurrencyNotFound);
            return;
        }

        gamedockGameData.getCurrenciesMap().get(currencyId).setLimit(limit);
        updateGameData(gamedockGameData);

        if (!userData.getWallet().getCurrenciesMap().containsKey(currencyId)) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.CurrencyNotFound);
            return;
        }

        userData.getWallet().getCurrenciesMap().get(currencyId).setLimit(limit);
        UserDataManager.getInstance(context).updateUserData(userData);

        GameDataEvent gameDataEvent = new GameDataEvent(context);
        gameDataEvent.setSetCurrencyLimit();

        gameDataEvent.addCustomData("id", currencyId);
        gameDataEvent.addCustomData("name", userData.getWallet().getCurrenciesMap().get(currencyId).getName());
        gameDataEvent.addCustomData("limit", limit);
    }

    /**
     * Method used to set a limit on a certain item. If the limit is reached an overflow value will be sent to the backend.
     *
     * @param itemId The id of the item for which the limit is set.
     * @param limit  The limit value.
     */
    public void setItemLimit(int itemId, int limit) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        GamedockGameData gamedockGameData = getGameData();
        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.InventoryNotFound);
            return;
        }

        if (!gamedockGameData.getItemsMap().containsKey(itemId)) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.ItemNotFound);
            return;
        }

        getGameData().getItemsMap().get(itemId).setLimit(limit);
        updateGameData(gamedockGameData);

        if (!userData.getInventory().getItemsMap().containsKey(itemId)) {
            return;
        }

        userData.getInventory().getItemsMap().get(itemId).setLimit(limit);
        UserDataManager.getInstance(context).updateUserData(userData);

        GameDataEvent gameDataEvent = new GameDataEvent(context);
        gameDataEvent.setSetItemLimit();

        gameDataEvent.addCustomData("id", itemId);
        gameDataEvent.addCustomData("name", userData.getInventory().getItemsMap().get(itemId).getName());
        gameDataEvent.addCustomData("limit", limit);
    }

    /**
     * Method that updates the game data by storing it in the shared preferences.
     *
     * @param gamedockGameData The {@link GamedockGameData} object that needs to be updated.
     */
    public void updateGameData(GamedockGameData gamedockGameData) {

        storageUtil.putString(StorageUtil.Keys.SpilGameData, gson.toJson(gamedockGameData));

        this.gamedockGameData = gamedockGameData;
    }

    /**
     * Retrieves Player Data from the JSON file called "defaultGameData.json".
     *
     * @return The JSON String of the game data containing all the information.
     */
    @VisibleForTesting
    private String loadGameDataFromAssets(Context context, FileAssetsUtil fileAssetsUtil) {
        String json = null;

        try {
            InputStream is = fileAssetsUtil.getGameDataFileAsset(context);
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

        } catch (IOException ex) {
            LoggingUtil.e("The 'defaultGameData.json' file is missing from your assets folder. If you want to use the Wallet/Inventory/Shop functionality please include this file.");
        }

        return json;
    }

    public void resetGameData() {
        gamedockGameData = null;
        storageUtil.remove(StorageUtil.Keys.SpilGameData);
        mInstance = null;
    }
}
