package io.gamedock.sdk.events;

import android.content.Context;

import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.models.performance.PerformanceMetric;
import io.gamedock.sdk.models.tracking.TrackingCurrency;
import io.gamedock.sdk.models.tracking.TrackingItem;

public class GamedockTracking {
    public static abstract class BaseTracking {
        protected String eventName;
        protected JSONObject parameters = new JSONObject();
        protected EventActionListener eventActionListener = null;
        private Gson gson = new Gson();

        public BaseTracking addCustomParameter(String key, Object value) {
            try {
                parameters.put(key, value);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseTracking addTransactionId(String transactionId) {
            try {
                parameters.put("transactionId", transactionId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseTracking addEventListener(EventActionListener eventActionListener) {
            this.eventActionListener = eventActionListener;
            return this;
        }

        public void track(Context context) {
            GamedockSDK.getInstance(context).trackSpecialEvent(eventName, parameters, eventActionListener);
        }
    }

    public static class BaseCustomEvent extends BaseTracking {
        public BaseCustomEvent(String eventName) {
            this.eventName = eventName;
        }
    }

    public static BaseCustomEvent CustomEvent(String eventName) {
        return new BaseCustomEvent(eventName);
    }


    public static class BaseNotificationSent extends BaseTracking {
        public BaseNotificationSent(String uniqueNotificationId) {
            eventName = "notificationSent";
            try {
                parameters.put("uniqueNotificationId", uniqueNotificationId);
                parameters.put("type", "local");
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseNotificationSent NotificationSent(String uniqueNotificationId) {
        return new BaseNotificationSent(uniqueNotificationId);
    }

    public static class BaseNotificationReceived extends BaseTracking {
        public BaseNotificationReceived(String uniqueNotificationId) {
            eventName = "notificationReceived";
            try {
                parameters.put("uniqueNotificationId", uniqueNotificationId);
                parameters.put("type", "local");
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseNotificationReceived NotificationReceived(String uniqueNotificationId) {
        return new BaseNotificationReceived(uniqueNotificationId);
    }

    public static class BaseNotificationOpened extends BaseTracking {
        public BaseNotificationOpened(String uniqueNotificationId, boolean didLaunchApp) {
            eventName = "notificationOpened";
            try {
                parameters.put("uniqueNotificationId", uniqueNotificationId);
                parameters.put("didLaunchApp", didLaunchApp);
                parameters.put("type", "local");
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseNotificationOpened NotificationOpened(String uniqueNotificationId, boolean didLaunchApp) {
        return new BaseNotificationOpened(uniqueNotificationId, didLaunchApp);
    }

    public static class BaseNotificationDismissed extends BaseTracking {
        public BaseNotificationDismissed(String uniqueNotificationId) {
            eventName = "notificationDismissed";

            try {
                parameters.put("uniqueNotificationId", uniqueNotificationId);
                parameters.put("type", "local");
            } catch (JSONException e) {
                e.printStackTrace();
            }

        }
    }

    public static BaseNotificationDismissed NotificationDismissed(String uniqueNotificationId) {
        return new BaseNotificationDismissed(uniqueNotificationId);
    }

    public static class BaseMilestoneAchieved extends BaseTracking {
        public BaseMilestoneAchieved(String name) {
            eventName = "milestoneAchieved";
            try {
                parameters.put("name", name);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseMilestoneAchieved addMilestoneDescription(String description) {
            try {
                parameters.put("description", description);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMilestoneAchieved addScore(float score) {
            try {
                parameters.put("score", score);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMilestoneAchieved addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMilestoneAchieved addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseMilestoneAchieved MilestoneAchieved(String name) {
        return new BaseMilestoneAchieved(name);
    }

    public static class BaseLevelStart extends BaseTracking {
        public BaseLevelStart(String level) {
            eventName = "levelStart";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelStart addLevelId(String levelId) {
            try {
                parameters.put("levelId", levelId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addDifficulty(String difficulty) {
            try {
                parameters.put("difficulty", difficulty);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addCustomCreated(boolean customCreated) {
            try {
                parameters.put("customCreated", customCreated);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addCreatorId(String creatorId) {
            try {
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addActiveBooster(ArrayList<String> activeBooster) {
            try {
                parameters.put("activeBooster", new JSONArray(super.gson.toJson(activeBooster)));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addAchievement(String achievement) {
            try {
                parameters.put("achievement", achievement);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelStart addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelStart LevelStart(String level) {
        return new BaseLevelStart(level);
    }

    public static class BaseLevelComplete extends BaseTracking {
        public BaseLevelComplete(String level) {
            eventName = "levelComplete";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelComplete addLevelId(String levelId) {
            try {
                parameters.put("levelId", levelId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addDifficulty(String difficulty) {
            try {
                parameters.put("difficulty", difficulty);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addScore(float score) {
            try {
                parameters.put("score", score);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addStars(int stars) {
            try {
                parameters.put("stars", stars);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addSpeed(int speed) {
            try {
                parameters.put("speed", speed);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addMoves(int moves) {
            try {
                parameters.put("moves", moves);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addTurns(int turns) {
            try {
                parameters.put("turns", turns);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addCustomCreated(boolean customCreated) {
            try {
                parameters.put("customCreated", customCreated);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addCreatorId(String creatorId) {
            try {
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addObjectUsed(ArrayList<Object> objectUsed) {
            try {
                parameters.put("objectUsed", new JSONArray(super.gson.toJson(objectUsed)));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addAchievement(String achievement) {
            try {
                parameters.put("achievement", achievement);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addAvgCombos(float avgCombos) {
            try {
                parameters.put("avgCombos", avgCombos);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addMovesLeft(int movesLeft) {
            try {
                parameters.put("movesLeft", movesLeft);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addRating(String rating) {
            try {
                parameters.put("rating", rating);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addTimeLeft(int timeLeft) {
            try {
                parameters.put("timeLeft", timeLeft);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelComplete addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelComplete LevelComplete(String level) {
        return new BaseLevelComplete(level);
    }

    public static class BaseLevelFailed extends BaseTracking {
        public BaseLevelFailed(String level) {
            eventName = "levelFailed";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelFailed addLevelId(String levelId) {
            try {
                parameters.put("levelId", levelId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addDifficulty(String difficulty) {
            try {
                parameters.put("difficulty", difficulty);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addScore(float score) {
            try {
                parameters.put("score", score);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addStars(int stars) {
            try {
                parameters.put("stars", stars);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addSpeed(int speed) {
            try {
                parameters.put("speed", speed);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addMoves(int moves) {
            try {
                parameters.put("moves", moves);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addTurns(int turns) {
            try {
                parameters.put("turns", turns);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addCustomCreated(boolean customCreated) {
            try {
                parameters.put("customCreated", customCreated);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addCreatorId(String creatorId) {
            try {
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addObjectUsed(ArrayList<Object> objectUsed) {
            try {
                parameters.put("objectUsed", new JSONArray(super.gson.toJson(objectUsed)));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addAchievement(String achievement) {
            try {
                parameters.put("achievement", achievement);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addAvgCombos(float avgCombos) {
            try {
                parameters.put("avgCombos", avgCombos);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addMovesLeft(int movesLeft) {
            try {
                parameters.put("movesLeft", movesLeft);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addRating(String rating) {
            try {
                parameters.put("rating", rating);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addTimeLeft(int timeLeft) {
            try {
                parameters.put("timeLeft", timeLeft);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelFailed addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelFailed LevelFailed(String level) {
        return new BaseLevelFailed(level);
    }

    public static class BaseLevelUp extends BaseTracking {
        public BaseLevelUp(String level, String objectId) {
            eventName = "levelUp";
            try {
                parameters.put("level", level);
                parameters.put("objectId", objectId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelUp addSkillId(String skillId) {
            try {
                parameters.put("skillId", skillId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelUp addSourceId(String sourceId) {
            try {
                parameters.put("sourceId", sourceId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelUp addSourceUniqueId(String sourceUniqueId) {
            try {
                parameters.put("sourceUniqueId", sourceUniqueId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelUp addObjectUniqueId(String objectUniqueId) {
            try {
                parameters.put("objectUniqueId", objectUniqueId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseLevelUp addObjectUniqueIdType(String objectUniqueIdType) {
            try {
                parameters.put("objectUniqueIdType", objectUniqueIdType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelUp LevelUp(String level, String objectId) {
        return new BaseLevelUp(level, objectId);
    }

    public static class BaseEquip extends BaseTracking {
        public BaseEquip(String equippedItem) {
            eventName = "equip";
            try {
                parameters.put("equippedItem", equippedItem);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseEquip addEquippedTo(String equippedTo) {
            try {
                parameters.put("equippedTo", equippedTo);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseEquip addUnequippedFrom(String unequippedFrom) {
            try {
                parameters.put("unequippedFrom", unequippedFrom);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseEquip Equip(String equippedItem) {
        return new BaseEquip(equippedItem);
    }

    public static class BaseUpgrade extends BaseTracking {
        public BaseUpgrade(String upgradeId, String level) {
            eventName = "upgrade";
            try {
                parameters.put("upgradeId", upgradeId);
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseUpgrade addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUpgrade addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUpgrade addAchievement(String achievement) {
            try {
                parameters.put("achievement", achievement);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUpgrade addKey(String key) {
            try {
                parameters.put("key", key);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseUpgrade Upgrade(String upgradeId, String level) {
        return new BaseUpgrade(upgradeId, level);
    }

    public static class BaseLevelCreate extends BaseTracking {
        public BaseLevelCreate(String levelId, String level, String creatorId) {
            eventName = "levelCreate";
            try {
                parameters.put("levelId", levelId);
                parameters.put("level", level);
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseLevelCreate LevelCreate(String levelId, String level, String creatorId) {
        return new BaseLevelCreate(levelId, level, creatorId);
    }

    public static class BaseLevelDownload extends BaseTracking {
        public BaseLevelDownload(String levelId, String creatorId) {
            eventName = "levelDownload";
            try {
                parameters.put("levelId", levelId);
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelDownload addRating(float rating) {
            try {
                parameters.put("rating", rating);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelDownload LevelDownload(String levelId, String creatorId) {
        return new BaseLevelDownload(levelId, creatorId);
    }

    public static class BaseLevelRate extends BaseTracking {
        public BaseLevelRate(String levelId, String creatorId) {
            eventName = "levelRate";
            try {
                parameters.put("levelId", levelId);
                parameters.put("creatorId", creatorId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelRate addRating(float rating) {
            try {
                parameters.put("rating", rating);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelRate LevelRate(String levelId, String creatorId) {
        return new BaseLevelRate(levelId, creatorId);
    }

    public static class BaseEndlessModeStart extends BaseTracking {
        public BaseEndlessModeStart() {
            eventName = "endlessModeStart";
        }
    }

    public static BaseEndlessModeStart EndlessModeStart() {
        return new BaseEndlessModeStart();
    }

    public static class BaseEndlessModeEnd extends BaseTracking {
        public BaseEndlessModeEnd(int distance) {
            eventName = "endlessModeEnd";
            try {
                parameters.put("distance", distance);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseEndlessModeEnd EndlessModeEnd(int distance) {
        return new BaseEndlessModeEnd(distance);
    }

    public static class BasePlayerDies extends BaseTracking {
        public BasePlayerDies(String level) {
            eventName = "playerDies";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BasePlayerDies PlayerDies(String level) {
        return new BasePlayerDies(level);
    }

    public static class BaseWalletInventoryEvent extends BaseTracking {
        public BaseWalletInventoryEvent(String reason, String location) {
            eventName = "updatePlayerData";
            try {
                parameters.put("reason", reason);
                parameters.put("location", location);
                parameters.put("trackingOnly", true);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseWalletInventoryEvent addWallet(ArrayList<TrackingCurrency> currencyList) {
            HashMap<String, Object> wallet = new HashMap<>();
            try {
                wallet.put("currencies", new JSONArray(super.gson.toJson(currencyList)));
                wallet.put("offset", 0);
                parameters.put("wallet", wallet);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseWalletInventoryEvent addInventory(ArrayList<TrackingItem> itemsList) {
            HashMap<String, Object> inventory = new HashMap<>();
            try {
                inventory.put("items", new JSONArray(super.gson.toJson(itemsList)));
                inventory.put("offset", 0);
                parameters.put("inventory", inventory);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseWalletInventoryEvent addReasonDetails(String reasonDetails) {
            try {
                parameters.put("reasonDetails", reasonDetails);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseWalletInventoryEvent addTransactionId(String transactionId) {
            try {
                parameters.put("transactionId", transactionId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseWalletInventoryEvent WalletInventoryEvent(String reason, String location) {
        return new BaseWalletInventoryEvent(reason, location);
    }

    public static class BaseIAPPurchased extends BaseTracking {
        public BaseIAPPurchased(String skuId, String transactionId) {
            eventName = "iapPurchased";
            try {
                parameters.put("skuId", skuId);
                parameters.put("transactionId", transactionId);
                parameters.put("purchaseDate", new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZZZZZ", Locale.getDefault()).format(new Date()));
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseIAPPurchased addToken(String token) {
            if (!token.equals("")) {
                try {
                    parameters.put("token", token);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }

            return this;
        }

        public BaseIAPPurchased addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addReasonDetails(String reasonDetails) {
            try {
                parameters.put("reasonDetails", reasonDetails);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addLocalPrice(String localPrice) {
            if (localPrice != null && !localPrice.isEmpty()) {
                String newLocalPrice = "";
                for (char c : localPrice.toCharArray()) {
                    if (Character.isDigit(c) || c == '.' || c == ',') {
                        newLocalPrice += c;
                    }
                }

                localPrice = newLocalPrice;
            }

            try {
                parameters.put("localPrice", localPrice);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addLocalCurrency(String localCurrency) {
            try {
                parameters.put("localCurrency", localCurrency);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addAmazonUserId(String amazonUserId) {
            try {
                parameters.put("amazonUserId", amazonUserId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPPurchased addIsSubscription(boolean isSubscription) {
            try {
                parameters.put("isSubscription", isSubscription);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseIAPPurchased IAPPurchased(String skuId, String transactionId) {
        return new BaseIAPPurchased(skuId, transactionId);
    }

    public static class BaseIAPRestored extends BaseTracking {
        public BaseIAPRestored(String skuId, String originalTransactionId, String originalPurchaseDate) {
            eventName = "iapRestored";
            try {
                parameters.put("skuId", skuId);
                parameters.put("originalTransactionId", originalTransactionId);
                parameters.put("originalPurchaseDate", originalPurchaseDate);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseIAPRestored addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseIAPRestored IAPRestored(String skuId, String originalTransactionId, String originalPurchaseDate) {
        return new BaseIAPRestored(skuId, originalTransactionId, originalPurchaseDate);
    }

    public static class BaseIAPFailed extends BaseTracking {
        public BaseIAPFailed(String skuId, String errorDescription) {
            eventName = "iapFailed";
            try {
                parameters.put("skuId", skuId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            try {
                parameters.put("errorDescription", errorDescription);
                parameters.put("purchaseDate", new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZZZZZ", Locale.getDefault()).format(new Date()));
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseIAPFailed addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseIAPFailed addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseIAPFailed IAPFailed(String skuId, String errorDescription) {
        return new BaseIAPFailed(skuId, errorDescription);
    }

    public static class BaseTutorialComplete extends BaseTracking {
        public BaseTutorialComplete() {
            eventName = "tutorialComplete";
        }
    }

    public static BaseTutorialComplete TutorialComplete() {
        return new BaseTutorialComplete();
    }

    public static class BaseTutorialSkipped extends BaseTracking {
        public BaseTutorialSkipped() {
            eventName = "tutorialSkipped";
        }
    }

    public static BaseTutorialSkipped TutorialSkipped() {
        return new BaseTutorialSkipped();
    }

    public static class BaseRegister extends BaseTracking {
        public BaseRegister(String platform) {
            eventName = "register";
            try {
                parameters.put("platform", platform);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseRegister Register(String platform) {
        return new BaseRegister(platform);
    }

    public static class BaseShare extends BaseTracking {
        public BaseShare(String platform) {
            eventName = "share";
            try {
                parameters.put("platform", platform);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseShare addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseShare addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseShare Share(String platform) {
        return new BaseShare(platform);
    }

    public static class BaseInvite extends BaseTracking {
        public BaseInvite(String platform) {
            eventName = "invite";
            try {
                parameters.put("platform", platform);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseInvite addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseInvite Invite(String platform) {
        return new BaseInvite(platform);
    }

    public static class BaseLevelAppeared extends BaseTracking {
        public BaseLevelAppeared(String level) {
            eventName = "levelAppeared";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelAppeared addDifficulty(String difficulty) {
            try {
                parameters.put("difficulty", difficulty);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelAppeared LevelAppeared(String level) {
        return new BaseLevelAppeared(level);
    }

    public static class BaseLevelDiscarded extends BaseTracking {
        public BaseLevelDiscarded(String level) {
            eventName = "levelDiscarded";
            try {
                parameters.put("level", level);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseLevelDiscarded addDifficulty(String difficulty) {
            try {
                parameters.put("difficulty", difficulty);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseLevelDiscarded LevelDiscarded(String level) {
        return new BaseLevelDiscarded(level);
    }

    public static class BaseErrorShown extends BaseTracking {
        public BaseErrorShown(String reason) {
            eventName = "errorShown";
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseErrorShown ErrorShown(String reason) {
        return new BaseErrorShown(reason);
    }

    public static class BaseTimeElapLoad extends BaseTracking {
        public BaseTimeElapLoad(int timeElap, String pointInGame) {
            eventName = "timeElapLoad";
            try {
                parameters.put("timeElap", timeElap);
                parameters.put("pointInGame", pointInGame);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseTimeElapLoad addStartPoint(String startPoint) {
            try {
                parameters.put("startPoint", startPoint);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseTimeElapLoad TimeElapLoad(int timeElap, String pointInGame) {
        return new BaseTimeElapLoad(timeElap, pointInGame);
    }

    public static class BaseTimeoutDetected extends BaseTracking {
        public BaseTimeoutDetected(int timeElap, String pointInGame) {
            eventName = "timeoutDetected";
            try {
                parameters.put("timeElap", timeElap);
                parameters.put("pointInGame", pointInGame);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseTimeoutDetected TimeoutDetected(int timeElap, String pointInGame) {
        return new BaseTimeoutDetected(timeElap, pointInGame);
    }

    public static class BaseObjectStateChanged extends BaseTracking {
        public BaseObjectStateChanged(String changedObject, String status, String reason) {
            eventName = "objectStateChanged";
            try {
                parameters.put("changedObject", changedObject);
                parameters.put("status", status);
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseObjectStateChanged addChangedProperties(String changedProperties) {
            try {
                parameters.put("changedProperties", changedProperties);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseObjectStateChanged addOptionConditions(String optionConditions) {
            try {
                parameters.put("optionConditions", optionConditions);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseObjectStateChanged addSituation(String situation) {
            try {
                parameters.put("situation", situation);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseObjectStateChanged addAllChoiceResults(String allChoiceResults) {
            try {
                parameters.put("allChoiceResults", allChoiceResults);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseObjectStateChanged addAllSelectedChoices(String allSelectedChoices) {
            try {
                parameters.put("allSelectedChoices", allSelectedChoices);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseObjectStateChanged addInvolvedParties(String involvedParties) {
            try {
                parameters.put("involvedParties", involvedParties);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseObjectStateChanged ObjectStateChanged(String changedObject, String status, String reason) {
        return new BaseObjectStateChanged(changedObject, changedObject, reason);
    }

    public static class BaseUIElementClicked extends BaseTracking {
        public BaseUIElementClicked(String element) {
            eventName = "uiElementClicked";
            try {
                parameters.put("element", element);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseUIElementClicked addType(String type) {
            try {
                parameters.put("type", type);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUIElementClicked addScreenName(String screenName) {
            try {
                parameters.put("screenName", screenName);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUIElementClicked addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseUIElementClicked addGrade(int grade) {
            try {
                parameters.put("grade", grade);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseUIElementClicked UIElementClicked(String element) {
        return new BaseUIElementClicked(element);
    }

    public static class BaseSendGift extends BaseTracking {
        public BaseSendGift(String platform) {
            eventName = "sendGift";
            try {
                parameters.put("platform", platform);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseSendGift addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseSendGift SendGift(String platform) {
        return new BaseSendGift(platform);
    }

    public static class BaseLevelTimeOut extends BaseTracking {
        public BaseLevelTimeOut() {
            eventName = "levelTimeOut";
        }
    }

    public static BaseLevelTimeOut LevelTimeOut() {
        return new BaseLevelTimeOut();
    }

    public static class BaseDialogueChosen extends BaseTracking {
        public BaseDialogueChosen(String name, String choice, String choiceType, boolean hasToken, boolean isPremiumChoice, boolean isQuiz, boolean isForced, boolean isTimed) {
            eventName = "dialogueChosen";
            try {
                parameters.put("name", name);
                parameters.put("choice", choice);
                parameters.put("choiceType", choiceType);
                parameters.put("hasToken", hasToken);
                parameters.put("isPremiumChoice", isPremiumChoice);
                parameters.put("isQuiz", isQuiz);
                parameters.put("isForced", isForced);
                parameters.put("isTimed", isTimed);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseDialogueChosen addTime(int time) {
            try {
                parameters.put("time", time);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseDialogueChosen addIteration(int iteration) {
            try {
                parameters.put("iteration", iteration);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseDialogueChosen addIterationReason(ArrayList<String> iteration) {
            JSONArray iterationArray = null;
            try {
                iterationArray = new JSONArray(super.gson.toJson(iteration));
                parameters.put("iterationReason", iterationArray);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseDialogueChosen DialogueChosen(String name, String choice, String choiceType, boolean hasToken, boolean isPremiumChoice, boolean isQuiz, boolean isForced, boolean isTimed) {
        return new BaseDialogueChosen(name, choice, choiceType, hasToken, isPremiumChoice, isQuiz, isForced, isTimed);
    }

    public static class BaseFriendAdded extends BaseTracking {
        public BaseFriendAdded(String friend) {
            eventName = "friendAdded";
            try {
                parameters.put("friend", friend);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseFriendAdded addPlatform(String platform) {
            try {
                parameters.put("platform", platform);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseFriendAdded FriendAdded(String friend) {
        return new BaseFriendAdded(friend);
    }

    public static class BaseGameObjectInteraction extends BaseTracking {
        public BaseGameObjectInteraction() {
            eventName = "gameObjectInteraction";
        }
    }

    public static BaseGameObjectInteraction GameObjectInteraction() {
        return new BaseGameObjectInteraction();
    }

    public static class BaseGameResult extends BaseTracking {
        public BaseGameResult() {
            eventName = "gameResult";
        }

        public BaseGameResult addItemId(String itemId) {
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseGameResult addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseGameResult addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseGameResult addMatchId(String matchId) {
            try {
                parameters.put("matchId", matchId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseGameResult GameResult() {
        return new BaseGameResult();
    }

    public static class BaseItemCrafted extends BaseTracking {
        public BaseItemCrafted(String itemId) {
            eventName = "itemCrafted";
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseItemCrafted addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseItemCrafted ItemCrafted(String itemId) {
        return new BaseItemCrafted(itemId);
    }

    public static class BaseItemCreated extends BaseTracking {
        public BaseItemCreated(String itemId) {
            eventName = "itemCreated";
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseItemCreated addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseItemCreated ItemCreated(String itemId) {
        return new BaseItemCreated(itemId);
    }

    public static class BaseItemUpdated extends BaseTracking {
        public BaseItemUpdated(String content, String itemId) {
            eventName = "itemUpdated";
            try {
                parameters.put("content", content);
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseItemUpdated addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseItemUpdated ItemUpdated(String content, String itemId) {
        return new BaseItemUpdated(content, itemId);
    }

    public static class BaseDeckUpdated extends BaseTracking {
        public BaseDeckUpdated(String content, String itemId) {
            eventName = "deckUpdated";
            try {
                parameters.put("content", content);
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseDeckUpdated addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseDeckUpdated addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseDeckUpdated DeckUpdated(String content, String itemId) {
        return new BaseDeckUpdated(content, itemId);
    }

    public static class BaseMatchComplete extends BaseTracking {
        public BaseMatchComplete() {
            eventName = "matchComplete";
        }

        public BaseMatchComplete addMatchId(String matchId) {
            try {
                parameters.put("matchId", matchId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchComplete addItemId(String itemId) {
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchComplete addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchComplete addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseMatchComplete MatchComplete() {
        return new BaseMatchComplete();
    }

    public static class BaseMatchLost extends BaseTracking {
        public BaseMatchLost() {
            eventName = "matchLost";
        }

        public BaseMatchLost addMatchId(String matchId) {
            try {
                parameters.put("matchId", matchId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchLost addItemId(String itemId) {
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchLost addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchLost addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseMatchLost MatchLost() {
        return new BaseMatchLost();
    }

    public static class BaseMatchTie extends BaseTracking {
        public BaseMatchTie() {
            eventName = "matchTie";
        }

        public BaseMatchTie addMatchId(String matchId) {
            try {
                parameters.put("matchId", matchId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchTie addItemId(String itemId) {
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchTie addItemType(String itemType) {
            try {
                parameters.put("itemType", itemType);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchTie addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseMatchTie MatchTie() {
        return new BaseMatchTie();
    }

    public static class BaseMatchWon extends BaseTracking {
        public BaseMatchWon() {
            eventName = "matchWon";
        }

        public BaseMatchWon addMatchId(String matchId) {
            try {
                parameters.put("matchId", matchId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchWon addItemId(String itemId) {
            try {
                parameters.put("itemId", itemId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchWon addItemType(ArrayList<String> itemType) {
            try {
                parameters.put("itemType", new JSONArray(super.gson.toJson(itemType)));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseMatchWon addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseMatchWon MatchWon() {
        return new BaseMatchWon();
    }

    public static class BasePawnMoved extends BaseTracking {
        public BasePawnMoved(String name) {
            eventName = "pawnMoved";
            try {
                parameters.put("name", name);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BasePawnMoved addReason(String reason) {
            try {
                parameters.put("reason", reason);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addDelta(String delta) {
            try {
                parameters.put("delta", delta);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addEnergy(String energy) {
            try {
                parameters.put("energy", energy);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addKind(String kind) {
            try {
                parameters.put("kind", kind);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addLocation(String location) {
            try {
                parameters.put("location", location);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addRarity(String rarity) {
            try {
                parameters.put("rarity", rarity);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePawnMoved addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BasePawnMoved PawnMoved(String name) {
        return new BasePawnMoved(name);
    }

    public static class BasePlayerLeagueChanged extends BaseTracking {
        public BasePlayerLeagueChanged() {
            eventName = "playerLeagueChanged";
        }
    }

    public static BasePlayerLeagueChanged PlayerLeagueChanged() {
        return new BasePlayerLeagueChanged();
    }

    public static class BaseTimedAction extends BaseTracking {
        public BaseTimedAction(String timedAction) {
            eventName = "timedAction";
            try {
                parameters.put("timedAction", timedAction);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseTimedAction addLabel(String label) {
            try {
                parameters.put("label", label);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseTimedAction addTimedObject(String timedObject) {
            try {
                parameters.put("timedObject", timedObject);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseTimedAction addTimeToFinish(int timeToFinish) {
            try {
                parameters.put("timeToFinish", timeToFinish);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseTimedAction addEffectMultiplier(float effectMultiplier) {
            try {
                parameters.put("effectMultiplier", effectMultiplier);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseTimedAction TimedAction(String timedAction) {
        return new BaseTimedAction(timedAction);
    }

    public static class BaseTransitionToMenu extends BaseTracking {
        public BaseTransitionToMenu() {
            eventName = "transitionToMenu";
        }
    }

    public static BaseTransitionToMenu TransitionToMenu() {
        return new BaseTransitionToMenu();
    }

    public static class BaseTransitionToGame extends BaseTracking {
        public BaseTransitionToGame(String type) {
            eventName = "transitionToGame";
            try {
                parameters.put("type", type);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    public static BaseTransitionToGame TransitionToGame(String type) {
        return new BaseTransitionToGame(type);
    }

    public static class BasePerformanceStats extends BaseTracking {
        public BasePerformanceStats(String pointInGame, String previousPointInGame, long timeSincePreviousStep, long timeSinceStart, ArrayList<PerformanceMetric> metrics, String currentScene) {
            eventName = "performanceStats";

            try {
                parameters.put("pointInGame", pointInGame);
                parameters.put("previousPointInGame", previousPointInGame);
                parameters.put("timeSincePreviousStep", timeSincePreviousStep);
                parameters.put("timeSinceStart", timeSinceStart);

                JSONArray metricsJson = new JSONArray();
                for (PerformanceMetric metric : metrics) {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("name", metric.getName());
                    jsonObject.put("currentValue", metric.getCurrentValue());
                    jsonObject.put("maxValue", metric.getMaxValue());
                    jsonObject.put("averageValue", metric.getAverageValue());
                    jsonObject.put("totalValue", metric.getTotalValue());

                    metricsJson.put(jsonObject);
                }

                parameters.put("metrics", metricsJson);
                parameters.put("currentScene", currentScene);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BasePerformanceStats addNextPointInGame(String nextPointInGame) {
            try {
                parameters.put("nextPointInGame", nextPointInGame);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BasePerformanceStats addSubPoints(ArrayList<PerformanceMetric.PerformanceSubPoints> subPoints) {
            try {
                parameters.put("subPoints", new JSONArray(super.gson.toJson(subPoints)));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BasePerformanceStats PerformanceStats(String pointInGame, String previousPointInGame, long timeBetweenSteps, long timeSinceStart, ArrayList<PerformanceMetric> metrics, String currentScene) {
        return new BasePerformanceStats(pointInGame, previousPointInGame, timeBetweenSteps, timeSinceStart, metrics, currentScene);
    }

    public static class BaseError extends BaseTracking {
        public BaseError(String type, String action) {
            eventName = "error";

            try {
                parameters.put("type", type);
                parameters.put("action", action);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        public BaseError addErrorCode(String code) {
            try {
                parameters.put("code", code);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseError addMessage(String message) {
            try {
                parameters.put("message", message);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }

        public BaseError addStackTrace(String stackTrace) {
            try {
                parameters.put("stackTrace", stackTrace);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            return this;
        }
    }

    public static BaseError Error(String type, String action) {
        return new BaseError(type, action);
    }
}