package io.gamedock.sdk.dailybonus;

import android.content.Context;
import android.content.Intent;

import com.google.gson.Gson;

import java.util.ArrayList;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.internal.OverlayEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.dailybonus.GamedockDailyBonus;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.reward.Reward;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.web.WebViewActivity;

public class DailyBonusManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "dailyBonus";

    private static volatile DailyBonusManager mInstance = null;
    private Context context;
    private Gson gson;

    private GamedockDailyBonus gamedockDailyBonus = new GamedockDailyBonus();

    private DailyBonusManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
    }

    public static DailyBonusManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new DailyBonusManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to request the Daily Bonus.
     */
    public void requestDailyBonus() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        OverlayEvent overlayEvent = new OverlayEvent(context);
        overlayEvent.setRequestDailyBonus();
        GamedockSDK.getInstance(context).trackEvent(overlayEvent, null);
    }

    /**
     * Method used to process the Daily Bonus response from the backend.
     *
     * @param dailyBonusConfiguration Daily Bonus configuration that is stored for when the game calls the show method.
     */
    public void processDailyBonusResponse(String dailyBonusConfiguration) {
        try {
            gamedockDailyBonus = gson.fromJson(dailyBonusConfiguration, GamedockDailyBonus.class);

            GamedockSDK.getInstance(context).getDailyBonusCallbacks().dailyBonusAvailable(gamedockDailyBonus.getType());
        } catch (Exception e) {
            GamedockSDK.getInstance(context).getDailyBonusCallbacks().dailyBonusNotAvailable();
        }
    }

    /**
     * Method used to show the cached Daily Bonus.
     */
    public void showDailyBonus() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (!NetworkUtil.isInternetAvailable(context)) {
            GamedockSDK.getInstance(context).getDailyBonusCallbacks().dailyBonusNotAvailable();
            return;
        }

        GamedockSDK.getInstance(context).isShowingChildActivity = true;

        Intent intent = new Intent(context, WebViewActivity.class);
        intent.putExtra("webViewUrl", gamedockDailyBonus.getUrl());
        intent.putExtra("eventName", "dailybonus");
        intent.putExtra("eventData", gson.toJson(gamedockDailyBonus));
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    public String getDailyBonusConfig() {
        return gson.toJson(gamedockDailyBonus);
    }

    /**
     * Method that process the Daily Bonus response received from the Gamedock backend.
     *
     * @param rewardList The list of rewards that has been received from the backend.
     */
    public void processRewardResponse(ArrayList<Reward> rewardList) {

        ArrayList<Reward> externalReward = null;

        if (rewardList != null) {
            for (int i = 0; i < rewardList.size(); i++) {
                switch (rewardList.get(i).type) {
                    case "CURRENCY":
                        GamedockSDK.getInstance(context).addCurrencyToWallet(rewardList.get(i).id, rewardList.get(i).amount, PlayerDataUpdateReasons.DailyBonus, null, null, null, false, null);
                        break;
                    case "ITEM":
                        Item gameItem = GamedockGameDataManager.getInstance(context).getItem(rewardList.get(i).id);

                        if (gameItem == null) {
                            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemNotFound);
                            return;
                        }

                        if (gameItem.isUnique()) {
                            UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
                            uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
                            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0,0, 0, PlayerDataUpdateReasons.DailyBonus);
                        } else {
                            GamedockSDK.getInstance(context).addItemToInventory(rewardList.get(i).id, rewardList.get(i).amount, PlayerDataUpdateReasons.DailyBonus, null, null, null, false, null);
                        }
                        break;
                    case "EXTERNAL":
                        if (externalReward == null) {
                            externalReward = new ArrayList<>();
                        }
                        externalReward.add(rewardList.get(i));
                        break;
                }
            }

            if (externalReward != null && !externalReward.isEmpty()) {
                sendRewardsToExternal(externalReward);
            }

            requestDailyBonus();
        }
    }

    /**
     * Method that send the Daily Bonus rewards to the external platform (Unity/AIR) if the rewards have the EXTERNAL tag.
     *
     * @param rewardList The list of rewards that has been received from the backend.
     */
    private void sendRewardsToExternal(ArrayList<Reward> rewardList) {
        String rewards = GamedockSDK.getInstance(context).getGson().toJson(rewardList);
        GamedockSDK.getInstance(context).getDailyBonusCallbacks().dailyBonusReward(rewards);
    }

    /**
     * Method that sends an event to the Gamedock backend specifying that the Daily Bonus needs to be collected.
     */
    public void sendCollectDailyBonus() {
        Event event = new Event(context);
        event.setName("collectDailyBonus");
        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    public void resetDailyBonusConfig() {
        gamedockDailyBonus = new GamedockDailyBonus();
    }
}
