package io.gamedock.sdk.activities;

import android.app.Application;
import android.content.Context;
import android.graphics.Bitmap;

import androidx.multidex.MultiDex;

import com.evernote.android.job.JobApi;
import com.evernote.android.job.JobConfig;
import com.evernote.android.job.JobManager;
import com.nostra13.universalimageloader.core.DisplayImageOptions;
import com.nostra13.universalimageloader.core.ImageLoader;
import com.nostra13.universalimageloader.core.ImageLoaderConfiguration;
import com.nostra13.universalimageloader.core.assist.ImageScaleType;
import com.nostra13.universalimageloader.core.assist.QueueProcessingType;
import com.nostra13.universalimageloader.utils.StorageUtils;

import java.io.File;

import io.gamedock.sdk.network.NetworkJobCreator;
import io.gamedock.sdk.utils.images.cache.ImageFileNameGenerator;
import io.gamedock.sdk.utils.images.cache.disk.LruDiskCache;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Gamedock Application class which extend basic Application.
 * Used for initialising Adjust SDK, Firebase SDK, Google Analytics and Image Loading SDK.
 * The application that implements the Gamedock SDK needs to extend from this Application in it's AndroidManifest.xml.
 */

public class GamedockSDKApplication extends Application {

    @Override
    public void onCreate() {
        try {
            Class.forName("android.os.AsyncTask");
        } catch (Throwable ignore) {
            // ignored
        }

        super.onCreate();

        //Event queuing
        try {
            JobConfig.setApiEnabled(JobApi.WORK_MANAGER, true);
            JobManager.create(this).addJobCreator(new NetworkJobCreator());
        } catch (Exception e) {
            LoggingUtil.e("Unable to create Job Manager. Offline events will not work for this session.");
        }

        //Image Loader
        initialiseImageLoader();
    }

    /**
     * Method used to initialise the MultiDex library.
     *
     * @param base Base Context of the Application.
     */
    @Override
    protected void attachBaseContext(Context base) {
        super.attachBaseContext(base);
        try {
            MultiDex.install(this);
        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("Multi-Dex dependency not included! Please add the multidex library to your Gradle File if you want to use Multi-Dexing");
        }
    }

    /**
     * Method that initialises the Universal Image Loader Library.
     */
    private void initialiseImageLoader() {
        try {
            DisplayImageOptions options = new DisplayImageOptions.Builder()
                    .cacheInMemory(false)
                    .cacheOnDisk(true)
                    .bitmapConfig(Bitmap.Config.RGB_565)
                    .imageScaleType(ImageScaleType.IN_SAMPLE_INT)
                    .build();

            File cacheDir = StorageUtils.getCacheDirectory(this);

            LruDiskCache lruDiskCache = new LruDiskCache(cacheDir, new ImageFileNameGenerator(), 100 * 1024 * 1024);

            ImageLoaderConfiguration config = new ImageLoaderConfiguration.Builder(this)
                    .threadPriority(Thread.NORM_PRIORITY - 2)
                    .denyCacheImageMultipleSizesInMemory()
                    .diskCache(lruDiskCache)
                    .tasksProcessingOrder(QueueProcessingType.LIFO)
                    .defaultDisplayImageOptions(options)
                    .build();

            ImageLoader.getInstance().init(config);

        } catch (Exception | NoClassDefFoundError e) {
            e.printStackTrace();
        }
    }

}