package io.gamedock.sdk;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import com.google.gson.Gson;
import com.unity3d.player.UnityPlayer;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;

import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.EventActionListener;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.internal.HeartbeatEvent;
import io.gamedock.sdk.events.response.ResponseEvent;
import io.gamedock.sdk.initialization.InitializationOptions;
import io.gamedock.sdk.models.performance.PerformanceMetric;
import io.gamedock.sdk.utils.IAP.IAPUtil;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.performance.PerformanceUtil;
import io.gamedock.sdk.utils.permissions.PermissionBuilder;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.utils.thread.GamedockHandlerThread;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Base sdk class that handles all the commands
 */
public abstract class GamedockSDKBase {

    protected static boolean sdkComponentsInitialised = false;
    protected static boolean sdkFeaturesInitialised = false;

    public Context context;

    private boolean isGameUnpublished = false;

    protected GamedockHandlerThread handlerThread;
    public Handler handler;
    public Handler mainThreadHandler;
    private Runnable runnable;
    private Runnable performanceRunnable;
    public LinkedHashMap<String, EventActionListener> actionListenerList = new LinkedHashMap<>();
    public EventActionListener globalEventActionListener;
    private StorageUtil storageUtil;
    private final Gson gson;

    public String appVersion = "0.0.0";

    protected boolean sdkInitialised;
    protected boolean isAppRunning;

    public boolean isShowingAgeGate;

    public boolean isShowingPrivacyPolicy;

    public boolean isShowingChildActivity;

    private String gamedockToken = null;
    protected boolean isCoppaSet = false;
    protected boolean coppaEnabled = false;

    private static GamedockEnvironment ENVIRONMENT = GamedockEnvironment.PRODUCTION;
    private boolean isUnitTesting;

    public boolean firebaseInitialised;
    public HashMap<String, String> externalIds = new HashMap<>();

    public InitializationOptions initializationOptions;

    private static FileAssetsUtil fileAssetsUtil;

    /**
     * GamedockSDKBase constructor necessary for creating the GamedockSDK Object. It initialises the first session,
     * sets the SDK settings, creates all the objects necessary by the GamedockSDK and retrieves the persistence list of events.
     *
     * @param context The current activity context.
     */
    GamedockSDKBase(Context context) {
        this.context = context;

        this.handlerThread = new GamedockHandlerThread("GamedockHandlerThread");
        this.handlerThread.setContext(context);
        this.handlerThread.start();

        this.mainThreadHandler = new Handler(Looper.getMainLooper());

        this.storageUtil = StorageUtil.getInstance(context);
        this.gson = new Gson();

        try {
            appVersion = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName;
        } catch (Exception e) {
            e.printStackTrace();
        }

        fileAssetsUtil = new FileAssetsUtil();

        FeaturesUtil.initFeaturesInformation(storageUtil, gson);
    }

    public StorageUtil getStorageUtil() {
        if (storageUtil == null) {
            storageUtil = StorageUtil.getInstance(context);
            return storageUtil;
        }

        return storageUtil;
    }

    public Gson getGson() {
        return gson;
    }

    /**
     * Method that tracks an {@link Event} without an {@link EventActionListener}.
     *
     * @param event The event that needs to be tracked.
     */
    public void trackEvent(Event event) {
        trackEvent(event, null);
    }

    /**
     * Main SDK method that tracks the events to the Gamedock server.
     * Adds the necessary default parameters to the event.
     *
     * @param event          The event that needs to be sent to the server.
     * @param actionListener The action listener needed in case you want to override the normal response processing flow.
     */
    public synchronized void trackEvent(Event event, final EventActionListener actionListener) {
        EventManager.getInstance(context).trackEvent(event, actionListener);
    }

    /**
     * Method that is called internally if the event has been already queued but still needs to be sent.
     *
     * @param queuedEvent    The event that needs to be sent to the server.
     * @param actionListener The action listener needed in case you want to override the normal response processing flow.
     */
    public void trackEventQueued(Event queuedEvent, EventActionListener actionListener) {
        EventManager.getInstance(context).trackEventQueued(queuedEvent, actionListener);
    }

    /**
     * Method that is used mainly by Unity in order to track custom events.
     * Also holds logic if the custom events are of specific nature such as IAP or tracking updatePlayerData.
     *
     * @param eventName  The name of the custom event
     * @param parameters Parameters that need to be added to the custom event
     */
    public void trackSpecialEvent(String eventName, JSONObject parameters, EventActionListener eventActionListener) {
        LoggingUtil.d("TrackSpecialEvent: " + eventName + " Data: " + parameters.toString());

        EventManager.getInstance(context).trackSpecialEvent(eventName, parameters, eventActionListener);
    }

    /**
     * Method used for processing all the response events received from the Gamedock server.
     * Decides how each response will be processed and to which processing util class it will be sent to.
     *
     * @param responseEvent The response event received from the Gamedock server.
     * @param listenerId    The listener id associated with the actionListener attached to the initial sent event.
     */
    public void processResponseEvent(ResponseEvent responseEvent, final String listenerId) {
        EventManager.getInstance(context).processResponseEvent(responseEvent, listenerId);
    }

    /**
     * Method that sets the information regarding the plugins/wrappers that are used in combination with the Gamedock SDK.
     *
     * @param pluginName    The plugin name, such as Unity, Flash etc.
     * @param pluginVersion The version of the plugin that is currently being used.
     */
    public void setPluginInformation(String pluginName, String pluginVersion) {
        getStorageUtil().putString(StorageUtil.Keys.PluginName, pluginName);
        getStorageUtil().putString(StorageUtil.Keys.PluginVersion, pluginVersion);
    }

    /**
     * Method that manually sets a package name used for event sending.
     *
     * @param packageName Value of the package name.
     */
    public void setApplicationPackageName(String packageName) {
        getStorageUtil().putString(StorageUtil.Keys.SpilApplicationPackageName, packageName);
    }

    public String getApplicationPackageName() {
        if (getStorageUtil().getString(StorageUtil.Keys.SpilApplicationPackageName, null) != null) {
            return getStorageUtil().getString(StorageUtil.Keys.SpilApplicationPackageName, null);
        } else {
            if (context == null) {
                return "";
            }

            setApplicationPackageName(context.getPackageName());
            return context.getPackageName();
        }
    }

    /**
     * Method that manually generates an UUID. Used more specifically for tests.
     */
    public void setManualUUID() {
        UserIDGenerator.manualUUIDGeneration(context, null);
    }

    /**
     * Method that manually generates an UUID. Used more specifically for tests.
     */
    public void setManualUUID(String userId) {
        UserIDGenerator.manualUUIDGeneration(context, userId);
    }

    /**
     * Method used to send information to the Unity layer.
     *
     * @param methodName The name of the method with which the SDK needs to communicate.
     * @param data       The data that needs to be sent to Unity.
     */
    public void sendDataToUnity(String methodName, String data) {
        try {
            UnityPlayer.UnitySendMessage("GamedockSDK", methodName, data);
        } catch (NoClassDefFoundError e) {
            LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
        }
    }

    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        PermissionBuilder.getInstance().onRequestPermissionResult(context, requestCode, permissions, grantResults);
    }

    /**
     * Method that initialises the heartbeat session.
     * The heartBeat events are fired every 60 seconds for as long as the app is opened
     */
    protected void startHeartbeatSession() {
        try {
            final int delay = 60000;

            runnable = new Runnable() {
                @Override
                public void run() {

                    HeartbeatEvent event = new HeartbeatEvent(context);
                    event.setHeartBeat();

                    try {
                        ArrayList<PerformanceMetric> metrics = PerformanceUtil.getInstance(context).populatePerformanceMetrics();
                        event.addCustomData("metrics", new JSONArray(gson.toJson(metrics)));
                    } catch (Exception ignored) {
                    }

                    trackEvent(event, null);

                    PerformanceUtil.getInstance(context).clearMetrics();

                    if (handler != null) {
                        handler.postDelayed(runnable, delay);
                    } else if (mainThreadHandler != null) {
                        mainThreadHandler.postDelayed(runnable, delay);
                    }
                }
            };

            if (handler != null) {
                handler.postDelayed(runnable, delay);
            } else if (mainThreadHandler != null) {
                mainThreadHandler.postDelayed(runnable, delay);
            }
        } catch (NullPointerException ignore) {
        }
    }

    /**
     * Method that stops the heartbeat session by removing the runnable.
     */
    protected void stopHeartbeatSession() {
        if (handler != null) {
            handler.removeCallbacks(runnable);
        } else if (mainThreadHandler != null) {
            mainThreadHandler.removeCallbacks(runnable);
        }
    }

    /**
     * Method that triggers every 9.9 seconds in order to record performance stats.
     * Is 9.9 seconds in order to reduce the colliding with the regular update for heartbeats.
     */
    protected void startPerformanceTracking() {
        final int delay = 9900;

        if (handler != null) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    PerformanceUtil.getInstance(context).recordPerformanceStats();
                }
            });
        }

        performanceRunnable = new Runnable() {
            @Override
            public void run() {
                PerformanceUtil.getInstance(context).recordPerformanceStats();
                handler.postDelayed(this, delay);
            }
        };

        if (handler != null) {
            handler.postDelayed(performanceRunnable, delay);
        }
    }

    /**
     * Method that stops the performance stat gathering by removing the runnable.
     */
    protected void stopPerformanceTracking() {
        if (handler != null) {
            handler.removeCallbacks(performanceRunnable);
        }
    }

    /**
     * Method that processes the IAP in order to add additional parameters that are needed.
     *
     * @param event The IAP event that will be processed.
     * @param SKU   The SKU id accompanied with the transaction.
     */
    public void processIAPPurchaseEvent(Event event, String SKU) {
        IAPUtil.processIapPurchaseEvent(context, event, SKU);
    }

    public boolean isAppRunning() {
        return isAppRunning;
    }

    /**
     * Method that retrieves the Gamedock UUID.
     *
     * @return The Gamedock UUID generated by the Gamedock SDK.
     */
    public String getGamedockUID() {
        return UserIDGenerator.getGamedockUserId(context);
    }

    public String getFirebaseInstanceId() {
        return UserIDGenerator.getFirebaseInstanceId();
    }

    public String getGamedockToken() {
        if (gamedockToken != null) {
            return gamedockToken;
        } else {
            gamedockToken = storageUtil.getString(StorageUtil.Keys.SpilToken, null);
            return gamedockToken;
        }
    }

    public void setGamedockToken(String gamedockToken) {
        this.gamedockToken = gamedockToken;

        if (gamedockToken != null) {
            storageUtil.putString(StorageUtil.Keys.SpilToken, gamedockToken);
        } else {
            storageUtil.remove(StorageUtil.Keys.SpilToken);
        }

    }

    public void clearLog() {
        LoggingUtil.clearLog();
    }

    public String getLog() {
        return LoggingUtil.getLog();
    }

    public long getFeatureVersionId(String feature) {
        return FeaturesUtil.getFeaturesVersionId(feature);
    }

    public GamedockEnvironment getEnvironment() {
        return ENVIRONMENT;
    }

    public void setEnvironment(GamedockEnvironment ENVIRONMENT) {
        GamedockSDKBase.ENVIRONMENT = ENVIRONMENT;
    }

    public void setGlobalEventActionListener(EventActionListener listener) {
        globalEventActionListener = listener;
    }

    public void setCoppaEnabled(boolean coppaEnabled) {
        storageUtil.putBoolean(StorageUtil.Keys.COPPAEnabled, coppaEnabled);
    }

    public boolean isCoppaEnabled() {
        if (isCoppaSet) {
            return coppaEnabled;
        }
        coppaEnabled = storageUtil.getBoolean(StorageUtil.Keys.COPPAEnabled, false);
        isCoppaSet = true;

        return coppaEnabled;
    }

    public boolean isUnitTesting() {
        return isUnitTesting;
    }

    public void setUnitTesting(boolean unitTesting) {
        isUnitTesting = unitTesting;
    }

    public InitializationOptions getInitializationOptions() {
        return initializationOptions;
    }

    public boolean isGameUnpublished() {
        return isGameUnpublished;
    }

    public void setGameUnpublished(boolean gameUnpublished) {
        isGameUnpublished = gameUnpublished;
    }

    public static FileAssetsUtil getFileAssetsUtil() {
        return fileAssetsUtil;
    }
}
