package io.gamedock.sdk;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.util.Log;

import androidx.browser.customtabs.CustomTabsIntent;

import com.google.android.gms.analytics.GoogleAnalytics;
import com.google.android.gms.analytics.Tracker;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.analytics.FirebaseAnalytics;
import com.google.firebase.crashlytics.FirebaseCrashlytics;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import io.gamedock.sdk.ads.MoreAppsCallbacks;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.assetbundles.AssetBundlesCallbacks;
import io.gamedock.sdk.assetbundles.AssetBundlesManager;
import io.gamedock.sdk.config.ConfigDataCallbacks;
import io.gamedock.sdk.config.ConfigManager;
import io.gamedock.sdk.config.firebase.FirebaseRemoteConfigManager;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.dailybonus.DailyBonusCallbacks;
import io.gamedock.sdk.dailybonus.DailyBonusManager;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.EventActionListener;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.internal.ServerDataEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataCallbacks;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.gamedata.packages.PackageManager;
import io.gamedock.sdk.gamedata.packages.PackagesCallbacks;
import io.gamedock.sdk.gamedata.promotions.PromotionsCallbacks;
import io.gamedock.sdk.gamedata.promotions.PromotionsManager;
import io.gamedock.sdk.google.dynamiclinks.DeepLinkCallbacks;
import io.gamedock.sdk.google.dynamiclinks.DeepLinkManager;
import io.gamedock.sdk.initialization.InitializationDataCallbacks;
import io.gamedock.sdk.initialization.InitializationManager;
import io.gamedock.sdk.initialization.InitializationOptions;
import io.gamedock.sdk.localization.LocalizationCallbacks;
import io.gamedock.sdk.localization.LocalizationManager;
import io.gamedock.sdk.mission.MissionConfigurationCallbacks;
import io.gamedock.sdk.mission.MissionConfigurationManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.perk.PerkItem;
import io.gamedock.sdk.models.mission.Container;
import io.gamedock.sdk.models.mission.Mission;
import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.ContainerProgress;
import io.gamedock.sdk.models.userdata.mission.MissionProgress;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.pushnotifications.NotificationManager;
import io.gamedock.sdk.reward.RewardCallbacks;
import io.gamedock.sdk.reward.RewardManager;
import io.gamedock.sdk.social.SocialCallbacks;
import io.gamedock.sdk.social.SocialManager;
import io.gamedock.sdk.splashscreen.SplashScreenCallbacks;
import io.gamedock.sdk.splashscreen.SplashscreenManager;
import io.gamedock.sdk.support.SupportActivity;
import io.gamedock.sdk.tier.TieredEventsCallbacks;
import io.gamedock.sdk.tier.TieredEventsManager;
import io.gamedock.sdk.userdata.UserDataCallbacks;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.UserDataTransaction;
import io.gamedock.sdk.userdata.missiondata.MissionDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.IAP.IAPCallbacks;
import io.gamedock.sdk.utils.IAP.IAPUtil;
import io.gamedock.sdk.utils.agegate.AgeGateActivity;
import io.gamedock.sdk.utils.agegate.AgeGateCallbacks;
import io.gamedock.sdk.utils.apprate.AppRateUtil;
import io.gamedock.sdk.utils.azerionConnect.AzerionConnectActivity;
import io.gamedock.sdk.utils.azerionConnect.AzerionConnectCallbacks;
import io.gamedock.sdk.utils.crash.FirebaseCrashlyticsUtil;
import io.gamedock.sdk.utils.device.DeviceUtil;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.gcm.RegisterDevice;
import io.gamedock.sdk.utils.images.ImageLoadCallbacks;
import io.gamedock.sdk.utils.images.ImageLoaderUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.performance.PerformanceUtil;
import io.gamedock.sdk.utils.permissions.PermissionCallbacks;
import io.gamedock.sdk.utils.permissions.PermissionUtil;
import io.gamedock.sdk.utils.privacy.PrivacyPolicyActivity;
import io.gamedock.sdk.utils.privacy.PrivacyPolicyCallbacks;
import io.gamedock.sdk.utils.server.ServerDataCallbacks;
import io.gamedock.sdk.utils.session.SessionUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;
import io.gamedock.sdk.utils.azerionConnect.AuthStateManager;


/**
 * Main sdk class that handles all the commands
 */

public class GamedockSDK extends GamedockSDKBase {

    private static final Object lock = new Object();

    private static volatile GamedockSDK mInstance;

    private static InitializationDataCallbacks initializationDataCallbacks;
    private static MoreAppsCallbacks moreAppsCallbacks;
    private static GamedockGameDataCallbacks gamedockGameDataCallbacks;
    private static UserDataCallbacks userDataCallbacks;
    private static MissionConfigurationCallbacks missionConfigurationCallbacks;
    private static RewardCallbacks rewardCallbacks;
    private static ConfigDataCallbacks configDataCallbacks;
    private static SplashScreenCallbacks splashScreenCallbacks;
    private static DailyBonusCallbacks dailyBonusCallbacks;
    private static ImageLoadCallbacks imageLoadCallbacks;
    private static IAPCallbacks iapCallbacks;
    private static ServerDataCallbacks serverDataCallbacks;
    private static PermissionCallbacks permissionCallbacks;
    private static SocialCallbacks socialCallbacks;
    private static AgeGateCallbacks ageGateCallbacks;
    private static PrivacyPolicyCallbacks privacyPolicyCallbacks;
    private static PackagesCallbacks packagesCallbacks;
    private static PromotionsCallbacks promotionsCallbacks;
    private static TieredEventsCallbacks tieredEventsCallbacks;
    private static AssetBundlesCallbacks assetBundlesCallbacks;
    private static DeepLinkCallbacks deepLinkCallbacks;
    private static LocalizationCallbacks localizationCallbacks;
    private static AzerionConnectCallbacks azerionConnectCallbacks;

    private Tracker mTracker;

    private GamedockSDK(Context context) {
        super(context);
    }

    /**
     * This method should always be called in order to retrieve the GamedockSDK object
     * as it check if an instance of this object has been created and if not creates one.
     *
     * @param activity The current activity
     * @return Returns the GamedockSDK Object
     */
    public static GamedockSDK getInstance(Activity activity) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new GamedockSDK(activity);
                }
            }
        }

        return mInstance;
    }

    /**
     * This method should always be called in order to retrieve the GamedockSDK object
     * as it check if an instance of this object has been created and if not creates one.
     *
     * @param context The current activity context
     * @return Returns the GamedockSDK Object
     */
    public static GamedockSDK getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new GamedockSDK(context);
                }
            }
        }

        if ((mInstance.context instanceof Application) && (context instanceof Activity)) {
            mInstance.context = context;
        }

        return mInstance;
    }

    /**
     * Method that should be called before creating the GamedockSDK Object.
     * This is need in order to reset the context as class cast exception can occur between the Activity Context and the Application Context.
     */
    public static void resetContext() {
        if (mInstance != null) {
            mInstance.context = null;
            mInstance = null;
        }
    }

    public static void resetContext(Context context) {
        if (context instanceof Activity && mInstance != null) {
            mInstance.context = context;
        }
    }

    /**
     * Method used in order to register the device with Google Cloud Messaging Service.
     */
    public void registerDevice(String store) {
        try {
            if (store.equals("Amazon")) {
                getInstance(context).getStorageUtil().putString(StorageUtil.Keys.Store, "com.amazon.venezia");
            } else {
                getInstance(context).getStorageUtil().putString(StorageUtil.Keys.Store, context.getPackageManager().getInstallerPackageName(context.getPackageName()));
            }
        } catch (Exception | Error e) {
            getInstance(context).getStorageUtil().putString(StorageUtil.Keys.Store, "com.android.vending");
        }

        NotificationManager.enableNotifications(context);
        RegisterDevice.register(context);
        DeviceUtil.reportAppStatus(context);

        IAPUtil.initIAP(context, store);
    }

    public void confirmUserIdChange() {
        InitializationManager.confirmUserIdChange(context);
    }

    /**
     * Method that allows the enabling of push notifications.
     */
    public void enableNotifications() {
        NotificationManager.enableNotifications(context);
    }

    /**
     * Method that disables the recevial of push notifications.
     */
    public void disableNotifications() {
        NotificationManager.disableNotifications(context);
    }

    /**
     * Method that schedule a notification to be desplayed at the requested {@link Date}. It uses the {@link android.app.AlarmManager} class to schedule.
     *
     * @param date    The date and time at which the notification will be displayed.
     * @param title   The title which will be displayed in the notification.
     * @param message The message which will be displayed in the notification.
     */
    public int scheduleLocalNotification(Date date, String title, String message) {
        return NotificationManager.scheduleLocalNotification(date, context, title, message);
    }

    /**
     * Method that cancels a scheduled notification which was scheduled with @scheduleLocalNotification method.
     *
     * @param requestCode A unique code used to identify the push notification in the System.
     */
    public void cancelLocalNotification(int requestCode) {
        NotificationManager.cancelLocalNotification(requestCode);
    }

    /**
     * Method that immediately displays a notification on the system tray.
     *
     * @param title   The title which will be displayed in the notification.
     * @param message The message which will be displayed in the notification.
     */
    public void showLocalNotification(String title, String message) {
        String uniqueNotificationID = UUID.randomUUID().toString();

        NotificationManager.showLocalNotification(context, title, message, uniqueNotificationID);

        NotificationManager.sendNotificationEvent(context, uniqueNotificationID, "notificationCreated", false, "local");
    }

    /**
     * Method used to save the external user id.
     *
     * @param provider The name of the provider associated with the user id.
     * @param userId   The user id that needs to be saved.
     */
    public void setUserId(String provider, String userId) {
        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData != null) {
            userData.setUserID(userId);
            userData.setProvider(provider);
        } else {
            userData = new UserData();
            userData.setWallet(new Wallet());
            userData.setInventory(new Inventory());
            userData.setUserID(userId);
            userData.setProvider(provider);
        }

        UserDataManager.getInstance(context).updateUserData(userData);
    }

    /**
     * Method that retrieves the external user id that has been saved.
     *
     * @return The saved user id or null if no external id saved.
     */
    public String getUserId() {
        if (UserDataManager.getInstance(context).getUserData() != null) {
            return UserDataManager.getInstance(context).getUserData().getUserID();
        } else {
            return null;
        }
    }

    /**
     * Method that retrieves the external provider that has been saved.
     *
     * @return The saved provider or null if no external provider saved.
     */
    public String getUserProvider() {
        if (UserDataManager.getInstance(context).getUserData() != null) {
            return UserDataManager.getInstance(context).getUserData().getProvider();
        } else {
            return null;
        }
    }

    /**
     * Method that saves the public game state of the game.
     *
     * @param gameStateData The public game state data that needs to be saved. No strict structure is required.
     */
    public void setPublicGameState(String gameStateData, EventActionListener actionListener) {
        UserDataManager.getInstance(context).getGameStateManager().setPublicGameState(gameStateData, actionListener);
    }

    /**
     * Method that returns the saved public game state.
     *
     * @return The current public game state saved.
     */
    public String getPublicGameState() {
        return UserDataManager.getInstance(context).getGameStateManager().getPublicGameState();
    }

    /**
     * Method that saves the private game state of the game.
     *
     * @param gameStateData  The private game state data that needs to be saved. No strict structure is required.
     * @param actionListener Event listener
     */
    public void setPrivateGameState(String gameStateData, EventActionListener actionListener) {
        UserDataManager.getInstance(context).getGameStateManager().setPrivateGameState(gameStateData, actionListener);
    }

    /**
     * Method that returns the saved private game state.
     *
     * @return The current private game state saved.
     */
    public String getPrivateGameState() {
        return UserDataManager.getInstance(context).getGameStateManager().getPrivateGameState();
    }

    /**
     * Method that show a merge conflict dialog in which the user can select which state he should continue with in his game.
     *
     * @param title        The title of the dialog.
     * @param message      The message of the dialog.
     * @param localButton  The text of the local button.
     * @param remoteButton The text of the remote button.
     * @param mergeButton  The text of the merge button. Can be null or empty and if that is the case the button will not be shown.
     */
    public void showMergeConflictDialog(String title, String message, String localButton, String remoteButton, String mergeButton) {
        UserDataManager.getInstance(context).showMergeConflictDialog(title, message, localButton, remoteButton, mergeButton);
    }

    /**
     * Method that shows a sync error dialog in which the user can start the merging process.
     *
     * @param title       The title of the dialog.
     * @param message     The message of the dialog.
     * @param mergeButton The button text.
     */
    public void showSyncErrorDialog(String title, String message, String mergeButton) {
        UserDataManager.getInstance(context).showSyncErrorDialog(title, message, mergeButton);
    }

    /**
     * Method that shows a merge failed dialog in which the user can re-start the merging process.
     *
     * @param title       The title of the dialog.
     * @param message     The message of the dialog.
     * @param retryButton The text for the retry button.
     * @param mergeData   The merge data that has to be resend.
     * @param mergeType   The merge type that has to be resend.
     */
    public void showMergeFailedDialog(String title, String message, String retryButton, String mergeData, String mergeType) {
        UserDataManager.getInstance(context).showMergeFailedDialog(title, message, retryButton, mergeData, mergeType);
    }

    /**
     * Method used to request more apps.
     */
    public void requestMoreApps() {
        AdManager.getInstance().requestMoreApps(context);
    }

    /**
     * Method used to show more apps.
     * Only Gamedock is used.
     */
    public void playMoreApps() {
        AdManager.getInstance().playMoreApps(context);
    }

    /**
     * Method used internally to request the config from the Gamedock server.
     */
    public void requestConfig() {
        ConfigManager.getInstance(context).requestConfig();
    }

    /**
     * Method that returns the full config saved by the Gamedock SDK.
     *
     * @return The full config in a JSON format.
     */
    public String getConfigAll(boolean withSdkConfig) {
        return ConfigManager.getInstance(context).getConfigAll(withSdkConfig);
    }

    /**
     * Method that returns a value on the first level of the config based on the provided key.
     *
     * @param key The key for which the value should be returned.
     * @return The value saved in the config file.
     */
    public String getConfigValue(String key) {
        return ConfigManager.getInstance(context).getConfigValue(key);
    }

    /**
     * Method used internally to process the received push notification.
     */
    public void processNotification() {
        NotificationManager.processNotificationData(context);
    }

    public void validateSubscription(String skuId, String transactionId, String token) {
        IAPUtil.validateSubscription(context, skuId, transactionId, token);
    }

    /**
     * Method that returns the retrieved packages from the server.
     *
     * @return The packages information in a JSON format.
     */
    public String getAllPackages() {
        return PackageManager.getInstance(context).getAllPackages();
    }

    /**
     * Method that returns the specified package based on its id.
     *
     * @param packageId The package id for which the information needs to be returned.
     * @return The package information in JSON format.
     */
    public String getPackage(String packageId) {
        return PackageManager.getInstance(context).getPackage(packageId);
    }

    /**
     * Method that returns the retrieved packages from the server.
     *
     * @return The promotions information in a JSON format.
     */
    public String getAllPromotions() {
        return PromotionsManager.getInstance(context).getPromotions();
    }

    /**
     * Method that returns the promotion associated with a package if it exists.
     *
     * @param bundleId The bundle id for which the information needs to be returned.
     * @return The promotion information in JSON format or null if no information is present.
     */
    public String getBundlePromotion(int bundleId) {
        return PromotionsManager.getInstance(context).getBundlePromotion(bundleId);
    }

    /**
     * Method that returns the promotion associated with a package if it exists.
     *
     * @param pacakgeId The package id for which the information needs to be returned.
     * @return The promotion information in JSON format or null if no information is present.
     */
    public String getPackagePromotion(String pacakgeId) {
        return PromotionsManager.getInstance(context).getPackagePromotion(pacakgeId);
    }

    /**
     * Method that automatically requests gameData information from the Gamedock server.
     */
    public void requestGameData() {
        GamedockGameDataManager.getInstance(context).requestGameData();
    }

    /**
     * Method used to request the mission configurations from the backend.
     */
    public void requestMissionConfiguration() {
        MissionConfigurationManager.getInstance(context).requestMissionConfig();
    }

    /**
     * Method used to request the promotions configuration from the backend.
     */
    public void requestPromotions() {
        PromotionsManager.getInstance(context).requestPromotions();
    }

    /**
     * Method used to show the promotion screen based on a promotion id.
     *
     * @param promotionId The id of the promotion.
     */
    public void showPromotionScreen(int promotionId) {
        PromotionsManager.getInstance(context).showPromotionScreen(promotionId);
    }

    /**
     * Method used to retrieve the containers configuration.
     *
     * @return The containers list.
     */
    public ArrayList<Container> getContainersConfiguration() {
        return MissionConfigurationManager.getInstance(context).getContainersConfiguration();
    }

    /**
     * Method used to retrieve a specific container based on the id.
     *
     * @param containerId The container id.
     * @return The Container object.
     */
    public Container getContainerConfiguration(String containerId) {
        return MissionConfigurationManager.getInstance(context).getContainerConfiguration(containerId);
    }

    /**
     * Method used to retrieve the missions configuration full list or the list which match the container id.
     *
     * @param containerId The container id.
     * @return The missions list.
     */
    public ArrayList<Mission> getMissionsConfiguration(String containerId) {
        return MissionConfigurationManager.getInstance(context).getMissionsConfiguration(containerId);
    }

    /**
     * Method used to retrieve the missions matching a specific list of mission prerequisites.
     *
     * @param missionPrerequisites A list of mission ids.
     * @return The mission list matching the condition.
     */
    public ArrayList<Mission> getMissionsConfigurationWithPrerequisites(ArrayList<String> missionPrerequisites) {
        return MissionConfigurationManager.getInstance(context).getMissionsConfigurationWithPrerequisites(missionPrerequisites);
    }

    /**
     * Method used to retrieve the missions matching a specific list of mission unlock id.
     *
     * @param missionUnlocks A list of mission unlock ids.
     * @return The mission list matching the condition.
     */
    public ArrayList<Mission> getMissionsConfigurationWithUnlocks(ArrayList<String> missionUnlocks) {
        return MissionConfigurationManager.getInstance(context).getMissionsConfigurationWithUnlocks(missionUnlocks);
    }

    /**
     * Method used to retrieve a mission based on an id.
     *
     * @param missionId The mission id.
     * @return The Mission object or null if no mission was found.
     */
    public Mission getMissionConfiguration(String missionId) {
        return MissionConfigurationManager.getInstance(context).getMissionConfiguration(missionId);
    }

    /**
     * Method that automatically requests user data containing Game State and Player Data information from the Gamedock server.
     */
    public void requestUserData() {
        UserDataManager.getInstance(context).requestUserData();
    }

    /**
     * Method used to merge the data based on the selection made by the user/developer.
     * Sends an event to the backend to notify which data was merged.
     *
     * @param mergeData The merged data in a JSON String format
     * @param mergeType The merge type. Can be "local", "remote" or "merge".
     */
    public void mergeUserData(String mergeData, String mergeType) {
        UserDataManager.getInstance(context).mergeUserData(mergeData, mergeType);
    }

    /**
     * Method that updated the UserData with the new data.
     *
     * @param userData The userData that needs to be updated.
     */
    public void updateUserData(UserData userData) {
        UserDataManager.getInstance(context).updateUserData(userData);
    }

    /**
     * Method that pull player data changes from the Gamedock server.
     */
    public void updatePlayerData() {
        UserDataManager.getInstance(context).getPlayerDataManager().pullPlayerDataChanges();
    }

    /**
     * Method that requests information about other user's game states based on the provider and the list of ids provided.
     *
     * @param provider    The provider associated with the game state.
     * @param userIdsList A JSON paresed list of userIds for which the game state will be returned.
     */
    public void requestOtherUsersGameState(String provider, String userIdsList) {
        UserDataManager.getInstance(context).getGameStateManager().requestOtherUsersGameState(provider, userIdsList);
    }

    /**
     * Method that returns the full user profile information.
     *
     * @return The {@link UserData} structured in a JSON format.
     */
    public String getUserDataJSON() {
        return new Gson().toJson(UserDataManager.getInstance(context).getUserData());
    }

    /**
     * Method that returns the full user profile information.
     *
     * @return The {@link UserData} object.
     */
    public UserData getUserData() {
        return UserDataManager.getInstance(context).getUserData();
    }

    /**
     * Method that returns the wallet information.
     *
     * @return The {@link Wallet} structured in a JSON format.
     */
    public String getWallet() {
        return UserDataManager.getInstance(context).getPlayerDataManager().getWallet();
    }

    /**
     * Method that returns the stored game data.
     *
     * @return The {@link GamedockGameData} structured in a JSON format.
     */
    public String getGamedockGameData() {
        return GamedockGameDataManager.getInstance(context).getGameDataJSON();
    }

    /**
     * Method that returns the inventory information.
     *
     * @return The {@link Inventory} structured in a JSON format.
     */
    public String getInventory() {
        return UserDataManager.getInstance(context).getPlayerDataManager().getInventory();
    }

    public UserDataTransaction.TransactionSequence beginUserDataTransaction() {
        return UserDataTransaction.Builder();
    }

    /**
     * Method used to add currency to the wallet object.
     * Performs the calculations necessary to update the wallet.
     *
     * @param currencyId The id of the currency needed to be updated.
     * @param delta      The value with which the wallet needs to be updated.
     * @param reason     The reason of the operation. A list of default reasons can be seen at {@link PlayerDataUpdateReasons}.
     * @param location   The location from which the update was fired.
     */
    public UpdatedUserData addCurrencyToWallet(int currencyId, int delta, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks customUserDataCallback) {
        UserDataCallbacks userDataCallback;
        if (customUserDataCallback != null) {
            userDataCallback = customUserDataCallback;
        } else {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        if (delta <= 0) {
            if (!isTransaction) {
                userDataCallback.userDataError(io.gamedock.sdk.utils.error.ErrorCodes.CurrencyOperation);
            }
            LoggingUtil.e("Delta value should not be negative!");
            return null;
        }

        return UserDataManager.getInstance(context).getPlayerDataManager().updateWallet(currencyId, delta, reason, reasonDetails, location, transactionId, null, isTransaction, userDataCallback);
    }

    /**
     * Method used to subtract currency to the wallet object.
     * Performs the calculations necessary to update the wallet.
     *
     * @param currencyId The id of the currency needed to be updated.
     * @param delta      The value with which the wallet needs to be updated.
     * @param reason     The reason of the operation. A list of default reasons can be seen at {@link PlayerDataUpdateReasons}.
     * @param location   The location from which the update was fired.
     */
    public UpdatedUserData subtractCurrencyFromWallet(int currencyId, int delta, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks customUserDataCallback) {
        UserDataCallbacks userDataCallback;
        if (customUserDataCallback != null) {
            userDataCallback = customUserDataCallback;
        } else {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        if (delta <= 0) {
            if (!isTransaction) {
                userDataCallback.userDataError(io.gamedock.sdk.utils.error.ErrorCodes.CurrencyOperation);
            }
            LoggingUtil.e("Delta value should not be negative!");
            return null;
        }

        delta = -delta;
        return UserDataManager.getInstance(context).getPlayerDataManager().updateWallet(currencyId, delta, reason, reasonDetails, location, transactionId, null, isTransaction, userDataCallback);
    }

    /**
     * Method used to add an item to the inventory object.
     * Performs the calculations necessary to update the inventory.
     *
     * @param itemId   The id of the item needed to be updated.
     * @param amount   The value with which the inventory needs to be updated.
     * @param reason   The reason of the operation. A list of default reasons can be seen at {@link PlayerDataUpdateReasons}.
     * @param location The location from which the update was fired.
     */
    public UpdatedUserData addItemToInventory(int itemId, int amount, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks customUserDataCallback) {
        UserDataCallbacks userDataCallback;
        if (customUserDataCallback != null) {
            userDataCallback = customUserDataCallback;
        } else {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        if (amount <= 0) {
            if (!isTransaction) {
                userDataCallback.userDataError(io.gamedock.sdk.utils.error.ErrorCodes.ItemOperation);
            }
            LoggingUtil.e("Amount value should not be negative!");
            return null;
        }

        String action = PlayerDataManager.Add;
        return UserDataManager.getInstance(context).getPlayerDataManager().updateInventoryWithItem(itemId, amount, action, reason, reasonDetails, location, transactionId, null, isTransaction, userDataCallback);
    }

    /**
     * Method used to subtract an item to the inventory object.
     * Performs the calculations necessary to update the inventory.
     *
     * @param itemId   The id of the item needed to be updated.
     * @param amount   The value with which the inventory needs to be updated.
     * @param reason   The reason of the operation. A list of default reasons can be seen at {@link PlayerDataUpdateReasons}.
     * @param location The location from which the update was fired.
     */
    public UpdatedUserData subtractItemFromInventory(int itemId, int amount, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks customUserDataCallback) {
        UserDataCallbacks userDataCallback;
        if (customUserDataCallback != null) {
            userDataCallback = customUserDataCallback;
        } else {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        if (amount <= 0) {
            if (!isTransaction) {
                userDataCallback.userDataError(io.gamedock.sdk.utils.error.ErrorCodes.ItemOperation);
            }
            LoggingUtil.e("Amount value should not be negative!");
            return null;
        }

        String action = PlayerDataManager.Subtract;
        return UserDataManager.getInstance(context).getPlayerDataManager().updateInventoryWithItem(itemId, amount, action, reason, reasonDetails, location, transactionId, null, isTransaction, userDataCallback);
    }

    /**
     * Method use to create a new UniquePlayerItem.
     *
     * @param itemId   The id of the item from which the UniquePlayerItem should be created.
     * @param uniqueId Optional id that should be use instead of the random unique id creation.
     * @return The newly created UniquePlayerItem.
     */
    public UniquePlayerItem createUniqueItem(int itemId, String uniqueId) {
        return UserDataManager.getInstance(context).getPlayerDataManager().createUniqueItem(itemId, uniqueId);
    }

    /**
     * Method that adds a UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be added.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData addUniqueItemToInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        return UserDataManager.getInstance(context).getPlayerDataManager().addUniqueItemToInventory(uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method that updates the UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be updated.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData updateUniqueItemFromInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        return UserDataManager.getInstance(context).getPlayerDataManager().updateUniqueItemFromInventory(uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method that removes a UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be removed.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData removeUniqueItemFromInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        return UserDataManager.getInstance(context).getPlayerDataManager().removeUniqueItemFromInventory(uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method used to buy the bundle defined in the Gamedock server.
     * The bundle can contain one ore more items.
     * Currency is required in order to consume the bundle.
     *
     * @param bundleId      The id of the bundled that has to be consumed.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public void buyBundle(int bundleId, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        UserDataManager.getInstance(context).getPlayerDataManager().buyBundle(bundleId, reason, reasonDetails, location, transactionId, perkItems, userDataCallback);
    }

    /**
     * Method used to open the bundle defined in the Gamedock server.
     * The bundle can contain one ore more items.
     * Currency is required in order to consume the bundle.
     *
     * @param bundleId      The id of the bundled that has to be consumed.
     * @param amount        The number of bundles that need to be opened.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public void openBundle(int bundleId, int amount, String reason, String reasonDetails, String location, ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        UserDataManager.getInstance(context).getPlayerDataManager().openBundle(bundleId, amount, reason, reasonDetails, location, perkItems, userDataCallback);
    }

    /**
     * Method used to consume the obtained gacha {@link PlayerItem}.
     *
     * @param gachaId       The id of the gacha {@link PlayerItem}.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public UpdatedUserData openGacha(int gachaId, String reason, String reasonDetails, String location, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        return UserDataManager.getInstance(context).getPlayerDataManager().openGacha(gachaId, reason, reasonDetails, location, perkItems, isTransaction, userDataCallback);
    }

    /**
     * Method used to set a limit on a certain currency. If the limit is reached an overflow value will be sent to the backend.
     *
     * @param currencyId The id of the currency for which the limit is set.
     * @param limit      The limit value.
     */
    public void setCurrencyLimit(int currencyId, int limit) {
        GamedockGameDataManager.getInstance(context).setCurrencyLimit(currencyId, limit);
    }

    /**
     * Method used to set a limit on a certain item. If the limit is reached an overflow value will be sent to the backend.
     *
     * @param itemId The id of the item for which the limit is set.
     * @param limit  The limit value.
     */
    public void setItemLimit(int itemId, int limit) {
        GamedockGameDataManager.getInstance(context).setItemLimit(itemId, limit);
    }

    /**
     * Method used to get all container progress of the user
     *
     * @return The list of container progress.
     */
    public ArrayList<ContainerProgress> getUserAllContainerProgress() {
        return UserDataManager.getInstance(context).getMissionDataManager().getUserAllContainerProgress(MissionDataManager.Status.NULL);
    }

    /**
     * Method used to get all mission progress of the user
     *
     * @return The list of mission progress.
     */
    public ArrayList<MissionProgress> getUserAllMissionProgress() {
        return UserDataManager.getInstance(context).getMissionDataManager().getUserAllMissionProgress(MissionDataManager.Status.NULL);
    }

    /**
     * Method used to update both the mission and container progress in one operation based on two lists of progress supplied.
     *
     * @param containerProgressList The container progress list containing updates.
     * @param missionProgressList   The mission progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateContainerAndMissionProgress(ArrayList<ContainerProgress> containerProgressList, ArrayList<MissionProgress> missionProgressList, String reason, String reasonDetails, String location, String transactionId) {
        UserDataManager.getInstance(context).getMissionDataManager().updateContainerAndMissionProgress(containerProgressList, missionProgressList, reason, reasonDetails, location, transactionId);
    }

    /**
     * Method used to update container progress in one operation based on a lists of progress supplied.
     *
     * @param containerProgressList The container progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateContainerProgress(ArrayList<ContainerProgress> containerProgressList, String reason, String reasonDetails, String location, String transactionId) {
        UserDataManager.getInstance(context).getMissionDataManager().updateContainerProgress(containerProgressList, reason, reasonDetails, location, transactionId);
    }

    /**
     * Method used to update mission progress in one operation based on a lists of progress supplied.
     *
     * @param missionProgressList The mission progress list containing updates.
     * @param reason              The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails       The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location            The location from which the update event has been triggered.
     * @param transactionId       The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateMissionProgress(ArrayList<MissionProgress> missionProgressList, String reason, String reasonDetails, String location, String transactionId) {
        UserDataManager.getInstance(context).getMissionDataManager().updateMissionProgress(missionProgressList, reason, reasonDetails, location, transactionId);
    }

    /**
     * Method used to claim the rewards for a list of container if the @{@link MissionDataManager.Status} is PENDING_COLLECTION.
     *
     * @param containerProgressList The container progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void claimContainersReward(ArrayList<ContainerProgress> containerProgressList, String reason, String reasonDetails, String location, String transactionId) {
        UserDataManager.getInstance(context).getMissionDataManager().claimContainersReward(containerProgressList, reason, reasonDetails, location, transactionId);
    }

    /**
     * Method used to claim the rewards for a list of missions if the @{@link MissionDataManager.Status} is PENDING_COLLECTION.
     *
     * @param missionProgressList The mission progress list containing updates.
     * @param reason              The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails       The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location            The location from which the update event has been triggered.
     * @param transactionId       The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void claimMissionsReward(ArrayList<MissionProgress> missionProgressList, String reason, String reasonDetails, String location, String transactionId) {
        UserDataManager.getInstance(context).getMissionDataManager().claimMissionsReward(missionProgressList, reason, reasonDetails, location, transactionId);
    }

    /**
     * Method used to reset the information contained in the Player Wallet.
     */
    public void resetWallet() {
        UserDataManager.getInstance(context).getPlayerDataManager().resetWallet();
    }

    /**
     * Method used to reset the information contained in the Player Inventory.
     */
    public void resetInventory() {
        UserDataManager.getInstance(context).getPlayerDataManager().resetInventory();
    }

    /**
     * Method used to reset the information contained in the Player Data (Wallet and Inventory).
     */
    public void resetPlayerData() {
        UserDataManager.getInstance(context).getPlayerDataManager().resetPlayerData(true);
    }

    public void resetMissionData() {
        UserDataManager.getInstance(context).getMissionDataManager().resetMissionDataProgress();
    }

    /**
     * Method used to retrieve the locally stored image path, given by the url.
     *
     * @param url The image url for which the path needs to be retrieve. It can be part of an item, bundle or custom.
     * @return The local path stored after downloading the specified image.
     */
    public String getImagePath(String url) {
        return ImageLoaderUtil.getInstance(context).getImageLocalPath(url);
    }

    /**
     * Method that downloads an image given the url and stores it into a disk cache.
     *
     * @param url                The url from which the image has to be downloaded.
     * @param id                 The id of the item or the bundle if the image belongs to either of them.
     * @param imageType          The image type defined by either being item, bundle or custom.
     * @param imageLoadCallbacks Used only for native in case it is required to use local callbacks instead of the global ones. Should pass null if using Unity.
     */
    public void requestImage(String url, int id, String imageType, ImageLoadCallbacks imageLoadCallbacks) {
        ImageLoaderUtil.getInstance(context).requestImage(url, id, imageType, imageLoadCallbacks);
    }

    /**
     * Method that downloads all the images associated with items and bundles in the background.
     * Once finished, this method will call the global callback to notify the action has been completed.
     */
    public void preloadItemAndBundleImages() {
        ImageLoaderUtil.getInstance(context).preloadImages();
    }

    /**
     * Method that removes all the stored images from the disk cache.
     */
    public void clearDiskCache() {
        ImageLoaderUtil.getInstance(context).clearDiskCache();
    }

    /**
     * Method that returns a Bitmap object, given the local path of an image. Used only natively.
     *
     * @param localPath The local path of a given image.
     * @return The Bitmap object which contains the image.
     */
    public Bitmap getBitmapImage(String localPath) {
        return ImageLoaderUtil.getInstance(context).getBitmapImage(localPath);
    }

    /**
     * Method used to open a WebView that shows the Web Zendesk Help Center.
     */
    public void showZendeskWebViewHelpCenter(String url, String tags) {
        GamedockSDK.getInstance(context).isShowingChildActivity = true;

        Intent intent = new Intent(context, SupportActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra("webViewUrl", url);
        intent.putExtra("tags", tags);
        context.startActivity(intent);
    }

    /**
     * Method used to request the daily bonus screen from the Gamedock server if available.
     */
    public void requestDailyBonus() {
        DailyBonusManager.getInstance(context).requestDailyBonus();
    }

    public void showDailyBonus() {
        DailyBonusManager.getInstance(context).showDailyBonus();
    }

    public String getDailyBonusConfig() {
        return DailyBonusManager.getInstance(context).getDailyBonusConfig();
    }

    public void collectDailyBonus() {
        DailyBonusManager.getInstance(context).sendCollectDailyBonus();
    }

    /**
     * Method used to request the splash screen from the Gamedock server if available.
     */
    public void requestSplashScreen(String type) {
        SplashscreenManager.getInstance(context).requestSplashScreen(type);
    }

    /**
     * Method used to show the splash screen from the Gamedock server if available.
     */
    public void showSplashScreen() {
        SplashscreenManager.getInstance(context).showSplashscreen();
    }

    /**
     * Method used to request all dangerous permissions automatically.
     */
    public void requestAllDangerousPermissions() {
        PermissionUtil.requestAllDangerousPermissions(context, false, true, true);
    }

    /**
     * Method used to request all dangerous permissions automatically.
     *
     * @param isAIR Defines if the request comes from the AIR platform.
     */
    public void requestAllDangerousPermissions(boolean isAIR) {
        if (isAIR) {
            PermissionUtil.requestAllDangerousPermissionsForAir(context);
        }
    }

    /**
     * Method used to request a permission at a given time.
     *
     * @param permission The permission that needs to be request. Should have the format defined in the official Android documentation or use the @{@link android.Manifest.permission} class.
     * @param rationale  The reason for which the permission has been requested.
     */
    public void requestPermission(String permission, String rationale, String denyRationale) {
        HashMap<String, PermissionUtil.PermissionRationaleTemp> permissionsMap = new HashMap<>();

        PermissionUtil.PermissionRationaleTemp permissionRationaleTemp = new PermissionUtil.PermissionRationaleTemp();
        permissionRationaleTemp.rationale = rationale;
        permissionRationaleTemp.denyRationale = denyRationale;
        permissionsMap.put(permission, permissionRationaleTemp);

        PermissionUtil.requestPermissions(context, permissionsMap, false, true, true);
    }

    /**
     * Method used to request a permission at a given time.
     *
     * @param permission The permission that needs to be request. Should have the format defined in the official Android documentation or use the @{@link android.Manifest.permission} class.
     * @param rationale  The reason for which the permission has been requested.
     * @param isAIR      Defines if the request comes from the AIR platform.
     */
    public void requestPermission(String permission, String rationale, String denyRationale, boolean isAIR) {
        if (isAIR) {
            PermissionUtil.requestPermissionForAir(permission, rationale, denyRationale, context);
        }
    }

    /**
     * Method that sends an event which tries to claim the reward associated to a token.
     *
     * @param token      The token that has been received either through deep link, push notification etc.
     * @param rewardType The reward type for the token. Can be deepLink, pushNotification etc.
     */
    public void claimToken(String token, String rewardType) {
        RewardManager.claimToken(context, token, rewardType);
    }

    /**
     * Method that checks if the app was opened using a dynamic link.
     */
    public void requestDynamicLinkApi() {
        if (handler == null) {
            return;
        }

        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                try {
                    DeepLinkManager.requestFirebaseDynamicLinkApi(context);
                } catch (NoClassDefFoundError | Exception e) {
                    LoggingUtil.e("Gamedock Firebase Module not included! If you want to use Firebase please include the gamedock-sdk-firebase dependency");
                }
            }
        }, 2000);
    }

    /**
     * Method that requests the server time.
     */
    public void requestServerTime() {
        ServerDataEvent serverDataEvent = new ServerDataEvent(context);
        serverDataEvent.setRequestServerTime();

        trackEvent(serverDataEvent, null);
    }

    public void requestTieredEvents() {
        TieredEventsManager.getInstance(context).requestTieredEvents();
    }

    public String getAllTieredEvents() {
        return TieredEventsManager.getInstance(context).getAllTieredEvents();
    }

    public String getTieredEvent(int tieredEventId) {
        return TieredEventsManager.getInstance(context).getTieredEvent(tieredEventId);
    }

    public String getTieredEventProgress(int tieredEventId) {
        return TieredEventsManager.getInstance(context).getTieredEventProgress(tieredEventId);
    }

    public void claimTierReward(int tieredEventId, int tierId) {
        TieredEventsManager.getInstance(context).claimTierReward(tieredEventId, tierId);
    }

    public void showTierProgress(int tieredEventId) {
        TieredEventsManager.getInstance(context).showTierProgress(tieredEventId);
    }

    /**
     * Method that initiates the Gamedock Social Login using the provided parameters.
     *
     * @param socialId       The social id received from the social network with which the user has logged in.
     * @param socialProvider The social provider (eg. Facebook, Google Play Games).
     * @param socialToken    The social token received from the social network after the user has logged in.
     */
    public void userLogin(String socialId, String socialProvider, String socialToken, String socialValidationData) {
        SocialManager.getInstance(context).userLogin(socialId, socialProvider, socialToken, socialValidationData);
    }

    /**
     * Method that initiates the Gamedock Social Logout.
     *
     * @param global Flag that indicates if a global logout should be initiated on the server (logging out all devices associated with the account).
     */
    public void userLogout(boolean global) {
        SocialManager.getInstance(context).userLogout(global);
    }

    /**
     * Method that informs the backend that the user has chosen to play as a guest and not login.
     * Only used when an OnAuthError occurs and the user selects this option and not to re-login.
     */
    public void userPlayAsGuest() {
        SocialManager.getInstance(context).userPlayAsGuest();
    }

    /**
     * Method used to trigger the unauthorized dialog if an 401 unauthorized error has occurred.
     *
     * @param title           The title of the dialog.
     * @param message         The message to be displayed by the dialog.
     * @param loginText       The text for the login button.
     * @param playAsGuestText The text for the play as guest button.
     */
    public void showUnauthorizedDialog(String title, String message, String loginText, String playAsGuestText) {
        SocialManager.getInstance(context).showUnauthorizedDialog(title, message, loginText, playAsGuestText);
    }

    public String getIapDetails(String skuIds) {
        String[] stringArray = getGson().fromJson(skuIds, String[].class);
        ArrayList<String> skuIdList = new ArrayList<>(Arrays.asList(stringArray));

        if (IAPUtil.store.equals("GooglePlay")) {
            return IAPUtil.getGooglePlayIapDetails(context, skuIdList);
        } else if (IAPUtil.store.equals("Amazon")) {
            return IAPUtil.getAmazonIapDetails(context, skuIdList);
        }

        return null;
    }

    /**
     * Method that checks if the user is logged in.
     *
     * @return Returns if the user is logged in or not.
     */
    public boolean isLoggedIn() {
        return getGamedockToken() != null;
    }

    public String getDeviceId() {
        return UserIDGenerator.getUniqueDeviceId(context);
    }

    /**
     * Method used to display a native dialog.
     *
     * @param title      The title of the dialog.
     * @param message    The message of the dialog.
     * @param buttonText The text of the button.
     */
    public void showNativeDialog(String title, String message, String buttonText) {
        DeviceUtil.showNativeDialog(context, title, message, buttonText);
    }

    /**
     * Method used to configure the GDPR settings in the SDK manually.
     *
     * @param withPersonalisedAds     Tells the SDK if personlised ads should be used.
     * @param withPersonalisedContent Tells the SDK if personlised content should be used.
     */
    public void setGDPRSettings(boolean withPersonalisedAds, boolean withPersonalisedContent) {
        int priv = 0;
        if (!withPersonalisedContent && !withPersonalisedAds) {
            priv = 0;
        } else if (withPersonalisedContent && !withPersonalisedAds) {
            priv = 1;
        } else if (!withPersonalisedContent && withPersonalisedAds) {
            priv = 2;
        } else if (withPersonalisedContent && withPersonalisedAds) {
            priv = 3;
        }

        savePrivValue(priv);
    }

    /**
     * Method used to retrieve the GDPR settings.
     *
     * @return A map containing the two keys (withPersonalisedAds, withPersonalisedContent) with the information.
     */
    public HashMap<String, Boolean> getGDPRSettings() {
        HashMap<String, Boolean> gdprSettings = new HashMap<>();

        boolean withPersonalisedAds = false;
        boolean withPersonalisedContent = false;

        int priv = getPrivValue();

        switch (priv) {
            case 0:
                withPersonalisedContent = false;
                withPersonalisedAds = false;
                break;
            case 1:
                withPersonalisedContent = true;
                withPersonalisedAds = false;
                break;
            case 2:
                withPersonalisedContent = false;
                withPersonalisedAds = true;
                break;
            case 3:
                withPersonalisedContent = true;
                withPersonalisedAds = true;
                break;
        }

        gdprSettings.put("withPersonalisedAds", withPersonalisedAds);
        gdprSettings.put("withPersonalisedContent", withPersonalisedContent);

        return gdprSettings;

    }

    public void savePrivValue(int priv) {
        getStorageUtil().putInt(StorageUtil.Keys.GDPRStatus, priv);
    }

    public int getPrivValue() {
        return getStorageUtil().getInt(StorageUtil.Keys.GDPRStatus, -1);
    }

    /**
     * Method used to send a request event to fetch the asset bundle configurations from the backend.
     * If no internet is available will check if any locally stored configurations are available and inform the game accordingly.
     */
    public void requestAssetBundles() {
        AssetBundlesManager.getInstance(context).requestAssetBundles();
    }

    public String getAssetBundles() {
        return AssetBundlesManager.getInstance(context).getAssetBundlesJSON();
    }

    /**
     * Method used to fetch the Firebase remote configuration from the Firebase backend.
     *
     * @param cacheExpiration The cache expiration value that Firebase takes into account on. It is used to determine if a server request will be done or the local values will be provided.
     */
    public void requestFirebaseRemoteConfig(long cacheExpiration) {
        FirebaseRemoteConfigManager.fetchFirebaseRemoteConfig(context, cacheExpiration);
    }

    public void setFirebaseRemoteConfigDefaults(String defaultValues) {
        FirebaseRemoteConfigManager.setDefaultParameterValues(defaultValues);
    }

    public boolean getBooleanFirebaseRemoteConfig(String key, String namespace) {
        return FirebaseRemoteConfigManager.getBoolean(key, namespace);
    }

    public double getDoubleFirebaseRemoteConfig(String key, String namespace) {
        return FirebaseRemoteConfigManager.getDouble(key, namespace);
    }

    public long getLongFirebaseRemoteConfig(String key, String namespace) {
        return FirebaseRemoteConfigManager.getLong(key, namespace);
    }

    public String getStringFirebaseRemoteConfig(String key, String namespace) {
        return FirebaseRemoteConfigManager.getString(key, namespace);
    }

    /**
     * Method used to request a localization based on a supplied locale.
     * @param locale The requested locale. For a list of the local codes, check {@link LocalizationManager.Locales}.
     * @param fallbackToDefaultLocale Flag to inform that when a key in the localization is not present, should it fall back to the default locale or not.
     */
    public void requestLocalization(String locale, boolean fallbackToDefaultLocale) {
        LocalizationManager.getInstance(context).requestLocalization(locale, fallbackToDefaultLocale);
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     * @param key The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue) {
        return LocalizationManager.getInstance(context).getLocalization(key, defaultValue);
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     * @param key The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @param args List of arguments which will be used to replace dynamic values in the string ({0}, {1}, ...).
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue, String ... args) {
        return LocalizationManager.getInstance(context).getLocalization(key, defaultValue, args);
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     * @param key The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @param args Map of key value pairs used to replace instances of the supplied keys in the localized string ({someKey}, {someOtherKey}, ...).
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue, HashMap<String, String> args) {
        return LocalizationManager.getInstance(context).getLocalization(key, defaultValue, args);
    }

    /**
     * Method used to show the application's settings menu.
     */
    public void showAppSettings() {
        DeviceUtil.showAppSettings(context);
    }

    public InitializationDataCallbacks getInitializationDataCallbacks() {
        if (initializationDataCallbacks != null) {
            return initializationDataCallbacks;
        }
        return new InitializationDataCallbacks();
    }

    public void setInitializationDataCallbacks(InitializationDataCallbacks initializationDataCallbacks) {
        GamedockSDK.initializationDataCallbacks = initializationDataCallbacks;
    }

    public MoreAppsCallbacks getMoreAppsCallbacks() {
        if (moreAppsCallbacks != null) {
            return moreAppsCallbacks;
        }
        return new MoreAppsCallbacks();
    }

    public void setMoreAppsCallbacks(MoreAppsCallbacks moreAppsCallbacks) {
        GamedockSDK.moreAppsCallbacks = moreAppsCallbacks;
    }

    public GamedockGameDataCallbacks getGameDataCallbacks() {
        if (gamedockGameDataCallbacks != null) {
            return gamedockGameDataCallbacks;
        }
        gamedockGameDataCallbacks = new GamedockGameDataCallbacks();
        return gamedockGameDataCallbacks;
    }

    public void setGameDataCallbacks(GamedockGameDataCallbacks gamedockGameDataCallbacks) {
        GamedockSDK.gamedockGameDataCallbacks = gamedockGameDataCallbacks;
    }

    public UserDataCallbacks getUserDataCallbacks() {
        if (userDataCallbacks != null) {
            return userDataCallbacks;
        }
        userDataCallbacks = new UserDataCallbacks();
        return userDataCallbacks;
    }

    public void setUserDataCallbacks(UserDataCallbacks userDataCallbacks) {
        GamedockSDK.userDataCallbacks = userDataCallbacks;
    }

    public MissionConfigurationCallbacks getMissionConfigurationCallbacks() {
        if (missionConfigurationCallbacks != null) {
            return missionConfigurationCallbacks;
        }
        missionConfigurationCallbacks = new MissionConfigurationCallbacks();
        return missionConfigurationCallbacks;
    }

    public void setMissionConfigurationCallbacks(MissionConfigurationCallbacks missionConfigurationCallbacks) {
        GamedockSDK.missionConfigurationCallbacks = missionConfigurationCallbacks;
    }

    public RewardCallbacks getRewardCallbacks() {
        if (rewardCallbacks != null) {
            return rewardCallbacks;
        }
        rewardCallbacks = new RewardCallbacks();
        return rewardCallbacks;
    }

    public void setRewardCallbacks(RewardCallbacks rewardCallbacks) {
        GamedockSDK.rewardCallbacks = rewardCallbacks;
    }

    public ConfigDataCallbacks getConfigDataCallbacks() {
        if (configDataCallbacks != null) {
            return configDataCallbacks;
        }
        configDataCallbacks = new ConfigDataCallbacks();
        return configDataCallbacks;
    }

    public void setConfigDataCallbacks(ConfigDataCallbacks configDataCallbacks) {
        GamedockSDK.configDataCallbacks = configDataCallbacks;
    }

    public SplashScreenCallbacks getSplashScreenCallbacks() {
        if (splashScreenCallbacks != null) {
            return splashScreenCallbacks;
        }
        splashScreenCallbacks = new SplashScreenCallbacks();
        return splashScreenCallbacks;
    }

    public void setSplashScreenCallbacks(SplashScreenCallbacks splashScreenCallbacks) {
        GamedockSDK.splashScreenCallbacks = splashScreenCallbacks;
    }

    public DailyBonusCallbacks getDailyBonusCallbacks() {
        if (dailyBonusCallbacks != null) {
            return dailyBonusCallbacks;
        }
        dailyBonusCallbacks = new DailyBonusCallbacks();
        return dailyBonusCallbacks;
    }

    public void setDailyBonusCallbacks(DailyBonusCallbacks dailyBonusCallbacks) {
        GamedockSDK.dailyBonusCallbacks = dailyBonusCallbacks;
    }

    public ImageLoadCallbacks getImageLoadCallbacks() {
        if (imageLoadCallbacks != null) {
            return imageLoadCallbacks;
        }
        imageLoadCallbacks = new ImageLoadCallbacks();
        return imageLoadCallbacks;
    }

    public void setImageLoadCallbacks(ImageLoadCallbacks imageLoadCallbacks) {
        GamedockSDK.imageLoadCallbacks = imageLoadCallbacks;
    }

    public IAPCallbacks getIapCallbacks() {
        if (iapCallbacks != null) {
            return iapCallbacks;
        }
        iapCallbacks = new IAPCallbacks();
        return iapCallbacks;
    }

    public void setIapCallbacks(IAPCallbacks iapCallbacks) {
        GamedockSDK.iapCallbacks = iapCallbacks;
    }

    public ServerDataCallbacks getServerDataCallbacks() {
        if (serverDataCallbacks != null) {
            return serverDataCallbacks;
        }
        serverDataCallbacks = new ServerDataCallbacks();
        return serverDataCallbacks;
    }

    public void setServerDataCallbacks(ServerDataCallbacks serverDataCallbacks) {
        GamedockSDK.serverDataCallbacks = serverDataCallbacks;
    }

    public PermissionCallbacks getPermissionCallbacks() {
        if (permissionCallbacks != null) {
            return permissionCallbacks;
        }
        permissionCallbacks = new PermissionCallbacks();
        return permissionCallbacks;
    }

    public void setPermissionCallbacks(PermissionCallbacks permissionCallbacks) {
        GamedockSDK.permissionCallbacks = permissionCallbacks;
    }

    public SocialCallbacks getSocialCallbacks() {
        if (socialCallbacks != null) {
            return socialCallbacks;
        }
        socialCallbacks = new SocialCallbacks();
        return socialCallbacks;
    }

    public void setSocialCallbacks(SocialCallbacks socialCallbacks) {
        GamedockSDK.socialCallbacks = socialCallbacks;
    }

    public AzerionConnectCallbacks getAzerionConnectCallbacks() {
        if (azerionConnectCallbacks != null) {
            return azerionConnectCallbacks;
        }
        azerionConnectCallbacks = new AzerionConnectCallbacks();
        return azerionConnectCallbacks;
    }

    public AgeGateCallbacks getAgeGateCallbacks() {
        if (ageGateCallbacks != null) {
            return ageGateCallbacks;
        }
        ageGateCallbacks = new AgeGateCallbacks();
        return ageGateCallbacks;
    }

    public void setAgeGateCallbacks(AgeGateCallbacks ageGateCallbacks) {
        GamedockSDK.ageGateCallbacks = ageGateCallbacks;
    }

    public PrivacyPolicyCallbacks getPrivacyPolicyCallbacks() {
        if (privacyPolicyCallbacks != null) {
            return privacyPolicyCallbacks;
        }
        privacyPolicyCallbacks = new PrivacyPolicyCallbacks();
        return privacyPolicyCallbacks;
    }

    public void setPrivacyPolicyCallbacks(PrivacyPolicyCallbacks privacyPolicyCallbacks) {
        GamedockSDK.privacyPolicyCallbacks = privacyPolicyCallbacks;
    }

    public PackagesCallbacks getPackagesCallbacks() {
        if (packagesCallbacks != null) {
            return packagesCallbacks;
        }
        packagesCallbacks = new PackagesCallbacks();
        return packagesCallbacks;
    }

    public void setPackagesCallbacks(PackagesCallbacks packagesCallbacks) {
        GamedockSDK.packagesCallbacks = packagesCallbacks;
    }

    public PromotionsCallbacks getPromotionsCallbacks() {
        if (promotionsCallbacks != null) {
            return promotionsCallbacks;
        }
        promotionsCallbacks = new PromotionsCallbacks();
        return promotionsCallbacks;
    }

    public void setPromotionsCallbacks(PromotionsCallbacks promotionsCallbacks) {
        GamedockSDK.promotionsCallbacks = promotionsCallbacks;
    }

    public TieredEventsCallbacks getTieredEventCallbacks() {
        if (tieredEventsCallbacks != null) {
            return tieredEventsCallbacks;
        }
        tieredEventsCallbacks = new TieredEventsCallbacks();
        return tieredEventsCallbacks;
    }

    public void setTieredEventCallbacks(TieredEventsCallbacks tieredEventsCallbacks) {
        GamedockSDK.tieredEventsCallbacks = tieredEventsCallbacks;
    }

    public AssetBundlesCallbacks getAssetBundlesCallbacks() {
        if (assetBundlesCallbacks != null) {
            return assetBundlesCallbacks;
        }
        assetBundlesCallbacks = new AssetBundlesCallbacks();
        return assetBundlesCallbacks;
    }

    public void setAssetBundlesCallbacks(AssetBundlesCallbacks assetBundlesCallbacks) {
        GamedockSDK.assetBundlesCallbacks = assetBundlesCallbacks;
    }

    public DeepLinkCallbacks getDeepLinkCallbacks() {
        if (deepLinkCallbacks != null) {
            return deepLinkCallbacks;
        }
        deepLinkCallbacks = new DeepLinkCallbacks();
        return deepLinkCallbacks;
    }

    public void setDeepLinkCallbacks(DeepLinkCallbacks deepLinkCallbacks) {
        GamedockSDK.deepLinkCallbacks = deepLinkCallbacks;
    }

    public LocalizationCallbacks getLocalizationCallbacks() {
        if (localizationCallbacks != null) {
            return localizationCallbacks;
        }
        localizationCallbacks = new LocalizationCallbacks();
        return localizationCallbacks;
    }

    public void setLocalizationCallbacks(LocalizationCallbacks localizationCallbacks) {
        GamedockSDK.localizationCallbacks = localizationCallbacks;
    }

    /**
     * Method called whenever the following functions are need (it is normally called in the Activity onResume):
     * - starts session timer
     * - requests game data
     * - requests player data
     * - requests game state data
     * - requests advertisement information
     * - tracks sessionStart event
     * - initialises heartbeat session
     */
    public void reportActivityStart() {
        boolean isPrivacyPolicyAccepted = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);

        if (isAppRunning || !isPrivacyPolicyAccepted) {
            LoggingUtil.i("App already running or Privacy Policy not accepted! Not firing a session start!");
            return;
        }

        isAppRunning = true;

        SessionUtil.startSession(context);

        Event event = new Event(context);
        event.setName("sessionStart");

        trackEvent(event, null);

        startHeartbeatSession();
        startPerformanceTracking();
    }

    /**
     * Method that does the following functions (it is normally called in the Activity onPause):
     * - stops session timer
     * - tracks sessionStop event
     * - stops heartbeat session
     */
    public void reportActivityStop() {
        boolean isPrivacyPolicyAccepted = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);

        if (!isAppRunning || !isPrivacyPolicyAccepted) {
            LoggingUtil.i("App is not running or Privacy Policy not accepted! Not firing new session stop!");
            return;
        }

        if (isShowingChildActivity) {
            return;
        }

        isAppRunning = false;

        Event event = new Event(context);
        event.setName("sessionStop");

        trackEvent(event, null);

        SessionUtil.stopSession(context);

        stopHeartbeatSession();
        stopPerformanceTracking();
    }

    public synchronized void init(Activity activity, InitializationOptions receivedInitializationOptions) {
        if (receivedInitializationOptions == null) {
            LoggingUtil.e("Gamedock SDK cannot initialize! Initialization options are null!");
            return;
        }

        if (activity == null) {
            LoggingUtil.e("Gamedock SDK cannot initialize! Activity is null!");
            return;
        }

        GamedockSDK.resetContext(activity);

        onCreate();

        requestAllDangerousPermissions();

        this.initializationOptions = receivedInitializationOptions;

        setPluginInformation(initializationOptions.getPluginName(), initializationOptions.getPluginVersion());
        setEnvironment(initializationOptions.getEnvironment());

        setInitializationDataCallbacks(initializationOptions.getInitializationDataCallbacks());
        setMoreAppsCallbacks(initializationOptions.getMoreAppsCallbacks());
        setGameDataCallbacks(initializationOptions.getGamedockGameDataCallbacks());
        setUserDataCallbacks(initializationOptions.getUserDataCallbacks());
        setMissionConfigurationCallbacks(initializationOptions.getMissionConfigurationCallbacks());
        setRewardCallbacks(initializationOptions.getRewardCallbacks());
        setConfigDataCallbacks(initializationOptions.getConfigDataCallbacks());
        setSplashScreenCallbacks(initializationOptions.getSplashScreenCallbacks());
        setDailyBonusCallbacks(initializationOptions.getDailyBonusCallbacks());
        setImageLoadCallbacks(initializationOptions.getImageLoadCallbacks());
        setIapCallbacks(initializationOptions.getIapCallbacks());
        setServerDataCallbacks(initializationOptions.getServerDataCallbacks());
        setPermissionCallbacks(initializationOptions.getPermissionCallbacks());
        setSocialCallbacks(initializationOptions.getSocialCallbacks());
        setAgeGateCallbacks(initializationOptions.getAgeGateCallbacks());
        setPrivacyPolicyCallbacks(initializationOptions.getPrivacyPolicyCallbacks());
        setPackagesCallbacks(initializationOptions.getPackagesCallbacks());
        setPromotionsCallbacks(initializationOptions.getPromotionsCallbacks());
        setTieredEventCallbacks(initializationOptions.getTieredEventsCallbacks());
        setAssetBundlesCallbacks(initializationOptions.getAssetBundlesCallbacks());
        setDeepLinkCallbacks(initializationOptions.getDeepLinkCallbacks());
        setLocalizationCallbacks(initializationOptions.getLocalizationCallbacks());

        if (initializationOptions.isWithAgeGate()) {
            if (initializationOptions.getAgeGateOptions() == null) {
                LoggingUtil.e("Gamedock SDK age gate cannot be shown! Age Gate Options are null!");
            }

            LoggingUtil.d("Gamedock SDK age gate requested at init point due to options");
            checkAgeGate(initializationOptions.getAgeGateOptions().shouldBlock, initializationOptions.getAgeGateOptions().requiredAge, true);
        } else if (initializationOptions.isWithPrivacyPolicy()) {
            LoggingUtil.d("Gamedock SDK privacy policy requested at init point due to options");
            checkPrivacyPolicy(true);
        } else {
            String externalIdsStoredString = getStorageUtil().getString(StorageUtil.Keys.ExternalPartnersIds, null);

            if (externalIdsStoredString != null) {
                Type listType = new TypeToken<HashMap<String, String>>() {
                }.getType();
                this.externalIds = getGson().fromJson(externalIdsStoredString, listType);
            }

            this.externalIds.putAll(initializationOptions.getExternalIds());

            getStorageUtil().putString(StorageUtil.Keys.ExternalPartnersIds, getGson().toJson(this.externalIds));

            getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyStatus, true);
            getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyAsked, false);

            LoggingUtil.w("Gamedock SDK will initialise without Privacy Policy consent or Age Gate!");
            setCoppaEnabled(initializationOptions.isCoppaEnabled());

            init();
        }
    }

    /**
     * Method used to initialise the SDK. Should be called by developer.
     */
    public synchronized void init(boolean withPrivacyPolicy, boolean coppaEnabled, HashMap<String, String> externalIds) {
        String externalIdsStoredString = getStorageUtil().getString(StorageUtil.Keys.ExternalPartnersIds, null);

        if (externalIdsStoredString != null) {
            Type listType = new TypeToken<HashMap<String, String>>() {
            }.getType();
            this.externalIds = getGson().fromJson(externalIdsStoredString, listType);
        }

        this.externalIds.putAll(externalIds);

        getStorageUtil().putString(StorageUtil.Keys.ExternalPartnersIds, getGson().toJson(this.externalIds));

        init(withPrivacyPolicy, coppaEnabled);
    }

    /**
     * Method used to initialise the SDK. Should be called by developer.
     */
    public synchronized void init(boolean withPrivacyPolicy, boolean coppaEnabled) {
        if (!withPrivacyPolicy) {
            getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyStatus, true);
            getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyAsked, false);

            LoggingUtil.w("Gamedock SDK will initialise without Privacy Policy consent!");
        }

        setCoppaEnabled(coppaEnabled);

        init();
    }

    /**
     * Method used to initialise the SDK. Should be called by developer.
     */
    private synchronized void init() {
        boolean isPrivacyPolicyAccepted = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);

        if (!isPrivacyPolicyAccepted) {
            LoggingUtil.w("Privacy Policy not accepted! SDK will not initialise!");
            return;
        }

        if (sdkInitialised) {
            LoggingUtil.w("SDK already initialised!");
            return;
        }

        sdkInitialised = true;

        initComponents();

        InitializationManager.initializeSDK(context);

        //Register runnable to execute features init if initializeSDK event fails
        if (handler != null) {
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    GamedockSDK.getInstance(context).initFeatures();
                }
            }, 8000);
        } else if (mainThreadHandler != null) {
            mainThreadHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    GamedockSDK.getInstance(context).initFeatures();
                }
            }, 8000);
        }
    }

    /**
     * Method used to initialise SDK features
     */

    public synchronized void initFeatures() {
        if (sdkFeaturesInitialised) {
            return;
        }

        LoggingUtil.d("Initialising SDK Features");

        sdkFeaturesInitialised = true;

        reportActivityStart();
        if (initializationOptions != null) {
            registerDevice(initializationOptions.getStore());
        } else {
            registerDevice("GooglePlay");
        }

        requestConfig();

        requestGameData();
        requestMissionConfiguration();
        requestPromotions();
        requestUserData();

        requestAssetBundles();

        requestDynamicLinkApi();
        processNotification();

        if (handler != null) {
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    getInitializationDataCallbacks().initializationCompleted();
                }
            }, 2000);
        } else if (mainThreadHandler != null) {
            mainThreadHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    getInitializationDataCallbacks().initializationCompleted();
                }
            }, 2000);
        }
    }

    /**
     * Method used to initialise Firebase, Google Analytics, Adjust, and other internal SDK components.
     */
    public synchronized void initComponents() {
        if (sdkComponentsInitialised) {
            return;
        }

        LoggingUtil.d("Initialising SDK Components");

        sdkComponentsInitialised = true;


        //Check debug mode
        GamedockConfigManager.getInstance(context).setDebugMode(context);

        //Firebase
        initialiseFirebase();

        //Google Analytics
        initialiseGoogleAnalytics();


        NetworkUtil.registerNetworkStatusReceiver(context);
    }

    /**
     * Method used by the developer to reset the game and user specific data of the SDK
     * Resets the following: config, packages, game data, player data and game state.
     */
    public void resetData() {
        if (!NetworkUtil.isInternetAvailable(context) && isLoggedIn()) {
            getInstance(context).getConfigDataCallbacks().configError(io.gamedock.sdk.utils.error.ErrorCodes.ConfigResetError);
            getInstance(context).getGameDataCallbacks().gameDataError(io.gamedock.sdk.utils.error.ErrorCodes.GameDataResetError);
            getInstance(context).getUserDataCallbacks().userDataError(io.gamedock.sdk.utils.error.ErrorCodes.UserDataResetError);

            return;
        }

        //Resetting config
        ConfigManager.getInstance(context).resetConfig();
        requestConfig();

        //Resetting packages
        PackageManager.getInstance(context).resetPackages();

        //Resetting game data
        GamedockGameDataManager.getInstance(context).resetGameData();
        requestGameData();

        MissionConfigurationManager.getInstance(context).resetMissionConfiguration();
        requestMissionConfiguration();

        PromotionsManager.getInstance(context).resetPromotionData();
        requestPromotions();

        //Resetting player data and game state
        StorageUtil.getInstance(context).clearUserSpecificData();
        UserDataManager.getInstance(context).resetUserData();
        if (SocialManager.getInstance(context).getSocialId() != null && SocialManager.getInstance(context).getSocialProvider() != null) {
            setUserId(SocialManager.getInstance(context).getSocialProvider(), SocialManager.getInstance(context).getSocialId());
            SocialManager.getInstance(context).setSocialId(null);
            SocialManager.getInstance(context).setSocialProvider(null);
        }
        requestUserData();

        //Resetting social manager
        SocialManager.getInstance(context).resetSocialManager();

        //Resetting tiered event manager
        TieredEventsManager.getInstance(context).resetTieredEvents();

        //Resetting storage util
        StorageUtil.getInstance(context).resetStorageUtil();

        //Resetting event manager
        EventManager.getInstance(context).resetEventManager();

        //Resetting Gamedock config manager
        GamedockConfigManager.getInstance(context).resetGamedockConfigManager();

        //Resetting Asset Bundles
        AssetBundlesManager.getInstance(context).resetAssetBundlesData();
        requestAssetBundles();

        //Resetting Daily Bonus
        DailyBonusManager.getInstance(context).resetDailyBonusConfig();
        DailyBonusManager.getInstance(context).requestDailyBonus();

        //Resetting Splashscreen
        SplashscreenManager.getInstance(context).resetSplashscreen();
    }


    /**
     * Method that initialises the Google Analytics SDK.
     */
    private synchronized void initialiseGoogleAnalytics() {
        try {
            LoggingUtil.d("Enabling Google Analytics");
            if (mTracker == null) {
                GoogleAnalytics analytics = GoogleAnalytics.getInstance(context);
                mTracker = analytics.newTracker(GamedockConfigManager.getInstance(context).googleServices.analyticsId);
                mTracker.enableExceptionReporting(true);

                mTracker.enableAutoActivityTracking(true);
            }
        } catch (Exception | NoClassDefFoundError e) {
            LoggingUtil.e("Google Analytics couldn't be initialised. " +
                    "Please make sure you have the necessary dependencies and you included the 'googleAnalyticsId' key-value in your 'defaultGameConfig.json' file");
        }
    }

    /**
     * Method that initialises the Firebase SDK.
     */
    private void initialiseFirebase() {
        try {
            if (!GamedockConfigManager.getInstance(context).firebase.applicationId.isEmpty()) {
                LoggingUtil.d("Enabling Firebase");
                FirebaseOptions firebaseOptions = new FirebaseOptions.Builder()
                        .setApplicationId(GamedockConfigManager.getInstance(context).firebase.applicationId)
                        .setApiKey(GamedockConfigManager.getInstance(context).firebase.apiKey)
                        .setGcmSenderId(GamedockConfigManager.getInstance(context).firebase.gcmSenderId)
                        .setDatabaseUrl(GamedockConfigManager.getInstance(context).firebase.databaseUrl)
                        .setStorageBucket(GamedockConfigManager.getInstance(context).firebase.storageBucket)
                        .setProjectId(GamedockConfigManager.getInstance(context).firebase.projectId)
                        .build();

                //Necessary since Robolectric doesn't support the loading of native libraries
                if (!isUnitTesting()) {
                    FirebaseApp.initializeApp(context, firebaseOptions);
                    FirebaseAnalytics.getInstance(context).setUserId(getGamedockUID());

                    FirebaseCrashlytics.getInstance().setUserId(UserIDGenerator.getGamedockUserId(context));

                    firebaseInitialised = true;

                    UserIDGenerator.getFirebaseInstanceId(context);
                }
            }
        } catch (Exception | NoClassDefFoundError e) {
            LoggingUtil.e("Gamedock Firebase Module not included! If you want to use Firebase please include the gamedock-sdk-firebase dependency");
        }
    }

    /**
     * Method used to sent a custom exception through Crashlytics.
     *
     * @param name            The name of the exception.
     * @param reason          The reason of the exception
     * @param stackTraceArray A JSON stack trace array containing all the stack traces related to the exception.
     */
    public void recordCustomException(String name, String reason, String stackTraceArray) {
        FirebaseCrashlyticsUtil.recordCustomException(name, reason, stackTraceArray);
    }

    /**
     * Method used to record the FPS value received from the game.
     *
     * @param fpsValue The FPS value.
     */
    public void recordFPSValue(double fpsValue) {
        PerformanceUtil.getInstance(context).recordFPSStat(fpsValue);
    }

    public void loginAzerionConnect(String provider){

        Log.i("LOGIN", "loginAzerionConnect called provider " + provider);

        Intent intent = new Intent(context, AzerionConnectActivity.class);
        intent.putExtra("action","login");
        intent.putExtra("provider",provider);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    public void logoutAzerionConnect(){
        Intent intent = new Intent(context, AzerionConnectActivity.class);
        intent.putExtra("action","logout");
        context.startActivity(intent);
    }

    public void getUserInfo(){

        Intent intent = new Intent(context, AzerionConnectActivity.class);
        intent.putExtra("action","userInfo");
        context.startActivity(intent);

    }
    public void openDashboard(){
        String url = "";
        if (GamedockSDK.getInstance(context).getEnvironment() == GamedockEnvironment.STAGING){
            url = "https://myaccount.azcdev.com/";
        }
        else {
            url = "https://myaccount.azerionconnect.com/";
        }
        CustomTabsIntent.Builder builder = new CustomTabsIntent.Builder();
        CustomTabsIntent customTabsIntent = builder.build();
        customTabsIntent.launchUrl(context, Uri.parse(url));
    }

    public synchronized void checkAgeGate(boolean shouldBlock, int requiredAged, boolean triggerCallback) {
        getStorageUtil().putBoolean(StorageUtil.Keys.AgeGateShouldBlock, shouldBlock);
        getStorageUtil().putInt(StorageUtil.Keys.AgeGateRequiredAge, requiredAged);
        getStorageUtil().putBoolean(StorageUtil.Keys.AgeGateShouldTriggerCallback, triggerCallback);

        boolean isAgeGatePassed = getStorageUtil().getBoolean(StorageUtil.Keys.AgeGatePassed, false);
        boolean ageGateAsked = getStorageUtil().getBoolean(StorageUtil.Keys.AgeGateAsked, false);

        if (!isShowingAgeGate && !ageGateAsked) {
            getStorageUtil().putBoolean(StorageUtil.Keys.AgeGatePassed, false);
            isShowingAgeGate = true;

            isShowingChildActivity = true;

            Intent intent = new Intent(context, AgeGateActivity.class);
            intent.putExtra("ageGateRequirement", requiredAged);
            intent.putExtra("ageGateShouldBlock", shouldBlock);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

            context.startActivity(intent);
        } else if (triggerCallback) {
            String ageGateAge = getStorageUtil().getString(StorageUtil.Keys.AgeGateAge, "0-12");
            boolean checkPrivacyPolicy = false;
            if (ageGateAge.equals("16+")) {
                checkPrivacyPolicy = true;
            }

            if (initializationOptions != null) {
                if (initializationOptions.isWithAutoProcessAgeGate() && isAgeGatePassed) {
                    if (initializationOptions.isWithPrivacyPolicy()) {
                        checkPrivacyPolicy(true);
                    } else {
                        init(false, true);
                    }
                }
            }

            getAgeGateCallbacks().ageGateStatus(isAgeGatePassed, ageGateAge, checkPrivacyPolicy);
        }
    }

    /**
     * Method used to check if the GDPR popup has been shown and if not, to display it.
     *
     * @param triggerCallback If the method should trigger a GDPR callback towards the game.
     */
    public synchronized void checkPrivacyPolicy(boolean triggerCallback) {
        getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyShouldTriggerCallback, triggerCallback);

        boolean isPrivacyPolicyAccepted = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);
        boolean privacyPolicyAsked = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyAsked, false);

        if (!isShowingPrivacyPolicy && !privacyPolicyAsked) {
            getStorageUtil().putBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);
            isShowingPrivacyPolicy = true;

            isShowingChildActivity = true;

            Intent intent = new Intent(context, PrivacyPolicyActivity.class);
            intent.putExtra("openId", 0);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

            context.startActivity(intent);
        } else if (triggerCallback) {
            if (initializationOptions != null) {
                if (isPrivacyPolicyAccepted && initializationOptions.isWithAutoProcessPrivacyPolicy()) {
                    init(true, coppaEnabled);
                }
            }

            getPrivacyPolicyCallbacks().privacyPolicyStatus(isPrivacyPolicyAccepted);
        }
    }

    /**
     * Method used to display the GDPR settings screen.
     */
    public synchronized void showPrivacyPolicySettings() {
        isShowingChildActivity = true;

        Intent intent = new Intent(context, PrivacyPolicyActivity.class);
        intent.putExtra("openId", 1);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        context.startActivity(intent);
    }

    /**
     * Method used to add a new external partner.
     *
     * @param externalPartner The name of the partner.
     * @param id              The id associated with the partner.
     */
    public void addExternalId(String externalPartner, String id) {
        this.externalIds.put(externalPartner, id);

        getStorageUtil().putString(StorageUtil.Keys.ExternalPartnersIds, getGson().toJson(this.externalIds));
    }

    /**
     * Method used to remove an external partner.
     *
     * @param externalPartner The name of the partner.
     */
    public void removeExternalId(String externalPartner) {
        this.externalIds.remove(externalPartner);

        getStorageUtil().putString(StorageUtil.Keys.ExternalPartnersIds, getGson().toJson(this.externalIds));
    }

    public void showAppRatePopup(String rejectText, String laterText, String message, String feedbackUrl, int triggerCount, int laterCount) {
        AppRateUtil.showAppRatePopup(context, rejectText, laterText, message, feedbackUrl, triggerCount, laterCount);
    }

    /**
     * Method that needs to be called in the Activity onCreate.
     */
    public void onCreate() {
        GamedockConfigManager.getInstance(context).retrieveGoogleAdvertisingId(context);
    }

    /**
     * Method that needs to be called in the Activity onStart.
     */
    public void onStart() {
    }

    /**
     * Method that needs to be called in the Activity onResume.
     */
    public void onResume() {
        boolean wasShowingAgeGate = getStorageUtil().getBoolean(StorageUtil.Keys.AgeGateWasShowing, false);
        boolean ageGateShouldBlock = getStorageUtil().getBoolean(StorageUtil.Keys.AgeGateShouldBlock, false);
        int ageGateRequiredAge = getStorageUtil().getInt(StorageUtil.Keys.AgeGateRequiredAge, 12);
        boolean ageGateShouldTriggerCallback = getStorageUtil().getBoolean(StorageUtil.Keys.AgeGateShouldTriggerCallback, false);

        boolean wasShowingPrivacyPolicy = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyWasShowing, false);
        boolean privacyPolicyShouldTriggerCallback = getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyShouldTriggerCallback, false);

        if (wasShowingAgeGate) {
            checkAgeGate(ageGateShouldBlock, ageGateRequiredAge, ageGateShouldTriggerCallback);
        } else if (wasShowingPrivacyPolicy) {
            checkPrivacyPolicy(privacyPolicyShouldTriggerCallback);
        } else {
            reportActivityStart();

            if (sdkInitialised) {
                requestDynamicLinkApi();
                NetworkUtil.registerNetworkStatusReceiver(context);
            }
        }
    }

    /**
     * Method that needs to be called in the Activity onPause.
     */
    public void onPause() {
        reportActivityStop();
        NetworkUtil.unregisterNetworkStatusReceiver(context);
        EventManager.getInstance(context).saveRecentEvents();
    }

    /**
     * Method that needs to be called in the Activity onDestroy.
     */
    public void onDestroy() {
        if (SessionUtil.isSessionStarted()) {
            SessionUtil.stopSession(context);
        }

        DeepLinkManager.processedDeepLinkUrl = null;

        if (handler != null) {
            handler.removeCallbacksAndMessages(null);
        }

        if (mainThreadHandler != null) {
            mainThreadHandler.removeCallbacksAndMessages(null);
        }

        if (handlerThread != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                handlerThread.quitSafely();
            } else {
                handlerThread.quit();
            }
        }

        sdkInitialised = false;
        sdkComponentsInitialised = false;
        context = null;
        mInstance = null;
    }

    /**
     * Method that needs to be called in the Activity onBackPressed.
     */
    public void onBackPressed() {
//        try {
//            if (Chartboost.onBackPressed()) {
//                return;
//            }
//        } catch (NoClassDefFoundError e) {
//            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob Module please include the gamedock-sdk-admob dependency");
//        }
    }
}
