package io.gamedock.sdk.web;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.webkit.JavascriptInterface;
import android.webkit.WebView;
import android.widget.Toast;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.dailybonus.DailyBonusManager;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.response.OverlayResponseEvent;
import io.gamedock.sdk.events.response.ResponseEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.gamedata.promotions.PromotionsManager;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.IAP.IAPUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;

/**
 * Class that is used for connecting the android WebView to the JavaScript code of the web page opened.
 * Class also handles the response of the web page and forwards it to the sdk
 */
public class JavascriptBridge {

    private Activity activity;
    private WebView webView;
    private String actionListener;
    private String eventData;

    public JavascriptBridge(Activity activity, WebView webView, Bundle extras) {
        this.activity = activity;
        this.webView = webView;
        actionListener = extras.getString("actionListener");
        eventData = extras.getString("eventData");
    }

    @JavascriptInterface
    public String getEventData() {
        return eventData;
    }

    @JavascriptInterface
    public void closeOverlay() {
        activity.finish();
    }

    @JavascriptInterface
    public void openGameShop() {
        GamedockSDK.getInstance(activity).getSplashScreenCallbacks().splashScreenOpenShop();
    }

    @JavascriptInterface
    public String getWallet() {
        return UserDataManager.getInstance(activity).getPlayerDataManager().getWallet();
    }

    @JavascriptInterface
    public String getInventory() {
        return UserDataManager.getInstance(activity).getPlayerDataManager().getInventory();
    }

    @JavascriptInterface
    public String getGameData() {
        return GamedockGameDataManager.getInstance(activity).getGameDataJSON();
    }

    @JavascriptInterface
    public void openUrl(String url) {
        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
        activity.startActivity(browserIntent);
    }

    @JavascriptInterface
    public void receiveReward(String rewardData) {
        GamedockSDK.getInstance(activity).getDailyBonusCallbacks().dailyBonusReward(rewardData);
    }

    @JavascriptInterface
    public void collect() {
        DailyBonusManager.getInstance(activity).sendCollectDailyBonus();
    }

    @JavascriptInterface
    public void iapPurchaseRequest(String skuId, String campaignName) {
        GamedockSDK.getInstance(activity).getIapCallbacks().iapRequestPurchase(skuId, campaignName);
    }

    @JavascriptInterface
    public void buyBundle(int bundleId, String reason, String reasonDetails, String location) {
        GamedockSDK.getInstance(activity).buyBundle(bundleId, reason, reasonDetails, location, null, null, null);
    }

    @JavascriptInterface
    public void openBundle(int bundleId, int amount, String reason, String reasonDetails, String location) {
        GamedockSDK.getInstance(activity).openBundle(bundleId, amount, reason, reasonDetails, location, null, null);
    }

    @JavascriptInterface
    public void openGacha(int gachaId, String reason, String reasonDetails, String location) {
        GamedockSDK.getInstance(activity).openGacha(gachaId, reason, reasonDetails, location, null, false, null);
    }

    @JavascriptInterface
    public void addItem(int itemId, int amount, String reason, String reasonDetails, String location) {
        if (amount > 0) {
            GamedockSDK.getInstance(activity).addItemToInventory(itemId, amount, reason, reasonDetails, location, null, false, null);
        } else {
            GamedockSDK.getInstance(activity).subtractItemFromInventory(itemId, amount, reason, reasonDetails, location, null, false, null);
        }
    }

    @JavascriptInterface
    public void createUniqueItem(int itemId) {
        Item gameItem = GamedockGameDataManager.getInstance(activity).getItem(itemId);

        if (gameItem == null) {
            GamedockSDK.getInstance(activity).getUserDataCallbacks().userDataError(ErrorCodes.ItemNotFound);
            return;
        }

        UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
        uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
        GamedockSDK.getInstance(activity).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0, 0, 0, PlayerDataUpdateReasons.Splashscreen);
    }

    @JavascriptInterface
    public void addCurrency(int currencyId, int amount, String reason, String reasonDetails, String location) {
        if (amount > 0) {
            GamedockSDK.getInstance(activity).addCurrencyToWallet(currencyId, amount, reason, reasonDetails, location, null, false, null);
        } else {
            GamedockSDK.getInstance(activity).subtractCurrencyFromWallet(currencyId, amount, reason, reasonDetails, location, null, false, null);
        }
    }

    @JavascriptInterface
    public void getBundlePromotion(int bundleId) {
        PromotionsManager.getInstance(activity).getBundlePromotionForWeb(bundleId);
    }

    @JavascriptInterface
    public void getPackagePromotion(String packageId) {
        PromotionsManager.getInstance(activity).getPackagePromotionForWeb(packageId);
    }

    @JavascriptInterface
    public void showLoader() {
        if (WebViewActivity.getActivity() != null) {
            WebViewActivity.getActivity().showAdvanceSpinner(activity);
        }
    }

    @JavascriptInterface
    public void hideLoader() {
        if (WebViewActivity.getActivity() != null) {
            WebViewActivity.getActivity().hideSpinner();
        }
    }

    @JavascriptInterface
    public void sendDataToGame(String payload) {
        GamedockSDK.getInstance(activity).getSplashScreenCallbacks().splashScreenData(payload);
    }

    @JavascriptInterface
    public String getIapDetails(String skuIds) {
        return GamedockSDK.getInstance(activity).getIapDetails(skuIds);
    }

    @JavascriptInterface
    public String getStore() {
        return IAPUtil.store;
    }

    @JavascriptInterface
    public String getTieredEvents() {
        return GamedockSDK.getInstance(activity).getAllTieredEvents();
    }

    @JavascriptInterface
    public void claimTierReward(int tieredEventId, int tierId) {
        GamedockSDK.getInstance(activity).claimTierReward(tieredEventId, tierId);
    }

    @JavascriptInterface
    public void sendEvent(String eventName, String customData) {
        Event event = new Event(activity);
        event.setName(eventName);

        JSONObject jsonObject;
        try {
            if (customData != null) {
                jsonObject = new JSONObject(customData);
                for (int i = 0; i < jsonObject.length(); i++) {
                    String key = jsonObject.names().getString(i);

                    if (jsonObject.get(key) instanceof String) {
                        event.addCustomData(key, jsonObject.getString(key));
                    } else if (jsonObject.get(key) instanceof Integer) {
                        event.addCustomData(key, jsonObject.getInt(key));
                    } else if (jsonObject.get(key) instanceof Boolean) {
                        event.addCustomData(key, jsonObject.getBoolean(key));
                    } else if (jsonObject.get(key) instanceof Double) {
                        event.addCustomData(key, jsonObject.getDouble(key));
                    } else if (jsonObject.get(key) instanceof Long) {
                        event.addCustomData(key, jsonObject.getLong(key));
                    } else if (jsonObject.get(key) instanceof JSONObject) {
                        event.addCustomData(key, jsonObject.getJSONObject(key));
                    } else {
                        String invalid = "INVALID PARAMETER TYPE";
                        event.addCustomData(key, invalid);
                    }
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
            String invalid = "INVALID PARAMETER TYPE";
            event.addCustomData("invalid", invalid);
        }

        GamedockSDK.getInstance(activity).trackEvent(event, null);
    }

    @JavascriptInterface
    public void nativeMessage(final String methodName, final String data) {
        if (activity != null) {
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (webView != null) {
                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.KITKAT) {
                            webView.evaluateJavascript("Spil.nativeMessage('" + methodName + "', " + data + ");", null);
                        } else {
                            webView.loadUrl("javascript:Spil.nativeMessage('" + methodName + "' , " + data + ");");
                        }
                    }
                }
            });
        }
    }

    @JavascriptInterface
    public void onEventReceived(String rawData) {
        Log.i("GamedockSDK Event", rawData);

        ResponseEvent re = new ResponseEvent();

        try {
            JSONObject jo = new JSONObject(rawData);

            re.setAction(jo.getString("action"));
            re.setName(jo.getString("name"));
            re.setType(jo.getString("type"));

            if (jo.has("data")) {
                re.responseData = jo.getJSONObject("data");
            }
        } catch (Exception e) {
            String TAG = "JavascriptBridge";
            Log.i(TAG, "error parsing json " + e.getMessage());
        }
        OverlayResponseEvent overlayResponseEvent = (OverlayResponseEvent) re;
        GamedockSDK.getInstance(activity).processResponseEvent(overlayResponseEvent, actionListener);
    }

    @JavascriptInterface
    public void displayMessage(final String message) {
        if (GamedockConfigManager.getInstance(activity).isDebugModeEnabled()) {
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    Toast.makeText(activity, message, Toast.LENGTH_LONG).show();
                }
            });
        }
    }

    public void setEventData(String eventData) {
        this.eventData = eventData;
    }
}