package io.gamedock.sdk.utils.userid;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.provider.Settings;

import androidx.annotation.NonNull;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.utils.gcm.GCMUtils;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Class that does the following:
 * - generates unique user id
 * - checks if the device already has a Gamedock user id
 * - stores user id on the sd card
 * - returns the user id
 */
public class UserIDGenerator {

    public static String firebaseInstanceId = "";

    /**
     * Method that returns the currently stored Gamedock User Id.
     * It either retrieves it from the {@link StorageUtil} or from the local file or it generates a new one.
     *
     * @param context The activity context.
     * @return Returns the Gamedock User Id in a {@link String} format.
     */
    public static String getGamedockUserId(Context context) {
        String userId = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.UserId, null);

        if (userId == null) {
            LoggingUtil.d("Create new uuid");
            userId = UUID.randomUUID().toString();

            if (userId != null) {
                GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.UserId, userId);
            }
        }

        LoggingUtil.d("Current uuid " + userId);
        return userId;
    }

    /**
     * Method that manually generates and sets a Gamedock User Id.
     *
     * @param context The activity context.
     * @param userId  The desired user id that needs to be set.
     */
    public static void manualUUIDGeneration(Context context, String userId) {
        String saveUserId;
        if (userId == null) {
            saveUserId = UUID.randomUUID().toString();
        } else {
            saveUserId = userId;
        }

        GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.UserId, saveUserId);
    }

    /**
     * Method that checks if the WRITE_EXTERNAL_STORAGE has been given by the player.
     *
     * @param context The activity context.
     * @return Returns {@code true} if the permission has been granted.
     */
    private static boolean checkWriteExternalPermission(Context context) {
        String permission = "android.permission.WRITE_EXTERNAL_STORAGE";
        int res;
        try {
            res = context.checkCallingOrSelfPermission(permission);
        } catch (NullPointerException e) {
            res = PackageManager.PERMISSION_DENIED;
        }

        return (res == PackageManager.PERMISSION_GRANTED);
    }

    /**
     * Method that retrieves an unique phone identifier based on the phone model, manufacturer and the phone serial number.
     *
     * @return Returns an unique phone identifier in a {@link String} format.
     */
    @SuppressLint("HardwareIds")
    public static String getUniqueDeviceId(Context context) {
        String deviceId = "";

        if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
            return "";
        }

        try {
            deviceId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
        } catch (NullPointerException e) {
            e.printStackTrace();
        }

        boolean valid = true;

        try {
            String zerosCheck = deviceId.replace("0", "");
            if (zerosCheck.length() < 1) {
                valid = false;
            }

            if (!deviceId.matches("^[a-fA-F0-9]*$")) {
                valid = false;
            }

            if (deviceId.length() < 6) {
                valid = false;
            }

        } catch (Exception e) {
            valid = false;
        }

        if (valid) {
            return deviceId;
        } else {
            String pseudoId = "35" + Build.BOARD.hashCode() % 10 + Build.BRAND.hashCode() % 10 + Build.DEVICE.hashCode() % 10 + Build.DISPLAY.hashCode() % 10 + Build.HOST.hashCode() % 10 + Build.ID.hashCode() % 10 + Build.MANUFACTURER.hashCode() % 10 + Build.MODEL.hashCode() % 10 + Build.PRODUCT.hashCode() % 10 + Build.TAGS.hashCode() % 10 + Build.TYPE.hashCode() % 10 + Build.USER.hashCode() % 10;

            String serial;
            try {
                serial = Build.class.getField("SERIAL").get(null).toString();
            } catch (Exception e) {
                serial = "serial";
            }

            String combined = deviceId + pseudoId + serial;

            try {
                MessageDigest messageDigest = MessageDigest.getInstance("MD5");
                messageDigest.update(combined.getBytes(), 0, combined.length());

                // get md5 bytes
                byte md5Bytes[] = messageDigest.digest();

                // creating a hex string
                String generatedId = "";

                for (byte md5Byte : md5Bytes) {
                    int b = (0xFF & md5Byte);

                    // if it is a single digit, make sure it have 0 in front (proper padding)
                    if (b <= 0xF) {
                        generatedId += "0";
                    }

                    // add number to string
                    generatedId += Integer.toHexString(b);
                }

                // hex string to uppercase
                generatedId = generatedId.toUpperCase();

                int length;
                if (generatedId.length() > 16) {
                    length = 16;
                } else {
                    length = generatedId.length();
                }
                generatedId = "m-" + generatedId.substring(0, length);
                return generatedId;
            } catch (NoSuchAlgorithmException e) {
                return "m-" + combined;
            }
        }
    }

    public static String getFirebaseInstanceId() {
        return firebaseInstanceId;
    }

    public static void getFirebaseInstanceId(final Context context) {
        FirebaseInstanceId.getInstance().getInstanceId().addOnCompleteListener(new OnCompleteListener<InstanceIdResult>() {
            @Override
            public void onComplete(@NonNull Task<InstanceIdResult> task) {
                if (!task.isCanceled() && task.isComplete() && task.isSuccessful()) {
                    if (task.getResult() != null) {
                        firebaseInstanceId = task.getResult().getId();
                        LoggingUtil.d("Received Firebase Instance Id: " + firebaseInstanceId);
                        GCMUtils.storeGCM(context, task.getResult().getToken());
                    }
                }
            }
        });
    }
}
