package io.gamedock.sdk.utils.storage;

import android.content.Context;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.encryption.storage.EncryptedPreferences;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles all the logic associated with local storage encryption.
 */
public class StorageUtil {

    private static final Object lock = new Object();

    private static volatile StorageUtil mInstance = null;

    private final EncryptedPreferences encryptedPreferences;

    private Context context;

    private StorageUtil(Context context) {
        this.context = context;
        encryptedPreferences = new EncryptedPreferences.Builder(context)
                .withEncryptionPassword(context.getPackageName())
                .build();
    }

    public synchronized static StorageUtil getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new StorageUtil(context);
                }
            }
        }

        return mInstance;
    }

    public synchronized int getInt(String key, int defaultValue) {
        return encryptedPreferences.getInt(key, defaultValue);
    }

    public synchronized long getLong(String key, long defaultValue) {
        return encryptedPreferences.getLong(key, defaultValue);
    }

    public synchronized boolean getBoolean(String key, boolean defaultValue) {
        return encryptedPreferences.getBoolean(key, defaultValue);
    }

    public synchronized float getFloat(String key, float defaultValue) {
        return encryptedPreferences.getFloat(key, defaultValue);
    }

    public synchronized String getString(String key, String defaultValue) {
        return encryptedPreferences.getString(key, defaultValue);
    }

    public synchronized void putString(final String key, final String value) {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    if (value != null) {
                        encryptedPreferences.edit().putString(key, value).apply();
                    } else {
                        LoggingUtil.d("Null value detected for key: " + key + ". No value will be saved!");
                    }
                }
            });
        } else {
            if (value != null) {
                encryptedPreferences.edit().putString(key, value).apply();
            } else {
                LoggingUtil.d("Null value detected for key: " + key + ". No value will be saved!");
            }
        }
    }

    public synchronized void putInt(final String key, final int value) {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    encryptedPreferences.edit().putInt(key, value).apply();
                }
            });
        } else {
            encryptedPreferences.edit().putInt(key, value).apply();
        }
    }

    public synchronized void putLong(final String key, final long value) {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    encryptedPreferences.edit().putLong(key, value).apply();
                }
            });
        } else {
            encryptedPreferences.edit().putLong(key, value).apply();
        }
    }

    public synchronized void putFloat(final String key, final float value) {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    encryptedPreferences.edit().putFloat(key, value).apply();
                }
            });
        } else {
            encryptedPreferences.edit().putFloat(key, value).apply();
        }
    }

    public synchronized void putBoolean(final String key, final boolean value) {
        encryptedPreferences.edit().putBoolean(key, value).apply();
    }

    public synchronized void remove(final String key) {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    encryptedPreferences.edit().remove(key).apply();
                }
            });
        } else {
            encryptedPreferences.edit().remove(key).apply();
        }
    }

    public synchronized void clear() {
        if (GamedockSDK.getInstance(context).handler != null) {
            GamedockSDK.getInstance(context).handler.post(new Runnable() {
                @Override
                public void run() {
                    encryptedPreferences.edit().clear().apply();
                }
            });
        } else {
            encryptedPreferences.edit().clear().apply();
        }
    }

    public synchronized boolean contains(String key) {
        return encryptedPreferences.contains(key);
    }

    public synchronized void clearUserSpecificData() {
        remove(Keys.SpilUserData);
        remove(Keys.WalletInit);
        remove(Keys.InventoryInit);
    }

    public void resetStorageUtil() {
        context = null;
        mInstance = null;
    }

    /**
     * Class that hold all the keys used for storing the values used by the SDK
     */
    public static class Keys {
        private static final String StorageVersion = "storageVersion";

        public static final String FeaturesInfo = "featuresInfo";
        public static final String FeaturesVersionIds = "featuresVersionIds";
        public static final String ChartboostAppId = "chartBoostAppId";
        public static final String ChartboostAppSignature = "chartBoostAppSignature";
        public static final String PluginName = "pluginName";
        public static final String PluginVersion = "pluginVersion";
        public static final String SpilApplicationPackageName = "SpilApplicationPackageName";
        public static final String SpilSDKPersistenceSpilEvents = "spilSDKPersistenceSpilEvents";
        public static final String SpilSDKPersistenceSize = "spilSDKPersistenceSize";
        public static final String SpilSDKRecentEventList = "spilSDKRecentEventList";
        public static final String GameConfig = "GameConfig";
        public static final String SpilSubdomain = "spilSubdomain";
        public static final String SpilSubdomainOverride = "spilSubdomainOverride";
        public static final String CheckedForFilePermission = "checkedForFilePermission";
        public static final String GoogleAdvertisingId = "googleAdvertisingId";
        public static final String SpilGameData = "spilGameData";
        public static final String SpilPromotions = "spilPromotions";
        public static final String SpilUserData = "spilUserProfile";
        public static final String SpilMissionConfiguration = "spilMissionConfiguration";
        public static final String WalletInit = "walletInit";
        public static final String InventoryInit = "inventoryInit";
        public static final String NotificationStatus = "notificationStatus";
        public static final String AppVersion = "appVersion";
        public static final String SpilSdkInstall = "spilSdkInstall";
        public static final String TotalTimeOpen = "total_time_open";
        public static final String LastTimeOpen = "last_time_open";
        public static final String SessionId = "sessionId";
        public static final String SessionDuration = "sessionDuration";
        public static final String UserId = "regID";
        public static final String GamePackages = "GamePackages";
        public static final String KeySignInCancellations = "SPIL_KEY_SIGN_IN_CANCELLATION";
        public static final String UnitySDKEnvironment = "unitySdkEnvironment";
        public static final String NotificationDataList = "notificationDataList";
        public static final String PermissionRequestDeny = "permissionRequestDeny";
        public static final String EventId = "eventId";
        public static final String SendId = "sendId";
        public static final String SpilToken = "spilToken";
        public static final String PrivacyPolicyStatus = "privacyPolicyStatus2";
        public static final String PrivacyPolicyAsked = "privacyPolicyPopupAsked2";
        public static final String PrivacyPolicyWasShowing = "privacyPolicyWasShowing";
        public static final String PrivacyPolicyShouldTriggerCallback = "privacyPolicyShouldTriggerCallback";
        public static final String AgeGatePassed = "ageGatePassed";
        public static final String AgeGateAsked = "ageGateAsked";
        public static final String AgeGateAge = "ageGateAge";
        public static final String AgeGateState = "ageGateState";
        public static final String AgeGateWasShowing = "ageGateWasShowing";
        public static final String AgeGateShouldBlock = "ageGateShouldBlock";
        public static final String AgeGateRequiredAge = "ageGateRequiredAge";
        public static final String AgeGateShouldTriggerCallback = "ageGateShouldTriggerCallback";
        public static final String COPPAEnabled = "coppaEnabled";
        public static final String GDPRStatus = "gdprStatus";
        public static final String AssetBundles = "spilAssetBundles";
        public static final String Store = "store";
        public static final String PreviouslyNetworkSent = "previouslyNetworkSent";
        public static final String PreviouslyNetworkReceived = "previouslyNetworkReceived";
        public static final String ExternalPartnersIds = "externalPartnersIds";
        public static final String AppRateTriggerCount = "appRateTriggerCount";
        public static final String AppRateShown = "appRateShown";
        public static final String AppRateNeverShow = "appRateNeverShow";
        public static final String AppRateLaterOptionSelected = "appRateLaterOptionSelected";
        public static final String AppRateLaterCount = "appRateLaterCount";
        public static final String LocalizationDefaultLocale = "localizationDefaultLocale";
        public static final String LocalizationLocale = "localizationLocale";
    }
}