package io.gamedock.sdk.utils.privacy;

import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.DividerItemDecoration;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import java.util.ArrayList;
import java.util.Locale;

import io.gamedock.sdk.R;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.models.config.PrivacyPolicy;
import io.gamedock.sdk.models.config.PrivacyPolicyPartner;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.views.checkbox.SmoothCheckBox;

import static androidx.recyclerview.widget.RecyclerView.HORIZONTAL;

public class PrivacyPolicySettingsFragment extends Fragment {
    private OnPrivacyPolicySettingsListener onPrivacyPolicySettingsListener;

    private ImageView privacyPolicyImage;

    private TextView privacyPolicySettingsTitle;
    private TextView privacyPolicyPersonalisedContentTextView;
    private TextView privacyPolicyPersonalisedAdsTextView;
    private TextView privacyPolicyRememberSettingsTextView;
    private TextView privacyPolicyUnderstandHowTextView;
    private TextView privacyPolicySocialMediaTextView;
    private TextView privacyPolicySettingsDescription;

    private RecyclerView privacyPolicyPartnersList;
    private PrivacyPolicyPartnerListAdapter privacyPolicyPartnerListAdapter;

    private SmoothCheckBox privacyPolicyPersonalizedAdsCheckBox;
    private SmoothCheckBox privacyPolicyPersonalizedContentCheckBox;

    private Button privacyPolicyLearnMoreButton;
    private Button privacyPolicyTermsButton;

    private Button privacyPolicySaveSettings;

    private boolean withPersonalisedAds;
    private boolean withPersonalisedContent;

    public static PrivacyPolicySettingsFragment newInstance(boolean withPersonalisedAds, boolean withPersonalisedContent) {
        PrivacyPolicySettingsFragment fragment = new PrivacyPolicySettingsFragment();

        Bundle args = new Bundle();
        args.putBoolean("withPersonalisedAds", withPersonalisedAds);
        args.putBoolean("withPersonalisedContent", withPersonalisedContent);
        fragment.setArguments(args);

        return fragment;
    }

    public PrivacyPolicySettingsFragment() {
        // Required empty public constructor
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (getArguments() != null) {
            withPersonalisedAds = getArguments().getBoolean("withPersonalisedAds");
            withPersonalisedContent = getArguments().getBoolean("withPersonalisedContent");
        }

    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_privacy_policy_settings, container, false);

        privacyPolicyImage = view.findViewById(R.id.privacyPolicyImage);

        privacyPolicySettingsTitle = view.findViewById(R.id.privacyPolicySettingsTitle);
        privacyPolicyPersonalisedContentTextView = view.findViewById(R.id.privacyPolicyPersonalisedContentTextView);
        privacyPolicyPersonalisedAdsTextView = view.findViewById(R.id.privacyPolicyPersonalisedAdsTextView);
        privacyPolicyRememberSettingsTextView = view.findViewById(R.id.privacyPolicyRememberSettingsTextView);
        privacyPolicyUnderstandHowTextView = view.findViewById(R.id.privacyPolicyUnderstandHowTextView);
        privacyPolicySocialMediaTextView = view.findViewById(R.id.privacyPolicySocialMediaTextView);
        privacyPolicySettingsDescription = view.findViewById(R.id.privacyPolicySettingsDescription);

        privacyPolicyPartnersList = (RecyclerView) view.findViewById(R.id.privacyPolicyPartnersList);

        privacyPolicyPersonalizedAdsCheckBox = view.findViewById(R.id.privacyPolicyPersonalizedAdsCheckBox);
        privacyPolicyPersonalizedContentCheckBox = view.findViewById(R.id.privacyPolicyPersonalizedContentCheckBox);

        privacyPolicyLearnMoreButton = view.findViewById(R.id.privacyPolicyLearnMoreButton);
        privacyPolicyTermsButton = view.findViewById(R.id.privacyPolicyTermsButton);

        privacyPolicySaveSettings = view.findViewById(R.id.privacyPolicySaveSettingsButton);

        try {
            if (getContext() != null) {
                int orientation = getResources().getConfiguration().orientation;
                String imageName = "";
                if (orientation == Configuration.ORIENTATION_PORTRAIT) {
                    imageName = "privacy_policy_portrait_custom";
                } else {
                    imageName = "privacy_policy_landscape_custom";
                }

                int resourceId = getResources().getIdentifier(imageName, "drawable", getContext().getPackageName());
                if (resourceId != 0) {
                    privacyPolicyImage.setBackgroundResource(resourceId);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        Locale loc;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            loc = getResources().getConfiguration().getLocales().get(0);
        } else {
            loc = getResources().getConfiguration().locale;
        }
        String deviceLocale = loc.getLanguage();

        String title = "";
        String personalisedContent = "";
        String personalisedAds = "";
        String rememberSettings = "";
        String understandHow = "";
        String socialMedia = "";
        String description = "";
        String learnMore = "";
        String learnMoreUrl = "";
        String privacyPolicy = "";
        String privacyPolicyUrl = "";
        String save = "";

        ArrayList<PrivacyPolicyPartner> partners = new ArrayList<>();

        boolean privacyPolicyTextSet = false;

        for (PrivacyPolicy policy : GamedockConfigManager.getInstance(getContext()).spil.privacyPolicies) {
            if (policy.locale.equals(deviceLocale)) {
                title = policy.settings.title;
                personalisedContent = policy.settings.personalisedContent;
                personalisedAds = policy.settings.personalisedAds;
                rememberSettings = policy.settings.rememberSettings;
                understandHow = policy.settings.understandHow;
                socialMedia = policy.settings.socialMedia;
                description = policy.settings.description;
                learnMore = policy.settings.learnMore;
                learnMoreUrl = policy.settings.learnMoreUrl;
                privacyPolicy = policy.settings.privacyPolicy;
                privacyPolicyUrl = policy.settings.privacyPolicyUrl;
                save = policy.settings.save;

                partners = policy.settings.partners;

                privacyPolicyTextSet = true;
                break;
            }
        }

        if (!privacyPolicyTextSet) {
            PrivacyPolicy englishPrivacyPolicy = GamedockConfigManager.getInstance(getContext()).spil.getDefaultEnglishPrivacyPolicy();
            if (englishPrivacyPolicy != null) {
                title = englishPrivacyPolicy.settings.title;
                personalisedContent = englishPrivacyPolicy.settings.personalisedContent;
                personalisedAds = englishPrivacyPolicy.settings.personalisedAds;
                rememberSettings = englishPrivacyPolicy.settings.rememberSettings;
                understandHow = englishPrivacyPolicy.settings.understandHow;
                socialMedia = englishPrivacyPolicy.settings.socialMedia;
                description = englishPrivacyPolicy.settings.description;
                learnMore = englishPrivacyPolicy.settings.learnMore;
                learnMoreUrl = englishPrivacyPolicy.settings.learnMoreUrl;
                privacyPolicy = englishPrivacyPolicy.settings.privacyPolicy;
                privacyPolicyUrl = englishPrivacyPolicy.settings.privacyPolicyUrl;
                save = englishPrivacyPolicy.settings.save;

                partners = englishPrivacyPolicy.settings.partners;
            } else {
                title = "";
                personalisedContent = "";
                personalisedAds = "";
                rememberSettings = "";
                understandHow = "";
                socialMedia = "";
                description = "";
                learnMore = "";
                learnMoreUrl = "";
                privacyPolicy = "";
                privacyPolicyUrl = "";
                save = "";
            }
        }

        privacyPolicySettingsTitle.setText(title);
        privacyPolicyPersonalisedContentTextView.setText(personalisedContent);
        privacyPolicyPersonalisedAdsTextView.setText(personalisedAds);
        privacyPolicyRememberSettingsTextView.setText(rememberSettings);
        privacyPolicyUnderstandHowTextView.setText(understandHow);
        privacyPolicySocialMediaTextView.setText(socialMedia);
        privacyPolicySettingsDescription.setText(description);
        privacyPolicyLearnMoreButton.setText(learnMore);
        privacyPolicyTermsButton.setText(privacyPolicy);
        privacyPolicySaveSettings.setText(save);

        privacyPolicyPartnerListAdapter = new PrivacyPolicyPartnerListAdapter(R.layout.list_privacy_policy_partners_item, partners, getContext());
        privacyPolicyPartnersList.setLayoutManager(new LinearLayoutManager(getActivity()));
        privacyPolicyPartnersList.setItemAnimator(new DefaultItemAnimator());

        DividerItemDecoration itemDecor = new DividerItemDecoration(getContext(), HORIZONTAL);
        privacyPolicyPartnersList.addItemDecoration(itemDecor);

        privacyPolicyPartnersList.setAdapter(privacyPolicyPartnerListAdapter);

        privacyPolicyPersonalizedAdsCheckBox.setChecked(withPersonalisedAds, false);
        privacyPolicyPersonalizedContentCheckBox.setChecked(withPersonalisedContent, false);

        privacyPolicyPersonalizedAdsCheckBox.setOnCheckedChangeListener(new SmoothCheckBox.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(SmoothCheckBox checkBox, boolean isChecked) {
                onPrivacyPolicySettingsListener.OnPersonalisedAdsSetting(isChecked);
            }
        });

        privacyPolicyPersonalizedContentCheckBox.setOnCheckedChangeListener(new SmoothCheckBox.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(SmoothCheckBox checkBox, boolean isChecked) {
                onPrivacyPolicySettingsListener.OnPersonalisedContentSetting(isChecked);
            }
        });

        final String finalLearnMoreUrl = learnMoreUrl;
        privacyPolicyLearnMoreButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!finalLearnMoreUrl.isEmpty()) {
                    try {
                        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalLearnMoreUrl));
                        getActivity().startActivity(browserIntent);
                    } catch (NullPointerException | ActivityNotFoundException e) {
                        LoggingUtil.w("Could not open Learn More link!");
                    }
                }
            }
        });

        final String finalPrivacyPolicyUrl = privacyPolicyUrl;
        privacyPolicyTermsButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!finalPrivacyPolicyUrl.isEmpty()) {
                    try {
                        Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalPrivacyPolicyUrl));
                        getActivity().startActivity(browserIntent);
                    } catch (NullPointerException | ActivityNotFoundException e) {
                        LoggingUtil.w("Could not open Privacy Policy link!");
                    }
                }
            }
        });

        privacyPolicySaveSettings.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onPrivacyPolicySettingsListener.OnSettingsSaved();
            }
        });

        return view;
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        if (context instanceof OnPrivacyPolicySettingsListener) {
            onPrivacyPolicySettingsListener = (OnPrivacyPolicySettingsListener) context;
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        onPrivacyPolicySettingsListener = null;
    }

    public interface OnPrivacyPolicySettingsListener {
        void OnPersonalisedAdsSetting(boolean enabled);

        void OnPersonalisedContentSetting(boolean enabled);

        void OnSettingsSaved();
    }
}
