package io.gamedock.sdk.utils.privacy;


import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import java.util.Locale;

import io.gamedock.sdk.R;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.models.config.PrivacyPolicy;
import io.gamedock.sdk.utils.views.checkbox.SmoothCheckBox;

public class PrivacyPolicyMainFragment extends Fragment {

    private OnPrivacyPolicyMainListener onPrivacyPolicyMainListener;

    private ImageView privacyPolicyImage;

    private TextView privacyPolicyDescription;

    private LinearLayout privacyPolicyAcceptButtonLayout;
    private SmoothCheckBox privacyPolicyAcceptCheckBox;
    private Button privacyPolicyAcceptButton;

    private Button privacyPolicySettings;

    private boolean openedSettings;

    public static PrivacyPolicyMainFragment newInstance(boolean openedSettings) {
        PrivacyPolicyMainFragment fragment = new PrivacyPolicyMainFragment();

        Bundle args = new Bundle();
        args.putBoolean("openedSettings", openedSettings);
        fragment.setArguments(args);

        return fragment;
    }

    public PrivacyPolicyMainFragment() {
        // Required empty public constructor
    }

    @SuppressLint("ClickableViewAccessibility")
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_privacy_policy_main, container, false);

        if (getArguments() != null) {
            openedSettings = getArguments().getBoolean("openedSettings");
        }

        privacyPolicyImage = view.findViewById(R.id.privacyPolicyImage);
        privacyPolicyDescription = view.findViewById(R.id.privacyPolicyDescription);
        privacyPolicyAcceptButtonLayout = view.findViewById(R.id.privacyPolicyAcceptButtonLayout);
        privacyPolicyAcceptCheckBox = view.findViewById(R.id.privacyPolicyAcceptCheckBox);
        privacyPolicyAcceptButton = view.findViewById(R.id.privacyPolicyAcceptButton);
        privacyPolicySettings = view.findViewById(R.id.privacyPolicySettings);

        try {
            if (getContext() != null) {
                int orientation = getResources().getConfiguration().orientation;
                String imageName = orientation == Configuration.ORIENTATION_PORTRAIT ? "privacy_policy_portrait_custom" : "privacy_policy_landscape_custom";

                int resourceId = getResources().getIdentifier(imageName, "drawable", getContext().getPackageName());
                if (resourceId != 0) {
                    privacyPolicyImage.setBackgroundResource(resourceId);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        Locale loc;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            loc = getResources().getConfiguration().getLocales().get(0);
        } else {
            loc = getResources().getConfiguration().locale;
        }
        String deviceLocale = loc.getLanguage();

        String description = "";
        String accept = "";
        String acceptAll = "";
        String settings = "";

        boolean privacyPolicyTextSet = false;

        for (PrivacyPolicy privacyPolicy : GamedockConfigManager.getInstance(getContext()).spil.privacyPolicies) {
            if (privacyPolicy.locale.equals(deviceLocale)) {
                description = privacyPolicy.main.description;
                accept = privacyPolicy.main.accept;
                acceptAll = privacyPolicy.main.acceptAll;
                settings = privacyPolicy.main.settings;

                privacyPolicyTextSet = true;
                break;
            }
        }

        if (!privacyPolicyTextSet) {
            PrivacyPolicy englishPrivacyPolicy = GamedockConfigManager.getInstance(getContext()).spil.getDefaultEnglishPrivacyPolicy();
            if (englishPrivacyPolicy != null) {
                description = englishPrivacyPolicy.main.description;
                accept = englishPrivacyPolicy.main.accept;
                acceptAll = englishPrivacyPolicy.main.acceptAll;
                settings = englishPrivacyPolicy.main.settings;
            } else {
                description = "";
                accept = "";
                acceptAll = "";
                settings = "";
            }
        }

        privacyPolicyDescription.setText(description);
        privacyPolicySettings.setText(settings);

        if (openedSettings) {
            privacyPolicyAcceptButton.setText(accept);
        } else {
            privacyPolicyAcceptButton.setText(acceptAll);
        }

        float scale = getResources().getDisplayMetrics().density;
        float dimension = accept.length() * 18;
        if (dimension < 110) {
            dimension = 110;
        }
        privacyPolicyAcceptButtonLayout.getLayoutParams().width = (int) (dimension * scale + 0.5f);

        privacyPolicyAcceptCheckBox.setChecked(true, true);
        privacyPolicyAcceptCheckBox.setClickable(false);
        privacyPolicyAcceptCheckBox.setEnabled(false);

        privacyPolicyAcceptButtonLayout.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getAction() == MotionEvent.ACTION_DOWN) {
                    LinearLayout layout = (LinearLayout) v;
                    layout.setBackground(getResources().getDrawable(R.drawable.accept_button_highlight));
                    return true;
                } else if (event.getAction() == MotionEvent.ACTION_UP) {
                    LinearLayout layout = (LinearLayout) v;
                    layout.setBackground(getResources().getDrawable(R.drawable.accept_button));
                    layout.performClick();
                    return true;
                }
                return false;
            }
        });
        privacyPolicyAcceptButtonLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onPrivacyPolicyMainListener.OnPrivacyPolicyAccepted();
            }
        });

        privacyPolicySettings.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onPrivacyPolicyMainListener.OnShowSettingsScreen();
            }
        });

        return view;
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        if (context instanceof OnPrivacyPolicyMainListener) {
            onPrivacyPolicyMainListener = (OnPrivacyPolicyMainListener) context;
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        onPrivacyPolicyMainListener = null;
    }

    public interface OnPrivacyPolicyMainListener {
        void OnShowSettingsScreen();

        void OnPrivacyPolicyAccepted();
    }
}
