package io.gamedock.sdk.utils.permissions;


import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;

import androidx.core.content.ContextCompat;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.models.config.Permission;
import io.gamedock.sdk.models.config.PermissionDialogSettings;
import io.gamedock.sdk.models.config.PermissionRationale;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.permissions.air.AIRPermissionActivity;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class used by SDK to request permissions at runtime.
 */
public class PermissionUtil {

    public static String rationaleTitle = "Permission Rationale";
    public static String denyRationaleTitle = "Deny Rationale";
    public static String askAgain = "";
    public static String settings = "";

    public static final String[] DangerousPermissions = {Manifest.permission.READ_CALENDAR, Manifest.permission.WRITE_CALENDAR, Manifest.permission.CAMERA, Manifest.permission.READ_CONTACTS,
            Manifest.permission.WRITE_CONTACTS, Manifest.permission.GET_ACCOUNTS, Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION,
            Manifest.permission.RECORD_AUDIO, Manifest.permission.READ_PHONE_STATE, Manifest.permission.CALL_PHONE, Manifest.permission.SEND_SMS, Manifest.permission.RECEIVE_SMS,
            Manifest.permission.READ_SMS, Manifest.permission.READ_EXTERNAL_STORAGE, Manifest.permission.WRITE_EXTERNAL_STORAGE
    };

    /**
     * Method that requests the specified dangerous permissions to the Android System.
     *
     * @param permissionsMap The list of permissions that need to be requested with their specific rationale.
     * @param isAIR          Value that determines if the request for the permission displayed has been made from AIR.
     */
    public static void requestPermissions(final Context context, final HashMap<String, PermissionRationaleTemp> permissionsMap, final boolean isAIR, final boolean showRationale, final boolean showDenyRationale) {
        String[] permissionsValue = new String[permissionsMap.size()];

        int i = 0;
        for (String permission : permissionsMap.keySet()) {
            permissionsValue[i] = permission;
            i++;
        }

        LoggingUtil.d("Requested permissions: " + Arrays.toString(permissionsValue));

        PermissionBuilder.getInstance().requestPermissions(context, new PermissionBuilder.IOnPermissionResult() {
            @Override
            public void onPermissionResult(PermissionBuilder.ResultSet resultSet) {
                String finalRationale = "";
                boolean showAskAgain = true;
                for (Map.Entry<String, PermissionBuilder.Result> entry : resultSet.toMap().entrySet()) {
                    LoggingUtil.d("Permission result for " + entry.getKey() + ": " + resultSet.isPermissionGranted(entry.getKey()));

                    GamedockSDK.getInstance(context).getPermissionCallbacks().permissionResponse(entry.getKey(), resultSet.isPermissionGranted(entry.getKey()), resultSet.isPermissionPermanentlyDenied(entry.getKey()));

                    if(resultSet.isPermissionPermanentlyDenied(entry.getKey()) && getPermissionRequestDeny(context)) {
                        continue;
                    }

                    if (showAskAgain) {
                        showAskAgain = !resultSet.isPermissionPermanentlyDenied(entry.getKey());
                    }

                    try {
                        if (!resultSet.isPermissionGranted(entry.getKey()) && !permissionsMap.get(entry.getKey()).denyRationale.equals("")) {
                            if (resultSet.toMap().size() == 1 && showDenyRationale) {
                                PermissionBuilder.getInstance().showRationaleInDialog(context, denyRationaleTitle, permissionsMap.get(entry.getKey()).denyRationale, true, true, showAskAgain, null);
                            } else {
                                String permission = entry.getKey();
                                permission = permission.replace("android.permission.", "");
                                permission = permission.replace("_", " ");
                                finalRationale = finalRationale + permission + "\n" + permissionsMap.get(entry.getKey()).denyRationale + "\n\n";
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                if (isAIR && resultSet.areAllPermissionsGranted()) {
                    AIRPermissionActivity.closeActivity();
                }

                if (!finalRationale.equals("") && showDenyRationale) {
                    PermissionBuilder.getInstance().showRationaleInDialog(context, denyRationaleTitle, finalRationale, true, true, showAskAgain, null);
                }

                if (!showAskAgain) {
                    setPermissionRequestDeny(context, true);
                }
            }

            @Override
            public void onRationaleRequested(PermissionBuilder.IOnRationaleProvided callback, String... permissions) {
                if (showRationale) {
                    String finalRationale = "";
                    for (String permission : permissions) {
                        String rationale = permissionsMap.get(permission).rationale;
                        if (rationale != null && (permissions.length == 1) && !rationale.equals("")) {
                            PermissionBuilder.getInstance().showRationaleInDialog(context, rationaleTitle, rationale, false, false, false, callback);
                        } else {
                            permission = permission.replace("android.permission.", "");
                            permission = permission.replace("_", " ");
                            finalRationale = finalRationale + permission + "\n" + rationale + "\n\n";
                        }
                    }

                    if (!finalRationale.equals("")) {
                        PermissionBuilder.getInstance().showRationaleInDialog(context, rationaleTitle, finalRationale, false, false, false, callback);
                    }
                }
            }
        }, showRationale, permissionsValue);
    }

    /**
     * Method that requests the specified dangerous permissions to the Android System from AIR.
     *
     * @param permission The permission required.
     * @param rationale  The rational to be provided for the permission.
     * @param context    The context of the Activity.
     */
    public static void requestPermissionForAir(String permission, String rationale, String denyRationale, Context context) {
        GamedockSDK.getInstance(context).isShowingChildActivity = true;

        LoggingUtil.d("AIR: Requesting permission: " + permission + " And rationale: " + rationale);
        Intent intent = new Intent(context, AIRPermissionActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra("request", "single");
        intent.putExtra("permission", permission);
        intent.putExtra("rationale", rationale);
        intent.putExtra("denyRationale", denyRationale);

        context.startActivity(intent);
    }

    /**
     * Method that checks the AndroidManifest to see if automatic permission requesting should be initiated.
     *
     * @param context The activity context.
     * @return Returns {@code true} if the AndroidManifest is missing the disabling of automatic permission requesting meta tag.
     */
    private static boolean shouldAutomaticallyRequestPermissions(Context context) {
        try {
            ApplicationInfo info = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = info.metaData;

            return !bundle.getBoolean("gamedock.permissions.DisableAutoRequest", false);

        } catch (PackageManager.NameNotFoundException e) {
            LoggingUtil.d("Failed to load Spil Disable Auto Permission Request Meta-data! Ignore this message if you want the GamedockSDK to automatically ask for permissions");
        }

        return true;
    }

    /**
     * Method that requests all the dangerous permissions at once.
     * Check the shouldAutomaticallyRequestPermissions method to see if the operation is possible.
     *
     * @param context The activity context.
     * @param isAIR   Value that determines if the request for the permission displayed has been made from AIR.
     */
    public static void requestAllDangerousPermissions(Context context, boolean isAIR, boolean showRationale, boolean showDenyRationale) {
        if (!shouldAutomaticallyRequestPermissions(context)) {
            if (isAIR) {
                AIRPermissionActivity.closeActivity();
            }
            return;
        }

        PackageInfo info;
        try {
            info = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);

            if (info.requestedPermissions == null) {
                return;
            }

            ArrayList<String> requestedPermissions = new ArrayList<>();

            Collections.addAll(requestedPermissions, info.requestedPermissions);

            if (requestedPermissions.contains(Manifest.permission.READ_EXTERNAL_STORAGE) && requestedPermissions.contains(Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
                requestedPermissions.remove(Manifest.permission.READ_EXTERNAL_STORAGE);
            }

            if (requestedPermissions.contains(Manifest.permission.ACCESS_COARSE_LOCATION) && requestedPermissions.contains(Manifest.permission.ACCESS_FINE_LOCATION)) {
                requestedPermissions.remove(Manifest.permission.ACCESS_COARSE_LOCATION);
            }

            HashMap<String, PermissionRationaleTemp> permissionsMap = new HashMap<>();

            boolean dialogSettingsSet = false;

            for (String requiredPermission : requestedPermissions) {
                for (String dangerousPermission : DangerousPermissions) {
                    if (!requiredPermission.equals(dangerousPermission)) {
                        continue;
                    }

                    if(ContextCompat.checkSelfPermission(context, requiredPermission) != android.content.pm.PackageManager.PERMISSION_DENIED) {
                        GamedockSDK.getInstance(context).getPermissionCallbacks().permissionResponse(requiredPermission, true, false);
                        continue;
                    }

                    Permission permissionsDetails = GamedockConfigManager.getInstance(context).spil.getPermission(requiredPermission);
                    PermissionRationaleTemp permissionRationaleTemp = new PermissionRationaleTemp();

                    if (permissionsDetails == null) {
                        permissionRationaleTemp.rationale = "These permissions are required so that the game can reliably store data and provide you with the best experience.";
                        permissionRationaleTemp.denyRationale = "By denying these permission, the game will not be able to save user specific data to your SD card. This means the game might not be able to restore your progress if you reinstall the game.";
                    } else {
                        Locale loc;
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            loc = context.getResources().getConfiguration().getLocales().get(0);
                        } else {
                            loc = context.getResources().getConfiguration().locale;
                        }
                        String locale = loc.getLanguage();

                        boolean detailsSet = false;

                        for (PermissionRationale permissionRationale : permissionsDetails.permissionRationales) {
                            if (permissionRationale.locale.equals(locale)) {
                                permissionRationaleTemp.rationale = permissionRationale.rationale;
                                permissionRationaleTemp.denyRationale = permissionRationale.denyRationale;

                                detailsSet = true;
                            }
                        }

                        for (PermissionDialogSettings dialogSettings : GamedockConfigManager.getInstance(context).spil.permissionDialogSettings) {
                            if (dialogSettings.locale.equals(locale)) {
                                rationaleTitle = dialogSettings.rationaleTitle;
                                denyRationaleTitle = dialogSettings.denyRationaleTitle;
                                askAgain = dialogSettings.askAgain;
                                settings = dialogSettings.settings;

                                dialogSettingsSet = true;
                            }
                        }

                        if (!detailsSet) {
                            PermissionRationale englishRationale = permissionsDetails.getDefaultEnglishRationale();
                            if (englishRationale != null) {
                                permissionRationaleTemp.rationale = englishRationale.rationale;
                                permissionRationaleTemp.denyRationale = englishRationale.denyRationale;
                            } else {
                                permissionRationaleTemp.rationale = "This permission is required so that the game can reliably store data and provide you with the best experience possible.";
                                permissionRationaleTemp.denyRationale = "By denying this permission, the game is not able to store user specific data on your device and your progress cannot be restored.";
                            }
                        }

                        if(!dialogSettingsSet) {
                            PermissionDialogSettings englishDialogSettings = GamedockConfigManager.getInstance(context).spil.getDefaultPermissionDialogSettings();
                            if (englishDialogSettings != null) {
                                rationaleTitle = englishDialogSettings.rationaleTitle;
                                denyRationaleTitle = englishDialogSettings.denyRationaleTitle;
                                askAgain = englishDialogSettings.askAgain;
                                settings = englishDialogSettings.settings;
                            } else {
                                rationaleTitle = "Permission Rationale";
                                denyRationaleTitle = "Deny Rationale";
                                askAgain = "Ask Again";
                                settings = "Settings";
                            }
                        }
                    }

                    permissionsMap.put(requiredPermission, permissionRationaleTemp);
                }
            }

            requestPermissions(context, permissionsMap, isAIR, showRationale, showDenyRationale);

        } catch (NullPointerException | PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that requests all the dangerous permissions at once.
     * Check the shouldAutomaticallyRequestPermissions method to see if the operation is possible.
     *
     * @param context The activity context.
     */
    public static void requestAllDangerousPermissionsForAir(Context context) {
        GamedockSDK.getInstance(context).isShowingChildActivity = true;

        LoggingUtil.d("AIR: Requesting all permissions");
        Intent intent = new Intent(context.getApplicationContext(), AIRPermissionActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra("request", "all");

        context.startActivity(intent);
    }

    public static void setPermissionRequestDeny(Context context, boolean value) {
        GamedockSDK.getInstance(context).getStorageUtil().putBoolean(StorageUtil.Keys.PermissionRequestDeny, value);
    }

    public static boolean getPermissionRequestDeny(Context context) {
        return GamedockSDK.getInstance(context).getStorageUtil().getBoolean(StorageUtil.Keys.PermissionRequestDeny, false);
    }

    public static class PermissionRationaleTemp {
        public String rationale = "";
        public String denyRationale = "";
    }
}
