package io.gamedock.sdk.utils.images;

import com.google.gson.JsonObject;
import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.models.image.ImageContext;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Image Loading Callbacks.
 * Notifies the application about the different states of the Image Loading Process.
 * Informs the application if an error occurred while processing loading an Image.
 */
public class ImageLoadCallbacks {

    private OnImageLoadListener imageLoadListener;

    public ImageLoadCallbacks() {
        imageLoadListener = null;
    }

    public ImageLoadCallbacks(OnImageLoadListener imageLoadListener) {
        this.imageLoadListener = imageLoadListener;
    }

    /**
     * Method called when an image has been downloaded successfully to the local storage
     *
     * @param localPath    The path where the downloaded image can be found.
     * @param imageContext The {@link ImageContext} object containing information about the image.
     */
    public void imageLoadSuccess(String localPath, ImageContext imageContext) {
        if (imageLoadListener != null) {
            imageLoadListener.ImageLoadSuccess(localPath, imageContext);
        } else {
            try {

                JsonObject jsonObject = new JsonObject();
                jsonObject.addProperty("localPath", localPath);

                JsonObject imageContextJSON = new JsonObject();
                imageContextJSON.addProperty("id", imageContext.id);
                imageContextJSON.addProperty("imageType", imageContext.imageType);
                imageContextJSON.addProperty("imageUrl", imageContext.imageUrl);

                jsonObject.add("imageContext", imageContextJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "ImageLoadSuccess", jsonObject.toString());
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the ImageLoadListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while downloading the requested image.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void imageLoadFailed(ImageContext imageContext, ErrorCodes errorCode) {
        if (imageLoadListener != null) {
            imageLoadListener.ImageLoadFailed(imageContext, errorCode);
        } else {
            try {

                JSONObject jsonObject = new JSONObject();

                JSONObject imageContextJSON = new JSONObject();
                imageContextJSON.put("id", imageContext.id);
                imageContextJSON.put("imageType", imageContext.imageType);
                imageContextJSON.put("imageUrl", imageContext.imageUrl);

                jsonObject.put("imageContext", imageContextJSON);

                JSONObject errorCodeJSON = new JSONObject();
                errorCodeJSON.put("id", errorCode.getId());
                errorCodeJSON.put("name", errorCode.getName());
                errorCodeJSON.put("message", errorCode.getMessage());

                jsonObject.put("errorCode", errorCodeJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "ImageLoadFailed", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the ImageLoadListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when all the images have been downloaded via the pre-loading process.
     */
    public void imagePreLoadingCompleted() {
        if (imageLoadListener != null) {
            imageLoadListener.ImagePreLoadingCompleted();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "ImagePreloadingCompleted", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the ImageLoadListener in order to receive information or something went wrong with Unity");
            }
        }
    }
}
