package io.gamedock.sdk.utils.error;

import com.google.gson.JsonObject;

/**
 * Class that defines all the known error codes used through multiple features in the SDK.
 */
public enum ErrorCodes {
    LoadFailed(1, "LoadFailed", "Data container is empty!"),
    ItemNotFound(2, "ItemNotFound", "Item does not exist!"),
    CurrencyNotFound(3, "CurrencyNotFound", "Currency does not exist!"),
    BundleNotFound(4, "BundleNotFound", "Bundle does not exist!"),
    WalletNotFound(5, "WalletNotFound", "No wallet data stored!"),
    InventoryNotFound(6, "InventoryNotFound", "No inventory data stored!"),
    NotEnoughCurrency(7, "NotEnoughCurrency", "Not enough balance for currency!"),
    ItemAmountToLow(8, "ItemAmountToLow", "Could not remove item as amount is too low!"),
    CurrencyOperation(9, "CurrencyOperation", "Error updating wallet!"),
    ItemOperation(10, "ItemOperation", "Error updating item to player inventory!"),
    BundleOperation(11, "BundleOperation", "Error adding bundle to player inventory!"),
    PublicGameStateOperation(12, "UserIdMissing", "Error adding public game state data! A custom user id must be set in order to save public game state data"),
    GameStateOtherUsersServerError(13, "OtherUsersGameStateError", "Error when loading provided user id's game states from the server"),
    DailyBonusServerError(14, "DailyBonusServerError", "Error processing the reward from daily bonus"),
    DailyBonusLoadError(15, "DailyBonusLoadError", "Error loading the daily bonus page"),
    SplashScreenLoadError(16, "SplashScreenLoadError", "Error loading the splash screen"),
    ImageLoadFailedError(17, "ImageLoadFailedError", "Error loading image"),
    RewardServerError(18, "RewardServerError", ""),
    ConfigServerError(19, "ConfigServerError", "Error retrieving config"),
    GameDataServerError(20, "GameDataServerError", "Error retrieving game data"),
    GameStateServerError(21, "GameStateServerError", "Error communicating with the server for game state"),
    IAPServerError(22, "IAPServerError", "Error communication with the server for IAP"),
    PlayerDataServerError(23, "PlayerDataServerError", "Error retrieving player data from the server"),
    ServerTimeRequestError(24, "ServerTimeRequestError", "Error retrieving server time"),
    GachaNotFound(29, "GachaNotFound", "Gacha does not exist!"),
    GachaOperation(30, "GachaOperation", "Error opening gacha!"),
    NotEnoughGachaBoxes(31, "NotEnoughGachaBoxes", "Not enough gacha boxes in the inventory!"),
    InvalidSpilTokenError(32, "InvalidSpilTokenError", "Spil Token is invalid! Please login again!"),
    RequiresLoginError(33, "RequriesLoginError", "Event requires user login!"),
    InvalidSocialTokenError(34, "InvalidSocialTokenError", "The provided social token could not be verified with the backend"),
    UserAlreadyLinkedError(35, "UserAlreadyLinkedError", "User already has a social provider account"),
    SocialIdAlreadyLinkedError(36, "SocialIdAlreadyLinkedError", "The social id is already linked to another user!"),
    GamedockLoginServerError(37, "GamedockLoginServerError", "Error communicating with the server!"),
    UserDataServerError(38, "UserDataServerError", "Error retrieving user data from server (gameState and playerData)!"),
    ConfigResetError(39, "ConfigResetError", "Error while resetting game config. This may be caused by loss of internet connection."),
    GameDataResetError(40, "GameDataResetError", "Error while resetting game data. This may be caused by loss of internet connection."),
    UserDataResetError(41, "UserDataResetError", "Error while resetting user data. This may be caused by loss of internet connection."),
    TieredEventShowProgressError(42, "TieredEventShowProgressError", "Unable to show tiered event progress."),
    TieredEventUpdateProgressError(43, "TieredEventUpdateProgressError", "Unable to update the tiered event progress."),
    TieredEventClaimTierError(44, "TieredEventClaimTierError", "Unable to claim the tier reward."),
    UniqueItemAdd(45, "UniqueItemAdd", "Unable to add unique item to inventory because an item with the same unique id is present. Use Update or Remove methods first before trying again."),
    MissionConfigurationServerError(46, "MissionConfigurationServerError", "Error retrieving mission configuration."),
    ContainerOperation(47, "ContainerOperation", "Error updating container progress information!"),
    MissionOperation(48, "MissionOperation", "Error updating mission progress information."),
    TransactionOperation(49, "TransactionOperation", "Error performing transaction."),
    LocalizationError(50, "LocalizationError", "Error retrieving localization."),
    ConnectionError(51, "ConnectionError", "There was an error retrieving the data"),
    OperationTooFrequent(100, "OperationTooFrequent", "Operation triggered too frequently. Please reduce the amount of calls done for this operation: ");

    private int id;
    private String name;
    private String message;
    private ErrorContext errorContext = new ErrorContext();

    ErrorCodes(int id, String name, String message) {
        this.id = id;
        this.name = name;
        this.message = message;
    }

    public int getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getMessage() {
        return message;
    }

    public ErrorCodes setMessage(String message) {
        if (message != null) {
            this.message = message;
        }

        return this;
    }

    public ErrorContext getErrorContext() {
        return errorContext;
    }

    public ErrorCodes setErrorContext(ErrorContext errorContext) {
        if (errorContext != null) {
            this.errorContext = errorContext;
        }

        return this;
    }

    public ErrorCodes setOperationError(String operation) {
        if (message != null) {
            this.message = "Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + operation;
        }

        return this;
    }

    public static class ErrorContext {
        private String methodCalled;
        private String reason;
        private JsonObject data;

        public String getMethodCalled() {
            return methodCalled;
        }

        public void setMethodCalled(String methodCalled) {
            this.methodCalled = methodCalled;
        }

        public String getReason() {
            return reason;
        }

        public void setReason(String reason) {
            this.reason = reason;
        }

        public JsonObject getData() {
            return data;
        }

        public void setData(JsonObject data) {
            this.data = data;
        }
    }
}
