package io.gamedock.sdk.utils.IAP;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the IAP Callbacks.
 * Notifies the application when the IAP validation has been received and processed by the SDK.
 */
public class IAPCallbacks {

    private OnIAPListener iapDataListener;

    public IAPCallbacks() {
        iapDataListener = null;
    }

    public IAPCallbacks(OnIAPListener iapDataListener) {
        this.iapDataListener = iapDataListener;
    }


    /**
     * Method that is called when the IAP has been validated with the server.
     */
    void iapValid(String skuId) {
        if (iapDataListener != null) {
            iapDataListener.IAPValid(skuId);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "IAPValid", skuId);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when the IAP has failed validation with the server.
     */
    void iapInvalid(String skuId, String message) {
        if (iapDataListener != null) {
            iapDataListener.IAPInvalid(skuId, message);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("skuId", skuId);
                jsonObject.put("message", message);

                UnityPlayer.UnitySendMessage("GamedockSDK", "IAPInvalid", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    public void iapServerError(ErrorCodes error) {
        if (iapDataListener != null) {
            iapDataListener.IAPServerError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "IAPServerError", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    public void iapRequestPurchase(String skuId, String campaignName) {
        if (iapDataListener != null) {
            iapDataListener.IAPRequestPurchase(skuId, campaignName);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("skuId", skuId);
                jsonObject.put("campaignName", campaignName);

                UnityPlayer.UnitySendMessage("GamedockSDK", "IAPRequestPurchase", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }
}
