package io.gamedock.sdk.userdata.playerdata.functions;


import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.UserDataEvent;
import io.gamedock.sdk.models.gamedata.bundles.Bundle;
import io.gamedock.sdk.models.gamedata.perk.PerkItem;
import io.gamedock.sdk.models.gamedata.promotion.Promotion;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;

/**
 * Internal class used by the {@link PlayerDataManager} to perform all sending functions.
 */
public class PlayerDataSending {

    /**
     * Method that handles all the building and sending of the "updatePlayerData" event based on multiple cases.
     * It covers the updating of the {@link Wallet}, {@link Inventory} and the buying of {@link Bundle}.
     *
     * @param context       The activity context.
     * @param gson          The Gson object used for serializing and deserializing objects.
     * @param userData      The {@link UserData} containing all the updated information.
     * @param bundle        The {@link Bundle} object that has been purchased by the player. Only applicable if a buy operation was performed.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public synchronized static void sendUpdatePlayerDataEvent(Context context, Gson gson, UserData userData, Bundle bundle, String reason, String reasonDetails, String location, String transactionId, Promotion promotion, ArrayList<PerkItem> perkItems) {
        UserDataEvent event = new UserDataEvent(context);
        event.setUpdatePlayerData();

        try {
            //Wallet
            JSONObject walletObject = new JSONObject();
            ArrayList<PlayerCurrency> currencyList = new ArrayList<>();

            Wallet jsonWallet = userData.getWallet().BuildForJson();

            for (int i = 0; i < jsonWallet.getCurrencies().size(); i++) {
                if (jsonWallet.getCurrencies().get(i).getDelta() != 0) {
                    currencyList.add(jsonWallet.getCurrencies().get(i));
                }
            }

            JSONArray jsonArrayWal = new JSONArray(gson.toJson(currencyList));

            for (int i = 0; i < jsonArrayWal.length(); i++) {
                JSONObject currencyWalletJSON = jsonArrayWal.getJSONObject(i);
                currencyWalletJSON.remove(PlayerDataManager.InitialValue);
                currencyWalletJSON.remove(PlayerDataManager.DisplayName);
                currencyWalletJSON.remove(PlayerDataManager.DisplayDescription);
                currencyWalletJSON.remove(PlayerDataManager.ImageUrl);

                if (currencyWalletJSON.getInt(PlayerDataManager.Limit) == 0) {
                    currencyWalletJSON.remove(PlayerDataManager.Limit);
                }

                if (currencyWalletJSON.getInt(PlayerDataManager.Overflow) == 0) {
                    currencyWalletJSON.remove(PlayerDataManager.Overflow);
                }
            }
            walletObject.put(PlayerDataManager.Currencies, jsonArrayWal);

            walletObject.put(UserDataManager.Offset, userData.getWallet().getOffset());

            event.addCustomData(UserDataManager.Wallet, walletObject);

            if (!currencyList.isEmpty()) {
                //Implementation due to concurrency issues
                ArrayList<PlayerCurrency> tempCurrency = new ArrayList<>(userData.getWallet().getCurrenciesMap().values());

                for (int i = 0; i < tempCurrency.size(); i++) {
                    userData.getWallet().getCurrenciesMap().get(tempCurrency.get(i).getId()).setDelta(0);
                    userData.getWallet().getCurrenciesMap().get(tempCurrency.get(i).getId()).setOverflow(0);
                }
            }

            //Inventory
            JSONObject inventoryObject = new JSONObject();
            ArrayList<PlayerItem> itemsList = new ArrayList<>();
            ArrayList<UniquePlayerItem> uniqueItemsList = new ArrayList<>();

            Inventory inventory = userData.getInventory().BuildForJson();

            for (int i = 0; i < inventory.getItems().size(); i++) {
                if (inventory.getItems().get(i).getDelta() != 0) {
                    itemsList.add(inventory.getItems().get(i));
                }
            }

            for (int i = 0; i < inventory.getUniqueItems().size(); i++) {
                if (!inventory.getUniqueItems().get(i).getStatus().equals("NONE")) {
                    uniqueItemsList.add(inventory.getUniqueItems().get(i));
                }
            }

            JSONArray jsonArrayItemsInv = new JSONArray(gson.toJson(itemsList));

            for (int i = 0; i < jsonArrayItemsInv.length(); i++) {
                JSONObject itemsInvJSONObject = jsonArrayItemsInv.getJSONObject(i);
                itemsInvJSONObject.remove(PlayerDataManager.Content);
                itemsInvJSONObject.remove(PlayerDataManager.ImageUrl);
                itemsInvJSONObject.remove(PlayerDataManager.InitialValue);
                itemsInvJSONObject.remove(PlayerDataManager.DisplayName);
                itemsInvJSONObject.remove(PlayerDataManager.DisplayDescription);
                itemsInvJSONObject.remove(PlayerDataManager.Value);
                itemsInvJSONObject.remove(PlayerDataManager.Properties);

                if (itemsInvJSONObject.getInt(PlayerDataManager.Limit) == 0) {
                    itemsInvJSONObject.remove(PlayerDataManager.Limit);
                }

                if (itemsInvJSONObject.getInt(PlayerDataManager.Overflow) == 0) {
                    itemsInvJSONObject.remove(PlayerDataManager.Overflow);
                }

                if (!itemsInvJSONObject.getBoolean(PlayerDataManager.Gacha)) {
                    itemsInvJSONObject.remove(PlayerDataManager.AllowDuplicates);
                    itemsInvJSONObject.remove(PlayerDataManager.ShouldReroll);
                    itemsInvJSONObject.remove(PlayerDataManager.DuplicateReward);
                }

                if (itemsInvJSONObject.has(PlayerDataManager.ReportingName) && itemsInvJSONObject.getString(PlayerDataManager.ReportingName) == null) {
                    itemsInvJSONObject.remove(PlayerDataManager.ReportingName);
                }
            }
            inventoryObject.put(PlayerDataManager.Items, jsonArrayItemsInv);

            if (!itemsList.isEmpty()) {
                //Implementation due to concurrency issues
                ArrayList<PlayerItem> tempItems = new ArrayList<>(userData.getInventory().getItemsMap().values());

                for (int i = 0; i < tempItems.size(); i++) {
                    if (userData.getInventory().getItemsMap().containsKey(tempItems.get(i).getId())) {
                        userData.getInventory().getItemsMap().get(tempItems.get(i).getId()).setDelta(0);
                        userData.getInventory().getItemsMap().get(tempItems.get(i).getId()).setOverflow(0);
                    }
                }
            }

            JSONArray jsonArrayUniqueItemInv = new JSONArray(gson.toJson(uniqueItemsList));
            for (int i = 0; i < jsonArrayUniqueItemInv.length(); i++) {
                JSONObject uniqueItemInvJSONObject = jsonArrayUniqueItemInv.getJSONObject(i);
                uniqueItemInvJSONObject.remove(PlayerDataManager.Content);
                uniqueItemInvJSONObject.remove(PlayerDataManager.ImageUrl);
                uniqueItemInvJSONObject.remove(PlayerDataManager.InitialValue);
                uniqueItemInvJSONObject.remove(PlayerDataManager.DisplayName);
                uniqueItemInvJSONObject.remove(PlayerDataManager.DisplayDescription);
                uniqueItemInvJSONObject.remove(PlayerDataManager.Value);
                uniqueItemInvJSONObject.remove(PlayerDataManager.Properties);

                if (uniqueItemInvJSONObject.getInt(PlayerDataManager.Limit) == 0) {
                    uniqueItemInvJSONObject.remove(PlayerDataManager.Limit);
                }

                if (!uniqueItemInvJSONObject.getBoolean(PlayerDataManager.Gacha)) {
                    uniqueItemInvJSONObject.remove(PlayerDataManager.AllowDuplicates);
                    uniqueItemInvJSONObject.remove(PlayerDataManager.ShouldReroll);
                    uniqueItemInvJSONObject.remove(PlayerDataManager.DuplicateReward);
                }

                if (uniqueItemInvJSONObject.has(PlayerDataManager.ReportingName) && uniqueItemInvJSONObject.getString(PlayerDataManager.ReportingName) == null) {
                    uniqueItemInvJSONObject.remove(PlayerDataManager.ReportingName);
                }
            }
            inventoryObject.put(PlayerDataManager.UniqueItems, jsonArrayUniqueItemInv);

            if (!uniqueItemsList.isEmpty()) {
                //Implementation due to concurrency issues
                ArrayList<UniquePlayerItem> tempUniqueItems = new ArrayList<>(userData.getInventory().getUniqueItemsMap().values());

                for (int i = 0; i < tempUniqueItems.size(); i++) {
                    if (userData.getInventory().getUniqueItemsMap().containsKey(tempUniqueItems.get(i).getUniqueId())) {
                        if (userData.getInventory().getUniqueItemsMap().get(tempUniqueItems.get(i).getUniqueId()).getStatus().equals("REMOVE")) {
                            userData.getInventory().getUniqueItemsMap().remove(tempUniqueItems.get(i).getUniqueId());
                        } else {
                            userData.getInventory().getUniqueItemsMap().get(tempUniqueItems.get(i).getUniqueId()).setDelta(0);
                            userData.getInventory().getUniqueItemsMap().get(tempUniqueItems.get(i).getUniqueId()).setStatus("NONE");
                        }
                    }
                }
            }

            inventoryObject.put(UserDataManager.Offset, userData.getInventory().getOffset());
            event.addCustomData(UserDataManager.Inventory, inventoryObject);

            if (bundle != null) {
                JSONObject bundleJSON = new JSONObject(gson.toJson(bundle));
                bundleJSON.remove(PlayerDataManager.DisplayName);
                bundleJSON.remove(PlayerDataManager.DisplayDescription);
                bundleJSON.remove(PlayerDataManager.ImageUrl);
                bundleJSON.remove(PlayerDataManager.Properties);
                event.addCustomData(PlayerDataManager.Bundle, bundleJSON);
            }

            event.addCustomData(PlayerDataManager.Reason, reason);

            if (!GamedockSDK.getInstance(context).isCoppaEnabled()) {
                event.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));
            } else {
                event.addCustomData(UserDataManager.DeviceVersions, new JsonObject());
            }

            if (reasonDetails != null && !reasonDetails.equals("null")) {
                event.addCustomData(PlayerDataManager.ReasonDetails, reasonDetails);
            }

            if (location != null && !location.equals("null")) {
                event.addCustomData(PlayerDataManager.Location, location);
            }

            if (transactionId != null && !transactionId.equals("null")) {
                event.addCustomData(PlayerDataManager.TransactionId, transactionId);
            }

            if (promotion != null) {
                if (promotion.isValid()) {
                    JSONObject promotionJSON = new JSONObject(gson.toJson(promotion));
                    promotionJSON.remove(PlayerDataManager.Label);
                    promotionJSON.remove(PlayerDataManager.GameAssets);
                    event.addCustomData(PlayerDataManager.Promotion, promotionJSON);
                }
            }

            if (perkItems != null) {
                event.addCustomData(PlayerDataManager.Perk, new JSONArray(gson.toJson(perkItems)));
            }

            UserDataManager.getInstance(context).updateUserData(userData);
        } catch (Exception e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method used to make a request to the backend in order to retrieve the updates that occurred on the backend.
     *
     * @param context  The activity context.
     * @param userData The {@link UserData} containing all the updated information.
     */
    public synchronized static void sendUpdatePlayerDataEvent(Context context, UserData userData) {
        UserDataEvent event = new UserDataEvent(context);
        event.setUpdatePlayerData();

        try {
            JSONObject walletObject = new JSONObject();
            walletObject.put(UserDataManager.Offset, userData.getWallet().getOffset());

            event.addCustomData(UserDataManager.Wallet, walletObject);

            JSONObject inventoryObject = new JSONObject();
            inventoryObject.put(UserDataManager.Offset, userData.getInventory().getOffset());

            event.addCustomData(UserDataManager.Inventory, inventoryObject);

            event.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));
        } catch (Exception e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method used to send an update regarding the resetting of the {@link UserData}.
     *
     * @param context  The activity context.
     * @param gson     The Gson object used for serializing and deserializing objects.
     * @param userData The {@link UserData} containing all the updated information.
     * @param reason   The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}. For this specific case it would always be Reset.
     */
    public static void sendUpdatePlayerDataEvent(Context context, Gson gson, UserData userData, String reason) {
        UserDataEvent event = new UserDataEvent(context);
        event.setUpdatePlayerData();

        try {
            JSONObject walletObject = userData.getWallet().BuildForJson().toJson(gson);

            event.addCustomData(UserDataManager.Wallet, walletObject);

            JSONObject inventoryObject = userData.getInventory().BuildForJson().toJson(gson);

            event.addCustomData(UserDataManager.Inventory, inventoryObject);

            event.addCustomData(PlayerDataManager.Reason, reason);

            event.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));
        } catch (Exception e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }
}
