package io.gamedock.sdk.userdata.playerdata;

import android.content.Context;

import com.google.gson.Gson;

import java.util.ArrayList;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.bundles.Bundle;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.gamedata.perk.PerkItem;
import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataCallbacks;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.missiondata.MissionDataManager;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataInitialisation;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataOperations;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataProcessing;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataSending;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class that handles all the logic regarding player data information of the SDK.
 */
public class PlayerDataManager {
    public static final String Add = "add";
    public static final String Subtract = "subtract";
    public static final String Currencies = "currencies";
    public static final String CurrenciesMap = "currenciesMap";
    public static final String Items = "items";
    public static final String ItemsMap = "itemsMap";
    public static final String UniqueItems = "uniqueItems";
    public static final String UniqueItemsMap = "uniqueItemsMap";
    public static final String Bundle = "bundle";
    public static final String Reason = "reason";
    public static final String ReasonDetails = "reasonDetails";
    public static final String Location = "location";
    public static final String TransactionId = "transactionId";
    public static final String Promotion = "promotion";
    public static final String Perk = "perks";

    public static final String Id = "id";
    public static final String UniqueId = "uniqueId";
    public static final String Type = "type";
    public static final String Name = "name";
    public static final String Delta = "delta";
    public static final String CurrentBalance = "currentBalance";
    public static final String Amount = "amount";
    public static final String InitialValue = "initialValue";
    public static final String Content = "content";
    public static final String ImageUrl = "imageUrl";
    public static final String DisplayName = "displayName";
    public static final String DisplayDescription = "displayDescription";
    public static final String Value = "value";
    public static final String Gacha = "isGacha";
    public static final String AllowDuplicates = "allowDuplicates";
    public static final String ShouldReroll = "shouldReroll";
    public static final String DuplicateReward = "duplicateReward";
    public static final String Properties = "properties";
    public static final String Limit = "limit";
    public static final String Overflow = "overflow";
    public static final String ReportingName = "reportingName";
    public static final String Label = "label";
    public static final String GameAssets = "gameAssets";
    public static final String Status = "status";
    public static final String UniqueProperties = "uniqueProperties";

    public static final String Client = "CLIENT";
    public static final String Server = "SERVER";

    public static final String Currency = "CURRENCY";
    public static final String Item = "ITEM";
    public static final String BundleCheck = "BUNDLE";
    public static final String GachaCheck = "GACHA";
    public static final String PackageCheck = "PACKAGE";
    public static final String None = "NONE";
    public static final String UniqueItem = "UNIQUEITEM";


    private Context context;

    public final Gson gson;
    public final StorageUtil storageUtil;
    public final FileAssetsUtil fileAssetsUtil;

    public boolean loadFailedFired;

    public boolean initialValue = false;

    public PlayerDataManager(Context context, Gson gson, StorageUtil storageUtil, FileAssetsUtil fileAssetsUtil) {
        this.context = context;
        this.gson = gson;
        this.storageUtil = storageUtil;
        this.fileAssetsUtil = fileAssetsUtil;
    }

    /**
     * Method used to initialise the wallet and inventory initial value if there is no internet connection.
     *
     * @param userData The {@link UserData} object that contains the information.
     */
    public void initialisePlayerDataIfNoInternet(UserData userData) {
        initialValue = false;
        if (!NetworkUtil.isInternetAvailable(context)) {
            userData = setWalletInitialValues(userData);
            userData = setInventoryInitialValues(userData);

            if (userData != null && initialValue) {
                UpdatedUserData updatedUserData = new UpdatedUserData();

                ArrayList<PlayerCurrency> playerCurrencies = new ArrayList<>();
                ArrayList<PlayerItem> playerItems = new ArrayList<>();

                for (PlayerCurrency playerCurrency : userData.getWallet().getCurrenciesMap().values()) {
                    if (playerCurrency.getDelta() != 0) {
                        playerCurrencies.add(playerCurrency);
                    }
                }

                for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
                    if (playerItem.getDelta() != 0) {
                        playerItems.add(playerItem);
                    }
                }

                updatedUserData.currencies = playerCurrencies;
                updatedUserData.items = playerItems;

                UserDataManager.getInstance(context).updateUserData(userData);

                PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, PlayerDataUpdateReasons.InitialValue, null, "SDK", null, null, null);
                initialValue = false;

                GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.InitialValue, updatedUserData, getWallet(), getInventory());
            }

            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataAvailable(getWallet(), getInventory(), GamedockSDK.getInstance(context).getGson().toJson(UserDataManager.getInstance(context).getMissionDataManager().getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(UserDataManager.getInstance(context).getMissionDataManager().getUserAllMissionProgress(MissionDataManager.Status.NULL)));
        }
    }

    /**
     * Method used to initialise the Player Data with information received from the Gamedock backend.
     *
     * @param receivedWallet    The {@link Wallet} object received from the backend containing the initialisation information.
     * @param receivedInventory The {@link Inventory} object received from the backend containing the initialisation information.
     */
    public void processPlayerDataInit(Wallet receivedWallet, Inventory receivedInventory, boolean externalChange) {
        UpdatedUserData updatedUserData = new UpdatedUserData();

        //Gets the current UserData which contains the Wallet and Inventory information
        UserData userData = UserDataManager.getInstance(context).getUserData();

        //Checks if the user profile is empty and if so throws an error to the developer
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        //Update inventory information from the server
        if (receivedWallet != null) {
            userData = PlayerDataProcessing.walletInit(context, userData, receivedWallet, externalChange);

            updatedUserData.currencies = new ArrayList<>(receivedWallet.getCurrenciesMap().values());
        }

        //Update inventory information from the server
        if (receivedInventory != null) {
            userData = PlayerDataProcessing.inventoryInit(context, userData, receivedInventory, externalChange);

            //Save updated data
            updatedUserData.items = new ArrayList<>(receivedInventory.getItemsMap().values());
            updatedUserData.uniqueItems = new ArrayList<>(receivedInventory.getUniqueItemsMap().values());
        }

        //Send callback to developer if data has been updated
        if (userData != null && userData.updated) {
            UserDataManager.getInstance(context).updateUserData(userData);
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.ServerUpdate, updatedUserData, getWallet(), getInventory());

            if (!externalChange) {
                PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, PlayerDataUpdateReasons.SlotEvent, null, "sdk", null, null, null);
            }
        }

        if (initialValue) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, PlayerDataUpdateReasons.InitialValue, null, "sdk", null, null, null);
            initialValue = false;
        }
    }

    /**
     * Method used to process the update information received from the server for Wallet and Inventory.
     *
     * @param receivedWallet    The {@link Wallet} object received from the backend containing the initialisation information.
     * @param receivedInventory The {@link Inventory} object received from the backend containing the initialisation information.
     */
    public void processPlayerDataUpdate(Wallet receivedWallet, Inventory receivedInventory) {
        UpdatedUserData updatedUserData = new UpdatedUserData();

        //Gets the current UserData which contains the Wallet and Inventory information
        UserData userData = UserDataManager.getInstance(context).getUserData();

        //Checks if the user profile is empty and if so throws an error to the developer
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        //Update wallet information from the server
        if (receivedWallet != null) {

            userData = PlayerDataProcessing.walletUpdate(userData, receivedWallet);

            updatedUserData.currencies = new ArrayList<>(receivedWallet.getCurrenciesMap().values());
        }

        //Update inventory information from the server
        if (receivedInventory != null) {

            userData = PlayerDataProcessing.inventoryUpdate(context, userData, receivedInventory);

            updatedUserData.items = new ArrayList<>(receivedInventory.getItemsMap().values());
            updatedUserData.uniqueItems = new ArrayList<>(receivedInventory.getUniqueItemsMap().values());
        }

        //Send callback to developer if data has been updated
        if (userData.updated) {
            //Save the update user data information
            userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
            userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);
            userData.getUserDataMeta().serverTime = System.currentTimeMillis();
            UserDataManager.getInstance(context).updateUserData(userData);

            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.ServerUpdate, updatedUserData, getWallet(), getInventory());
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, PlayerDataUpdateReasons.SlotEvent, null, "sdk", null, null, null);
            userData.updated = false;
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataAvailable(getWallet(), getInventory(), GamedockSDK.getInstance(context).getGson().toJson(UserDataManager.getInstance(context).getMissionDataManager().getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(UserDataManager.getInstance(context).getMissionDataManager().getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method that initialises the {@link Wallet} object with the values defined in the {@link GamedockGameData}.
     *
     * @param userData The {@link UserData} object in which all the information related to the {@link Wallet} is stored.
     * @return Returns the updated {@link UserData}.
     */
    public UserData setWalletInitialValues(UserData userData) {
        boolean walletInit = storageUtil.getBoolean(StorageUtil.Keys.WalletInit, false);

        if (!walletInit) {
            userData = PlayerDataInitialisation.walletInitialValues(userData, context);

            storageUtil.putBoolean(StorageUtil.Keys.WalletInit, true);
            initialValue = true;
        }

        return userData;
    }

    /**
     * Method that initialises the {@link Inventory} object with the values defined in the {@link GamedockGameData}.
     *
     * @param userData The {@link UserData} object in which all the information related to the {@link Inventory} is stored.
     * @return Returns the updated {@link UserData}.
     */
    public UserData setInventoryInitialValues(UserData userData) {
        boolean inventoryInit = storageUtil.getBoolean(StorageUtil.Keys.InventoryInit, false);

        if (!inventoryInit) {
            userData = PlayerDataInitialisation.inventoryInitialValues(userData, context);

            storageUtil.putBoolean(StorageUtil.Keys.InventoryInit, true);
            initialValue = true;
        }

        return userData;
    }

    /**
     * Method that retrieves the player {@link Wallet} from the shared preferences.
     *
     * @return The {@link Wallet} object represented as a JSON {@link String}.
     */
    public String getWallet() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.WalletNotFound);
            return null;
        }

        Wallet wallet = userData.getWallet().BuildForJson();

        return gson.toJson(wallet);
    }

    /**
     * Method that retrieves the player {@link Inventory} from the shared preferences.
     *
     * @return The {@link Inventory} object represented as a JSON {@link String}.
     */
    public String getInventory() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.InventoryNotFound);
            return null;
        }

        Inventory inventory = userData.getInventory().BuildForJson();

        return gson.toJson(inventory);
    }

    /**
     * Method that retrieves the {@link PlayerCurrency} based on currency id from the {@link UserData}.
     *
     * @param currencyId The id for which the {@link PlayerCurrency} needs to be retrieved.
     * @return Returns the {@link PlayerCurrency} object that matches the id or {@code null} if no {@link PlayerCurrency} was found.
     */
    public PlayerCurrency getPlayerCurrency(int currencyId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData != null) {
            return userData.getWallet().getCurrenciesMap().get(currencyId);
        }

        return null;
    }

    /**
     * Method that resets the {@link Wallet} information and re-initialises the initial values.
     */
    public void resetWallet() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();


        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.WalletNotFound);
            return;
        }

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return;
        }

        userData = PlayerDataOperations.resetWallet(userData, gamedockGameData);

        GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.Reset, new UpdatedUserData(), getWallet(), getInventory());
        PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, PlayerDataUpdateReasons.Reset);
    }

    /**
     * Method that resets the {@link Inventory} information and re-initialises the initial values.
     */
    public void resetInventory() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.InventoryNotFound);
            return;
        }

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return;
        }

        userData = PlayerDataOperations.resetInventory(userData, gamedockGameData);

        GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.Reset, new UpdatedUserData(), getWallet(), getInventory());
        PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, PlayerDataUpdateReasons.Reset);
    }

    /**
     * Method that resets the {@link Wallet} and the {@link Inventory} information contained in the {@link UserData} and re-initialises the initial values.
     * Uses the resetWallet and resetInventory methods internally.
     */
    public void resetPlayerData(boolean shouldSendResetEvent) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (gamedockGameData == null) {
            GamedockSDK.getInstance(context).getGameDataCallbacks().gameDataError(ErrorCodes.LoadFailed);
            return;
        }

        userData = PlayerDataOperations.resetUserProfile(userData, gamedockGameData);
        UserDataManager.getInstance(context).updateUserData(userData);

        if (shouldSendResetEvent) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(PlayerDataUpdateReasons.Reset, new UpdatedUserData(), getWallet(), getInventory());
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, PlayerDataUpdateReasons.Reset);
        }
    }

    /**
     * Method that updates the wallet currency with an amount and then sends an updatePlayerData event with the data and a reason to the server.
     * Follows the following example @see <a href="https://gist.github.com/sebastian24/392bd6a37d6c09c4bec9a13cb0e1bf3a">Case 1</a>.
     *
     * @param currencyId    The id of the {@link PlayerCurrency} that needs to be updated.
     * @param delta         The value with which the {@link PlayerCurrency} has to be updated. Can be both a positive or a negative value.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData updateWallet(int currencyId, int delta, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        LoggingUtil.d("Update wallet: " + "id: " + currencyId + " delta: " + delta + " reason: " + reason);

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        //Currency id needs to be different from 0 and a reason needs to always be passed in order to process the data
        if (currencyId <= 0 || reason == null) {
            userDataCallback.userDataError(ErrorCodes.CurrencyOperation);
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.WalletNotFound);
            return null;
        }

        PlayerCurrency currency = getPlayerCurrency(currencyId);

        if (currency == null) {
            userDataCallback.userDataError(ErrorCodes.CurrencyNotFound);
            return null;
        }

        return PlayerDataOperations.updateWallet(context, userData, currency, gson, delta, reason, reasonDetails, location, transactionId, perkItems, isTransaction, userDataCallback);
    }

    /**
     * Method that updates the inventory item with an amount and then sends an updatePlayerData event with the data and a reason to the server.
     * Follows the following example @see <a href="https://gist.github.com/sebastian24/392bd6a37d6c09c4bec9a13cb0e1bf3a">Case 2</a>.
     *
     * @param itemId        The id of the {@link PlayerItem} that needs to be updated.
     * @param amount        The value with which the {@link PlayerItem} has to be updated. Can only be positive.
     * @param action        The action that has to be done by the update. Can be "add" or "subtract".
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData updateInventoryWithItem(int itemId, int amount, String action, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        LoggingUtil.d("Update inventory: " + "id: " + itemId + " amount: " + amount + " reason: " + reason);

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        Item gameItem = GamedockGameDataManager.getInstance(context).getItem(itemId);

        //Item id needs to be different from 0 and a reason needs to always be passed in order to process the data
        if (gameItem == null || itemId <= 0 || action == null || reason == null) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        return PlayerDataOperations.updateInventoryWithItem(context, userData, gameItem, gson, amount, action, reason, reasonDetails, location, transactionId, perkItems, isTransaction, userDataCallback);
    }

    /**
     * Method that creates a new UniquePlayerItem.
     *
     * @param itemId   The id of the item from which the UniquePlayerItem should be created.
     * @param uniqueId Optional id that should be use instead of the random unique id creation.
     * @return The newly created UniquePlayerItem.
     */
    public UniquePlayerItem createUniqueItem(int itemId, String uniqueId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        LoggingUtil.d("Creating unique item: " + "id: " + itemId + " uniqueId: " + uniqueId);

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        Item gameItem = GamedockGameDataManager.getInstance(context).getItem(itemId);

        if (gameItem == null || itemId <= 0) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        if (uniqueId != null && userData.getInventory().getUniqueItemsMap().containsKey(uniqueId)) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
        if (uniqueId != null) {
            uniquePlayerItem.setUniqueId(uniqueId);
        } else if (uniquePlayerItem.getUniqueId() == null) {
            uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
        }

        return uniquePlayerItem;
    }

    /**
     * Method that adds a UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be added.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData addUniqueItemToInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        if (uniquePlayerItem == null) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        LoggingUtil.d("Adding unique item to inventory: " + "id: " + uniquePlayerItem.getUniqueId());

        return PlayerDataOperations.addUniqueItemToInventory(context, userData, gson, uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method that updates the UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be updated.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData updateUniqueItemFromInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        if (uniquePlayerItem == null) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        LoggingUtil.d("Updating unique item to inventory: " + gson.toJson(uniquePlayerItem));

        return PlayerDataOperations.updateUniqueItemFromInventory(context, userData, gson, uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method that removes a UniquePlayerItem to the user's inventory.
     *
     * @param uniquePlayerItem The UniquePlayerItem that needs to be removed.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public UpdatedUserData removeUniqueItemFromInventory(UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        if (uniquePlayerItem == null) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        LoggingUtil.d("Removing unique item to inventory: " + "id: " + uniquePlayerItem.getUniqueId());

        return PlayerDataOperations.removeUniqueItemFromInventory(context, userData, gson, uniquePlayerItem, reason, reasonDetails, location, transactionId, isTransaction, userDataCallback);
    }

    /**
     * Method that updates the inventory and wallet by consuming the bundle and then sends an updatePlayerData event with the data and a reason to the server.
     * Follows the following example @see <a href="https://gist.github.com/sebastian24/392bd6a37d6c09c4bec9a13cb0e1bf3a">Case 3</a>.
     *
     * @param bundleId      The id of the {@link Bundle} that needs to be updated.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void buyBundle(int bundleId, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        LoggingUtil.d("Buying bundle: " + "id: " + bundleId + " reason: " + reason);

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return;
        }

        final Bundle bundle = GamedockGameDataManager.getInstance(context).getBundle(bundleId);

        if (bundle == null || bundleId <= 0 || reason == null) {
            userDataCallback.userDataError(ErrorCodes.BundleOperation);
            return;
        }

        PlayerDataOperations.buyBundle(context, userData, bundle, gson, reason, reasonDetails, location, transactionId, perkItems, userDataCallback);
    }

    /**
     * Method used to consume the contents of a bundle without deducting the price for it.
     * This method is only used internally.
     *
     * @param bundleId      The id of the {@link Bundle} that needs to be updated.
     * @param amount        The amount of bundles that need to be opened.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public void openBundle(int bundleId, int amount, String reason, String reasonDetails, String location, ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        LoggingUtil.d("Opening bundle: " + "id: " + bundleId + " reason: " + reason);

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return;
        }

        final Bundle bundle = GamedockGameDataManager.getInstance(context).getBundle(bundleId);

        if (bundle == null || bundleId <= 0 || reason == null) {
            userDataCallback.userDataError(ErrorCodes.BundleOperation);
            return;
        }

        for (int i = 0; i < amount; i++) {
            PlayerDataOperations.openBundle(context, userData, bundle, gson, reason, reasonDetails, location, perkItems, userDataCallback);
        }
    }

    /**
     * Method that opens the gacha box and adds the contents to the user's {@link Wallet} and {@link Inventory}.
     * Gacha box needs to be in the user's inventory
     *
     * @param gachaId       The id of the Gacha {@link Item} that needs to be opened.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public UpdatedUserData openGacha(int gachaId, String reason, String reasonDetails, String location, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        LoggingUtil.d("Opening gacha: " + "id: " + gachaId + " reason: " + reason);

        if (userDataCallback == null) {
            userDataCallback = GamedockSDK.getInstance(context).getUserDataCallbacks();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userDataCallback.userDataError(ErrorCodes.LoadFailed);
            return null;
        }

        PlayerItem gachaPlayerItem = userData.getInventory().getItemsMap().get(gachaId);
        Item gachaItem = GamedockGameDataManager.getInstance(context).getGacha(gachaId);

        if (gachaPlayerItem == null || gachaItem == null || gachaId <= 0 || reason == null || !gachaItem.isGacha()) {
            userDataCallback.userDataError(ErrorCodes.GachaOperation);
            return null;
        }

        gachaPlayerItem.populateValues(gachaItem);

        return PlayerDataOperations.openGacha(context, userData, gachaPlayerItem, reason, reasonDetails, location, perkItems, isTransaction, userDataCallback);
    }

    /**
     * Method that requests manually any changes that might have happened on the server.
     */
    public void pullPlayerDataChanges() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
        } else {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, userData);
        }
    }


}
