package io.gamedock.sdk.userdata;

import android.content.Context;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataSending;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class UserDataTransaction {
    public static class TransactionSequence {
        public static final String SourceWallet = "Wallet";
        public static final String SourceInventory = "Inventory";

        public static final String OperationAdd = "Add";
        public static final String OperationSubtract = "Subtract";
        public static final String OperationUpdate = "Update";

        public static final String TypeCurrency = "Currency";
        public static final String TypeItem = "Item";
        public static final String TypeUniqueItem = "UniqueItem";
        public static final String TypeGacha = "Gacha";

        private ArrayList<Transaction> transactions = new ArrayList<>();
        private UserDataCallbacks callback;

        public TransactionSequence() {
        }

        public TransactionSequence addCurrency(int currencyId, int amount) {
            transactions.add(new Transaction(SourceWallet, OperationAdd, TypeCurrency, currencyId, amount, null));
            return this;
        }

        public TransactionSequence subtractCurrency(int currencyId, int amount) {
            transactions.add(new Transaction(SourceWallet, OperationSubtract, TypeCurrency, currencyId, amount, null));
            return this;
        }

        public TransactionSequence addItem(int itemId, int amount) {
            transactions.add(new Transaction(SourceInventory, OperationAdd, TypeItem, itemId, amount, null));
            return this;
        }

        public TransactionSequence subtractItem(int itemId, int amount) {
            transactions.add(new Transaction(SourceInventory, OperationSubtract, TypeItem, itemId, amount, null));
            return this;
        }

        public TransactionSequence openGacha(int gachaId) {
            transactions.add(new Transaction(SourceInventory, OperationAdd, TypeGacha, gachaId, 1, null));
            return this;
        }

        public TransactionSequence addUniqueItem(UniquePlayerItem uniquePlayerItem) {
            transactions.add(new Transaction(SourceInventory, OperationAdd, TypeUniqueItem, 0, 1, uniquePlayerItem));
            return this;
        }

        public TransactionSequence removeUniqueItem(UniquePlayerItem uniquePlayerItem) {
            transactions.add(new Transaction(SourceInventory, OperationSubtract, TypeUniqueItem, 0, -1, uniquePlayerItem));
            return this;
        }

        public TransactionSequence updateUniqueItem(UniquePlayerItem uniquePlayerItem) {
            transactions.add(new Transaction(SourceInventory, OperationUpdate, TypeUniqueItem, 0, 0, uniquePlayerItem));
            return this;
        }

        public TransactionSequence addCustomCallback(UserDataCallbacks callback) {
            this.callback = callback;
            return this;
        }

        public void submit(Context context, String reason, String reasonDetails, String location, String transactionId) {
            UserDataCallbacks userDataCallbacks;
            if (this.callback != null) {
                userDataCallbacks = this.callback;
            } else {
                userDataCallbacks = GamedockSDK.getInstance(context).getUserDataCallbacks();
            }

            boolean validTransactions = validateTransactions(context, transactions);
            if (!validTransactions) {
                userDataCallbacks.userDataError(ErrorCodes.TransactionOperation);
                return;
            }

            UpdatedUserData updatedUserData = new UpdatedUserData();

            for (Transaction transaction : transactions) {
                UpdatedUserData tempUpdatedData = null;
                switch (transaction.source) {
                    case SourceWallet:
                        switch (transaction.operation) {
                            case OperationAdd:
                                tempUpdatedData = GamedockSDK.getInstance(context).addCurrencyToWallet(transaction.entityId, transaction.amount, reason, reasonDetails, location, transactionId, true, null);
                                break;
                            case OperationSubtract:
                                tempUpdatedData = GamedockSDK.getInstance(context).subtractCurrencyFromWallet(transaction.entityId, transaction.amount, reason, reasonDetails, location, transactionId, true, null);
                                break;
                        }
                        break;
                    case SourceInventory:
                        switch (transaction.type) {
                            case TypeItem:
                                if (transaction.operation.equals(OperationAdd)) {
                                    tempUpdatedData = GamedockSDK.getInstance(context).addItemToInventory(transaction.entityId, transaction.amount, reason, reasonDetails, location, transactionId, true, null);
                                } else if (transaction.operation.equals(OperationSubtract)) {
                                    tempUpdatedData = GamedockSDK.getInstance(context).subtractItemFromInventory(transaction.entityId, transaction.amount, reason, reasonDetails, location, transactionId, true, null);
                                }
                                break;
                            case TypeGacha:
                                tempUpdatedData = GamedockSDK.getInstance(context).openGacha(transaction.entityId, reason, reasonDetails, location, null, true, null);
                                break;
                            case TypeUniqueItem:
                                switch (transaction.operation) {
                                    case OperationAdd:
                                        tempUpdatedData = GamedockSDK.getInstance(context).addUniqueItemToInventory(transaction.uniquePlayerItem, reason, reasonDetails, location, transactionId, true, null);
                                        break;
                                    case OperationSubtract:
                                        tempUpdatedData = GamedockSDK.getInstance(context).removeUniqueItemFromInventory(transaction.uniquePlayerItem, reason, reasonDetails, location, transactionId, true, null);
                                        break;
                                    case OperationUpdate:
                                        tempUpdatedData = GamedockSDK.getInstance(context).updateUniqueItemFromInventory(transaction.uniquePlayerItem, reason, reasonDetails, location, transactionId, true, null);
                                        break;
                                }
                                break;
                        }
                        break;
                }

                if (tempUpdatedData == null) {
                    userDataCallbacks.userDataError(ErrorCodes.TransactionOperation);
                    return;
                }

                updateTransactionData(updatedUserData, tempUpdatedData);
            }

            updatedUserData.transactionId = transactionId;

            UserData userData = UserDataManager.getInstance(context).getUserData();

            if (userData == null) {
                userDataCallbacks.userDataError(ErrorCodes.TransactionOperation);
                return;
            }

            PlayerDataSending.sendUpdatePlayerDataEvent(context, GamedockSDK.getInstance(context).getGson(), userData, null, reason, reasonDetails, location, transactionId, null, null);
            userDataCallbacks.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        private boolean validateTransactions(Context context, ArrayList<Transaction> transactions) {
            LoggingUtil.d("Validating transactions");

            UserData userData = UserDataManager.getInstance(context).getUserData();

            if (userData == null) {
                return false;
            }

            for (Transaction transaction : transactions) {
                switch (transaction.source) {
                    case SourceWallet:
                        PlayerCurrency currency = UserDataManager.getInstance(context).getPlayerDataManager().getPlayerCurrency(transaction.entityId);

                        if (currency == null) {
                            return false;
                        }

                        if (transaction.operation.equals(OperationSubtract)) {
                            int currentBalance = currency.getCurrentBalance();
                            int updatedBalance = currentBalance - transaction.amount;
                            if (updatedBalance < 0) {
                                return false;
                            }
                        }
                        break;
                    case SourceInventory:
                        switch (transaction.type) {
                            case TypeItem:
                                Item gameItem = GamedockGameDataManager.getInstance(context).getItem(transaction.entityId);

                                if (gameItem == null) {
                                    return false;
                                }

                                PlayerItem inventoryItem = userData.getInventory().getItemsMap().get(gameItem.getId());

                                if (inventoryItem != null) {
                                    if (transaction.operation.equals(OperationSubtract)) {
                                        int currentAmount = inventoryItem.getAmount();
                                        int updatedAmount = currentAmount - transaction.amount;

                                        if (updatedAmount < 0) {
                                            return false;
                                        }
                                    }
                                }
                                break;
                            case TypeGacha:
                                PlayerItem gachaPlayerItem = userData.getInventory().getItemsMap().get(transaction.entityId);
                                Item gachaItem = GamedockGameDataManager.getInstance(context).getGacha(transaction.entityId);

                                if (gachaPlayerItem == null || gachaItem == null || !gachaItem.isGacha()) {
                                    return false;
                                }

                                if (gachaPlayerItem.getAmount() < 1 || gachaPlayerItem.getContent().isEmpty()) {
                                    return false;
                                }
                                break;
                            case TypeUniqueItem:
                                if (transaction.uniquePlayerItem == null) {
                                    return false;
                                }

                                switch (transaction.operation) {
                                    case OperationAdd:
                                        if (userData.getInventory().getUniqueItemsMap().containsKey(transaction.uniquePlayerItem.getUniqueId())) {
                                            return false;
                                        }
                                        break;
                                    case OperationSubtract:
                                    case OperationUpdate:
                                        if (!userData.getInventory().getUniqueItemsMap().containsKey(transaction.uniquePlayerItem.getUniqueId())) {
                                            return false;
                                        }
                                        break;
                                }
                                break;
                        }
                        break;
                }
            }

            LoggingUtil.d("Transactions valid");
            return true;
        }

        private void updateTransactionData(UpdatedUserData currentData, UpdatedUserData newData) {
            ArrayList<PlayerCurrency> tempNewCurrencies = new ArrayList<>();
            for (PlayerCurrency newCurrency : newData.currencies) {
                boolean isNew = true;
                for (PlayerCurrency currentCurrency : currentData.currencies) {
                    if (newCurrency.getId() == currentCurrency.getId()) {
                        currentCurrency.setCurrentBalance(newCurrency.getCurrentBalance());
                        currentCurrency.setDelta(currentCurrency.getDelta() + newCurrency.getDelta());
                        isNew = false;
                        break;
                    }
                }
                if (isNew) {
                    tempNewCurrencies.add(newCurrency);
                }
            }
            currentData.currencies.addAll(tempNewCurrencies);

            ArrayList<PlayerItem> tempNewItems = new ArrayList<>();
            for (PlayerItem newItem : newData.items) {
                boolean isNew = true;
                for (PlayerItem currentItem : currentData.items) {
                    if (newItem.getId() == currentItem.getId()) {
                        currentItem.setAmount(newItem.getAmount());
                        currentItem.setDelta(currentItem.getDelta() + newItem.getDelta());
                        isNew = false;
                        break;
                    }
                }
                if (isNew) {
                    tempNewItems.add(newItem);
                }
            }
            currentData.items.addAll(tempNewItems);
            currentData.uniqueItems.addAll(newData.uniqueItems);

            currentData.gachaId = newData.gachaId;
            currentData.gachaPosition = newData.gachaPosition;
        }
    }

    public static TransactionSequence Builder() {
        return new TransactionSequence();
    }

    public static class Transaction {
        public String source;
        public String operation;
        public String type;
        public int entityId;
        public int amount;
        public UniquePlayerItem uniquePlayerItem;

        public Transaction(String source, String operation, String type, int entityId, int amount, UniquePlayerItem uniquePlayerItem) {
            this.source = source;
            this.operation = operation;
            this.type = type;
            this.entityId = entityId;
            this.amount = amount;
            this.uniquePlayerItem = uniquePlayerItem;
        }
    }


}
