package io.gamedock.sdk.userdata;


import com.google.gson.Gson;
import com.unity3d.player.UnityPlayer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.ContainerProgress;
import io.gamedock.sdk.models.userdata.mission.MissionProgress;
import io.gamedock.sdk.models.userdata.mission.UpdatedMissionData;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.web.WebViewActivity;

public class UserDataCallbacks {

    private OnUserDataListener userDataListener;

    public UserDataCallbacks() {
        userDataListener = null;
    }

    private String callbackId;

    public UserDataCallbacks(OnUserDataListener userDataListener) {
        this.userDataListener = userDataListener;
    }

    /**
     * Method that is called when the Player Data has been initialised by the SDK.
     */
    public void userDataAvailable(String walletJSON, String inventoryJSON, String containersJSON, String missionJSON) {
        if (userDataListener != null) {
            userDataListener.UserDataAvailable(walletJSON, inventoryJSON, containersJSON, missionJSON);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("wallet", walletJSON);
                jsonObject.put("inventory", inventoryJSON);
                jsonObject.put("containers", containersJSON);
                jsonObject.put("missions", missionJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataAvailable", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while processing the Player Data.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void userDataError(ErrorCodes error) {
        LoggingUtil.d("Player data error with reason: " + error);
        try {
            Gson gson = new Gson();

            JSONObject jsonObject = new JSONObject();

            JSONObject errorJsonObject = new JSONObject();
            errorJsonObject.put("id", error.getId());
            errorJsonObject.put("name", error.getName());
            errorJsonObject.put("message", error.getMessage());
            errorJsonObject.put("errorContext", new JSONObject(gson.toJson(error.getErrorContext())));
            jsonObject.put("error", errorJsonObject);

            if (callbackId != null) {
                jsonObject.put("callbackId", callbackId);
            }

            if (userDataListener != null) {
                userDataListener.UserDataError(error);
            } else {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataError", jsonObject.toString());
            }

            if (WebViewActivity.getActivity() != null) {
                WebViewActivity.getActivity().javascriptBridge.nativeMessage("userDataError", jsonObject.toString());
            }
        } catch (NoClassDefFoundError | JSONException e) {
            LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
        }
    }

    /**
     * Method that is called to notify the game a merge conflict has been triggered. Contains the remote and local data.
     *
     * @param localData  The local user data stored on the device.
     * @param remoteData The remote user data received from the backend.
     */
    public void userDataMergeConflict(JSONObject localData, JSONObject remoteData) {
        LoggingUtil.d("Merge conflict detected. Local data: " + localData.toString() + "/n Remote data: " + remoteData);
        if (userDataListener != null) {
            userDataListener.UserDataMergeConflict(localData, remoteData);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("localData", localData);
                jsonObject.put("remoteData", remoteData);

                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataMergeConflict", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called to notify the game that the user data merge has been successful.
     */
    public void userDataMergeSuccessful(String walletJSON, String inventoryJSON, String containersJSON, String missionJSON) {
        LoggingUtil.d("User data merge successful!");
        if (userDataListener != null) {
            userDataListener.UserDataMergeSuccessful(walletJSON, inventoryJSON, containersJSON, missionJSON);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("wallet", walletJSON);
                jsonObject.put("inventory", inventoryJSON);
                jsonObject.put("containers", containersJSON);
                jsonObject.put("missions", missionJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataMergeSuccessful", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called to notify the game that the user data merging has failed.
     *
     * @param mergeData The user data that should have been merged.
     * @param mergeType The typ of merge that was selected by the game.
     */
    public void userDataMergeFailed(JSONObject mergeData, String mergeType) {
        LoggingUtil.d("User data merge successful!");
        if (userDataListener != null) {
            userDataListener.UserDataMergeFailed(mergeData, mergeType);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("mergeData", mergeData);
                jsonObject.put("mergeType", mergeType);

                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataMergeFailed", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called to notify the game that a write lock occurred on the server and no data was saved.
     */
    public void userDataLockError() {
        LoggingUtil.d("Lock error occurred!");
        if (userDataListener != null) {
            userDataListener.UserDataLockError();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataLockError", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called to notify the game that a sync error occurred on the server and data will need to be merged.
     */
    public void userDataSyncError() {
        LoggingUtil.d("Sync error occurred!");
        if (userDataListener != null) {
            userDataListener.UserDataSyncError();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataSyncError", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called to notify the game what type of merge was selected by the user, if the game is using the built in merge dialog.
     *
     * @param mergeType The merge type. Can be LOCAL, REMOTE, MERGE.
     */
    public void userDataHandleMerge(String mergeType) {
        if (userDataListener != null) {
            userDataListener.UserDataHandleMerge(mergeType);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserDataHandleMerge", mergeType);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is fired whenever a Player Data operation has occurred and the value that the player has have changed.
     * Includes both local and server updates.
     *
     * @param reason          The reason for which the Player Data has been updated.
     * @param updatedUserData The data that has been updated in the operation displayed in a JSON {@link String} format.
     */
    public void playerDataUpdated(String reason, UpdatedUserData updatedUserData, String walletJSON, String inventoryJSON) {
        if (userDataListener != null) {
            userDataListener.PlayerDataUpdated(reason, updatedUserData, walletJSON, inventoryJSON);
        } else {
            try {
                Gson gson = new Gson();

                JSONObject jsonObject = new JSONObject();
                jsonObject.put("reason", reason);
                jsonObject.put("bundleId", updatedUserData.bundleId);
                jsonObject.put("gachaId", updatedUserData.gachaId);
                jsonObject.put("gachaPosition", updatedUserData.gachaPosition);

                ArrayList<PlayerCurrency> playerCurrencies = updatedUserData.currencies;
                JSONArray currencyJSONArray = new JSONArray();

                for (PlayerCurrency playerCurrency : playerCurrencies) {
                    JSONObject currencyJSON = new JSONObject(gson.toJson(playerCurrency));
                    currencyJSONArray.put(currencyJSON);
                }

                jsonObject.put("currencies", currencyJSONArray);

                ArrayList<PlayerItem> playerItems = updatedUserData.items;
                JSONArray itemsJSONArray = new JSONArray();

                for (PlayerItem playerItem : playerItems) {
                    JSONObject itemJSON = new JSONObject(gson.toJson(playerItem));
                    itemsJSONArray.put(itemJSON);
                }

                jsonObject.put("items", itemsJSONArray);

                ArrayList<UniquePlayerItem> uniquePlayerItems = updatedUserData.uniqueItems;
                JSONArray uniqueItemsJSONArray = new JSONArray();

                for (UniquePlayerItem uniquePlayerItem : uniquePlayerItems) {
                    JSONObject uniqueItemJSON = new JSONObject(gson.toJson(uniquePlayerItem));
                    uniqueItemsJSONArray.put(uniqueItemJSON);
                }

                jsonObject.put("uniqueItems", uniqueItemsJSONArray);

                if (updatedUserData.perkItems != null) {
                    JSONArray perkItemsJSON = new JSONArray(gson.toJson(updatedUserData.perkItems));
                    jsonObject.put("perkItems", perkItemsJSON);
                }

                jsonObject.put("claimedContainers", new JSONArray(updatedUserData.claimedContainers));
                jsonObject.put("claimedMissions", new JSONArray(updatedUserData.claimedMissions));

                jsonObject.put("wallet", walletJSON);
                jsonObject.put("inventory", inventoryJSON);

                if (callbackId != null) {
                    jsonObject.put("callbackId", callbackId);
                }

                UnityPlayer.UnitySendMessage("GamedockSDK", "PlayerDataUpdated", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is fired when a new UniquePlayerItem has been created from buyBundle, openBundle or openGacha methods.
     *
     * @param uniquePlayerItem The newly created UniquePlayerItem.
     */
    public void playerDataNewUniqueItem(UniquePlayerItem uniquePlayerItem, int bundleId, int gachaId, int gachaPosition, int tierId, String reason) {
        if (userDataListener != null) {
            userDataListener.PlayerDataNewUniqueItem(uniquePlayerItem);
        } else {
            try {
                Gson gson = new Gson();
                JSONObject jsonObject = new JSONObject();

                JSONObject newUniqueItemJSON = new JSONObject();
                newUniqueItemJSON.put("uniqueItem", new JSONObject(gson.toJson(uniquePlayerItem)));
                newUniqueItemJSON.put("bundleId", bundleId);
                newUniqueItemJSON.put("gachaId", gachaId);
                newUniqueItemJSON.put("gachaPosition", gachaPosition);
                newUniqueItemJSON.put("tierId", tierId);
                newUniqueItemJSON.put("reason", reason);
                jsonObject.put("newUniqueItemJSON", newUniqueItemJSON);

                if (callbackId != null) {
                    jsonObject.put("callbackId", callbackId);
                }

                UnityPlayer.UnitySendMessage("GamedockSDK", "PlayerDataNewUniqueItem", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when nothing was received from the gacha box.
     */
    public void playerDataEmptyGacha() {
        if (userDataListener != null) {
            userDataListener.PlayerDataEmptyGacha();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "PlayerDataEmptyGacha", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when the Game State data has been updated from the server. The data can be either "private" or "public".
     *
     * @param access The type of access the Game State belongs to. Can be "public" or "private".
     */
    public void gameStateUpdated(String access) {
        if (userDataListener != null) {
            userDataListener.GameStateUpdated(access);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "GameStateUpdated", access);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is fired whenever a Mission Data operation has occurred and the value that the player has have changed.
     * Includes both local and server updates.
     *
     * @param reason             The reason for which the Mission Data has been updated.
     * @param updatedMissionData The data that has been updated in the operation displayed in a JSON {@link String} format.
     */
    public void missionDataUpdated(String reason, UpdatedMissionData updatedMissionData, String containersJSON, String missionsJSON) {
        if (userDataListener != null) {
            userDataListener.MissionDataUpdated(reason, updatedMissionData, containersJSON, missionsJSON);
        } else {
            try {
                Gson gson = new Gson();

                JSONObject jsonObject = new JSONObject();
                jsonObject.put("reason", reason);

                JSONArray containerProgressJSONArray = new JSONArray();
                for (ContainerProgress containerProgress : updatedMissionData.getContainerProgressList()) {
                    JSONObject currencyJSON = new JSONObject(gson.toJson(containerProgress));
                    containerProgressJSONArray.put(currencyJSON);
                }

                jsonObject.put("containerProgress", containerProgressJSONArray);

                JSONArray missionProgressJSONArray = new JSONArray();
                for (MissionProgress missionProgress : updatedMissionData.getMissionProgressList()) {
                    JSONObject missionProgressJSON = new JSONObject(gson.toJson(missionProgress));
                    missionProgressJSONArray.put(missionProgressJSON);
                }

                jsonObject.put("missionProgress", missionProgressJSONArray);

                jsonObject.put("containers", containersJSON);
                jsonObject.put("missions", missionsJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "MissionDataUpdated", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when data regarding other users has been received from the backend.
     * Contains a list of "public" data associated with the ids requested.
     *
     * @param provider The provider associated with the ids for which the game states were requested.
     * @param data     The "public" game state data for the requested users.
     */
    public void otherUsersGameStateLoaded(String provider, JSONObject data) {
        LoggingUtil.d("Other Users Game State loaded with provider: " + provider + " and data: " + data.toString());

        if (userDataListener != null) {
            userDataListener.OtherUsersGameStateLoaded(provider, data);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("provider", provider);
                jsonObject.put("data", data);

                UnityPlayer.UnitySendMessage("GamedockSDK", "OtherUsersGameStateLoaded", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the UserDataCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    public String getCallbackId() {
        return callbackId;
    }

    public void setCallbackId(String callbackId) {
        this.callbackId = callbackId;
    }
}
