package io.gamedock.sdk.tier;

import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.TierEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.reward.Reward;
import io.gamedock.sdk.models.tier.TieredEvent;
import io.gamedock.sdk.models.tier.TieredEventProgress;
import io.gamedock.sdk.models.tier.TieredEventTier;
import io.gamedock.sdk.models.tier.TieredEventsOverview;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.web.WebViewActivity;

/**
 * Class used to manage all Tiered Events operations.
 */
public class TieredEventsManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "tieredEvents";

    public static final String TieredEventId = "tieredEventId";
    public static final String TieredEvents = "tieredEvents";
    public static final String TierEventName = "tieredEventName";
    public static final String EntityId = "entityId";
    public static final String EntityType = "entityType";
    public static final String EntityAmount = "entityAmount";
    public static final String TierName = "tierName";
    public static final String Tiers = "tiers";
    public static final String Id = "id";
    public static final String Rewards = "rewards";
    public static final String Currency = "CURRENCY";
    public static final String Item = "ITEM";
    public static final String Progress = "progress";
    public static final String Gacha = "GACHA";

    private static volatile TieredEventsManager mInstance = null;
    private Context context;

    private Gson gson;

    private TieredEventsOverview tieredEventsOverview = new TieredEventsOverview();

    private TieredEventsManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
    }

    public static TieredEventsManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new TieredEventsManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to send a tiered event configuration request.
     */
    public void requestTieredEvents() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        TierEvent tierEvent = new TierEvent(context);
        tierEvent.setRequestTieredEvent();

        GamedockSDK.getInstance(context).trackEvent(tierEvent, null);
    }

    /**
     * Method used to process the tiered event configuration response from the backend.
     *
     * @param responseData The tiered event configuration data.
     */
    public void processRequestTieredEvent(JSONObject responseData) {
        try {
            JSONArray tieredEventsJSON = responseData.getJSONArray(TieredEvents);
            JsonParser jsonParser = new JsonParser();

            for (int i = 0; i < tieredEventsJSON.length(); i++) {
                TieredEvent tieredEvent = gson.fromJson(tieredEventsJSON.getJSONObject(i).toString(), TieredEvent.class);
                if(tieredEventsJSON.getJSONObject(i).has("properties")) {
                    JsonObject properties = (JsonObject) jsonParser.parse(tieredEventsJSON.getJSONObject(i).getJSONObject("properties").toString());
                    tieredEvent.setProperties(properties);
                }

                for (int j = 0; j < tieredEventsJSON.getJSONObject(i).getJSONArray("tiers").length(); j++) {
                    if (tieredEventsJSON.getJSONObject(i).getJSONArray("tiers").getJSONObject(j).has("properties")) {
                        JsonObject tierProperties = (JsonObject) jsonParser.parse(tieredEventsJSON.getJSONObject(i).getJSONArray("tiers").getJSONObject(j).getJSONObject("properties").toString());
                        tieredEvent.getTiers().get(j).setProperties(tierProperties);
                    }
                 }

                tieredEventsOverview.getTieredEvents().put(tieredEvent.getId(), tieredEvent);
            }

            JSONArray tieredProgressJSON = responseData.getJSONArray(Progress);

            for (int i = 0; i < tieredProgressJSON.length(); i++) {
                TieredEventProgress tieredProgress = gson.fromJson(tieredProgressJSON.getJSONObject(i).toString(), TieredEventProgress.class);
                tieredEventsOverview.getProgress().put(tieredProgress.getTieredEventId(), tieredProgress);
            }

            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsAvailable();
        } catch (Exception e) {
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsNotAvailable();
            e.printStackTrace();
        }
    }

    /**
     * Method used to send an event towards the backend informing a specific tier in the tiered event has been updated with progress.
     *
     * @param tieredEventId The tiered event id.
     * @param tierId        The tier id that is contained in the tiered event.
     * @param entityId      The entity id of the entity that has been updated.
     * @param entityType    The type of entity that has been updated (ex.: currency, item, etc.)
     * @param entityAmount  The value with which it has been updated.
     */
    public void updateTierProgress(int tieredEventId, int tierId, int entityId, String entityType, int entityAmount) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        try {

        } catch (Exception e) {
            e.printStackTrace();
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsError(ErrorCodes.TieredEventUpdateProgressError);
        }

        TierEvent tierEvent = new TierEvent(context);
        tierEvent.setUpdateTierProgress();

        tierEvent.addCustomData(TieredEventId, tieredEventId);
        tierEvent.addCustomData(TierEventName, tieredEventsOverview.getTieredEvents().get(tieredEventId).getName());
        tierEvent.addCustomData(EntityId, entityId);
        tierEvent.addCustomData(EntityType, entityType);
        tierEvent.addCustomData(EntityAmount, entityAmount);

        String tierName = "";
        for (TieredEventTier tier : tieredEventsOverview.getTieredEvents().get(tieredEventId).getTiers()) {
            if (tier.getId() == tierId) {
                tierName = tier.getName();
                break;
            }
        }

        tierEvent.addCustomData(TierName, tierName);

        GamedockSDK.getInstance(context).trackEvent(tierEvent, null);
    }

    /**
     * Method used to process the response from the backend for the tier update.
     *
     * @param responseData The tier progress information.
     */
    public void processUpdateTierProgress(JSONObject responseData) {
        try {
            TieredEventProgress tieredProgress = gson.fromJson(responseData.toString(), TieredEventProgress.class);

            tieredEventsOverview.getProgress().put(tieredProgress.getTieredEventId(), tieredProgress);

            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventUpdated(tieredProgress);
        } catch (Exception e) {
            e.printStackTrace();
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsError(ErrorCodes.TieredEventUpdateProgressError);
        }
    }

    /**
     * Method used to send a claim reward event, notifying the backend that a tier reward should be claimed.
     *
     * @param tieredEventId The id of the tiered event.
     * @param tierId        The id of the tier that is to be claimed.
     */
    public void claimTierReward(int tieredEventId, int tierId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        TieredEvent tieredEvent = tieredEventsOverview.getTieredEvents().get(tieredEventId);

        if (tieredEvent == null) {
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsError(ErrorCodes.TieredEventClaimTierError);
            return;
        }

        TierEvent tierEvent = new TierEvent(context);
        tierEvent.setClaimTierReward();

        tierEvent.addCustomData(TieredEventId, tieredEventId);
        tierEvent.addCustomData(TierEventName, tieredEventsOverview.getTieredEvents().get(tieredEventId).getName());

        String tierName = "";
        for (TieredEventTier tier : tieredEventsOverview.getTieredEvents().get(tieredEventId).getTiers()) {
            if (tier.getId() == tierId) {
                tierName = tier.getName();
                break;
            }
        }

        tierEvent.addCustomData(TierName, tierName);

        GamedockSDK.getInstance(context).trackEvent(tierEvent, null);
    }

    /**
     * Method used to process the response from the backend regarding the claiming of a tier reward.
     *
     * @param responseData The tier reward information.
     */
    public void processClaimTierReward(JSONObject responseData) {
        try {
            int tieredEventId = responseData.getInt(TieredEventId);

            JSONArray tiersJSON = responseData.getJSONArray(Tiers);
            for (int i = 0; i < tiersJSON.length(); i++) {
                int tierId = tiersJSON.getJSONObject(i).getInt(Id);

                String tierName = "";
                for (TieredEventTier tier : tieredEventsOverview.getTieredEvents().get(tieredEventId).getTiers()) {
                    if (tier.getId() == tierId) {
                        tierName = tier.getName();
                        break;
                    }
                }

                JSONArray tierRewardsJSON = tiersJSON.getJSONObject(i).getJSONArray(Rewards);
                for (int j = 0; j < tierRewardsJSON.length(); j++) {
                    Reward reward = gson.fromJson(tierRewardsJSON.getJSONObject(i).toString(), Reward.class);

                    if (reward.type.equals(Currency)) {
                        GamedockSDK.getInstance(context).addCurrencyToWallet(reward.id, reward.amount, PlayerDataUpdateReasons.TierReward, tierName, "sdk", null, false, null);
                    } else if (reward.type.equals(Item) || reward.type.equals(Gacha)) {
                        io.gamedock.sdk.models.gamedata.items.Item gameItem = GamedockGameDataManager.getInstance(context).getItem(reward.id);

                        if (gameItem == null) {
                            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemNotFound);
                            return;
                        }

                        if (gameItem.isUnique()) {
                            UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
                            uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
                            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0, 0, tierId, PlayerDataUpdateReasons.TierReward);
                        } else {
                            GamedockSDK.getInstance(context).addItemToInventory(reward.id, reward.amount, PlayerDataUpdateReasons.TierReward, tierName, "sdk", null, false, null);
                        }
                    }
                }
            }

            if (WebViewActivity.getActivity() != null) {
                WebViewActivity.getActivity().javascriptBridge.nativeMessage("claimTierReward", responseData.toString());
            }
        } catch (Exception e) {
            e.printStackTrace();
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsError(ErrorCodes.TieredEventClaimTierError);
        }
    }

    /**
     * Method used to send a request for showing the tiered event. This will trigger a tiered event splash screen.
     *
     * @param tieredEventId The tiered event id.
     */
    public void showTierProgress(int tieredEventId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        TieredEvent tieredEvent = tieredEventsOverview.getTieredEvents().get(tieredEventId);

        if (tieredEvent == null) {
            GamedockSDK.getInstance(context).getTieredEventCallbacks().tieredEventsError(ErrorCodes.TieredEventShowProgressError);
            return;
        }

        TierEvent tierEvent = new TierEvent(context);
        tierEvent.setShowTierProgress();

        tierEvent.addCustomData(TieredEventId, tieredEventId);
        tierEvent.addCustomData(TierEventName, tieredEventsOverview.getTieredEvents().get(tieredEventId).getName());

        GamedockSDK.getInstance(context).trackEvent(tierEvent, null);
    }

    /**
     * Method used to retrieve all the tiered events configurations.
     *
     * @return A JSON list string containing all the tiered events.
     */
    public String getAllTieredEvents() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return gson.toJson(new ArrayList<>());
        }

        ArrayList<TieredEvent> tieredEvents = new ArrayList<>(tieredEventsOverview.getTieredEvents().values());

        return gson.toJson(tieredEvents);
    }

    /**
     * Method used to retrieve a specific tiered event based on an id.
     *
     * @param tieredEventId The id of the tiered event.
     * @return The desired tiered event in a JSON string format or null if no tiered events were found.
     */
    public String getTieredEvent(int tieredEventId) {
        return gson.toJson(tieredEventsOverview.getTieredEvents().get(tieredEventId));
    }

    /**
     * Method used to retrieve a specific tiered event progress based on an id.
     *
     * @param tieredEventId The id of the tiered event.
     * @return The desired tiered event progress in a JSON string format or null if no tiered event progress was found.
     */
    public String getTieredEventProgress(int tieredEventId) {
        return gson.toJson(tieredEventsOverview.getProgress().get(tieredEventId));
    }

    public TieredEventsOverview getTieredEventsOverview() {
        return tieredEventsOverview;
    }

    public void resetTieredEvents() {
        tieredEventsOverview = new TieredEventsOverview();
        mInstance = null;
    }
}
