package io.gamedock.sdk.splashscreen;

import android.content.Context;
import android.content.Intent;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.OverlayEvent;
import io.gamedock.sdk.models.splashscreen.GamedockSplashscreen;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.web.WebViewActivity;

public class SplashscreenManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "splashScreens";

    private static volatile SplashscreenManager mInstance = null;
    private Context context;

    private GamedockSplashscreen delayedSplashscreen;

    private SplashscreenManager(Context context) {
        this.context = context;
    }

    public static SplashscreenManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new SplashscreenManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to request the Splash Screen.
     */
    public void requestSplashScreen(String type) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        OverlayEvent overlayEvent = new OverlayEvent(context);
        overlayEvent.setRequestSplashscreen();

        if (type != null) {
            overlayEvent.addCustomData("type", type);
        }

        GamedockSDK.getInstance(context).trackEvent(overlayEvent, null);
    }

    /**
     * Method used to process the splashscreen response from Gamedock backend.
     *
     * @param splashscreenData The data contained in the splashscreen.
     */
    public void processSplashscreenResponse(JSONObject splashscreenData) {
        try {
            GamedockSplashscreen tempSplashscreen = new GamedockSplashscreen();

            if (splashscreenData.has("campaignId")) {
                tempSplashscreen.setCampaignId(splashscreenData.getInt("campaignId"));
            }

            if (splashscreenData.has("url")) {
                tempSplashscreen.setUrl(splashscreenData.getString("url"));
            }

            if (splashscreenData.has("type")) {
                tempSplashscreen.setType(splashscreenData.getString("type"));
            }

            if (splashscreenData.has("isDelayable")) {
                tempSplashscreen.setDelayable(splashscreenData.getBoolean("isDelayable"));
            }

            if (splashscreenData.has("priority")) {
                tempSplashscreen.setPriority(splashscreenData.getInt("priority"));
            }

            tempSplashscreen.setData(splashscreenData.toString());

            if (tempSplashscreen.getType().equals("assetBundle")) {

            } else if (tempSplashscreen.getType().equals("webView")) {
                if (tempSplashscreen.isDelayable()) {
                    if (delayedSplashscreen == null) {
                        delayedSplashscreen = tempSplashscreen;

                        GamedockSDK.getInstance(context).getSplashScreenCallbacks().splashScreenAvailable(delayedSplashscreen.getType());

                        return;
                    }
                    OverlayEvent droppedSplashscreenEvent = new OverlayEvent(context);
                    droppedSplashscreenEvent.setDroppedSplashscreen();

                    if (tempSplashscreen.getPriority() >= delayedSplashscreen.getPriority()) {
                        droppedSplashscreenEvent.addCustomData("droppedSplashscreenId", delayedSplashscreen.getCampaignId());
                        droppedSplashscreenEvent.addCustomData("newSplashscreenId", tempSplashscreen.getCampaignId());

                        delayedSplashscreen = tempSplashscreen;
                    } else {
                        droppedSplashscreenEvent.addCustomData("droppedSplashscreenId", tempSplashscreen.getCampaignId());
                        droppedSplashscreenEvent.addCustomData("newSplashscreenId", delayedSplashscreen.getCampaignId());
                    }

                    GamedockSDK.getInstance(context).trackEvent(droppedSplashscreenEvent, null);
                } else {
                    displaySplashscreenView(tempSplashscreen.getUrl(), tempSplashscreen.getData());
                }
            }

        } catch (NullPointerException | JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to show a delayed splashscreen.
     */
    public void showSplashscreen() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (delayedSplashscreen == null) {
            GamedockSDK.getInstance(context).getSplashScreenCallbacks().splashScreenNotAvailable();

            return;
        }

        displaySplashscreenView(delayedSplashscreen.getUrl(), delayedSplashscreen.getData());
    }

    private void displaySplashscreenView(String splashscreenUrl, String splashscreenData) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        try {
            GamedockSDK.getInstance(context).isShowingChildActivity = true;

            Intent intent = new Intent(context, WebViewActivity.class);
            intent.putExtra("webViewUrl", splashscreenUrl);
            intent.putExtra("eventName", "splashscreen");
            intent.putExtra("eventData", splashscreenData);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(intent);
        } catch (NullPointerException e) {
            e.printStackTrace();

            GamedockSDK.getInstance(context).getSplashScreenCallbacks().splashScreenNotAvailable();
        }
    }

    public GamedockSplashscreen getDelayedSplashscreen() {
        return delayedSplashscreen;
    }

    public void setDelayedSplashscreen(GamedockSplashscreen delayedSplashscreen) {
        this.delayedSplashscreen = delayedSplashscreen;
    }

    public void resetSplashscreen() {
        delayedSplashscreen = null;
    }
}
