package io.gamedock.sdk.social;


import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.R;
import io.gamedock.sdk.events.internal.SocialLoginEvent;
import io.gamedock.sdk.models.social.SocialLogin;
import io.gamedock.sdk.network.NetworkCall;
import io.gamedock.sdk.utils.dialog.MaterialDialog;
import io.gamedock.sdk.utils.dialog.MaterialStyledDialog;
import io.gamedock.sdk.utils.dialog.internal.DialogAction;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Class that handles all the logic regarding the Spil Social Login as well as the errors that might occur.
 */
public class SocialManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "socialLogin";

    private static volatile SocialManager mInstance = null;
    private Context context;

    private boolean unauthorized;
    private String newUid;

    private String socialId;
    private String socialProvider;

    public static String Facebook = "facebook";
    public static String GooglePlayGames = "google_play_games";
    public static String GameCenter = "game_center";

    private SocialManager(Context context) {
        this.context = context;
    }

    public static SocialManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new SocialManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used internally to initiate the Social Login event.
     *
     * @param socialId       The social id received from the social network with which the user has logged in.
     * @param socialProvider The social provider (eg. Facebook, Google Play Games).
     * @param socialToken    The social token received from the social network after the user has logged in.
     */
    public void userLogin(String socialId, String socialProvider, String socialToken, String socialValidationData) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        SocialLoginEvent socialLoginEvent = new SocialLoginEvent(context);
        socialLoginEvent.setUserLogin();

        socialLoginEvent.addCustomData("socialId", socialId);
        socialLoginEvent.addCustomData("socialProvider", socialProvider.toLowerCase());
        socialLoginEvent.addCustomData("socialToken", socialToken);

        try {
            if(socialValidationData != null && !socialValidationData.equals("null")) {
                JSONObject socialValidationDataJSON = new JSONObject(socialValidationData);
                socialLoginEvent.addCustomData("socialValidationData", socialValidationDataJSON);
            }
        }
        catch (JSONException ex) {
            LoggingUtil.w("userLogin: " + ex.getMessage() + " Stack trace: ");
            ex.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(socialLoginEvent, null);
    }

    /**
     * Method used to process the response from the backend when a user first logged in.
     *
     * @param socialLoginJSON The JSON information sent by the backend.
     */
    public void processUserRegister(String socialLoginJSON) {
        if (socialLoginJSON == null) {
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
            return;
        }

        SocialLogin socialLogin = GamedockSDK.getInstance(context).getGson().fromJson(socialLoginJSON, SocialLogin.class);

        if (socialLogin.getUid() != null && socialLogin.getSpilToken() != null && socialLogin.getSocialProvider() != null && socialLogin.getSocialId() != null) {
            String spilToken = socialLogin.getSpilToken();
            String socialProvider = socialLogin.getSocialProvider();
            String socialId = socialLogin.getSocialId();

            GamedockSDK.getInstance(context).setGamedockToken(spilToken);
            this.socialId = socialId;
            this.socialProvider = socialProvider;
            GamedockSDK.getInstance(context).setUserId(socialProvider, socialId);

            unauthorized = false;
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginSuccessful(false, socialProvider, socialId, false);
        } else {
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
        }
    }

    /**
     * Method used to process the response from the backend when the user login and is already registered in the backend.
     *
     * @param socialLoginJSON The JSON information sent by the backend.
     */
    public void processUserLogin(String socialLoginJSON) {
        if (socialLoginJSON == null) {
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
            return;
        }

        SocialLogin socialLogin = GamedockSDK.getInstance(context).getGson().fromJson(socialLoginJSON, SocialLogin.class);

        if (socialLogin.getUid() != null && socialLogin.getSpilToken() != null && socialLogin.getSocialProvider() != null && socialLogin.getSocialId() != null) {
            String receivedUid = socialLogin.getUid();
            String spilToken = socialLogin.getSpilToken();
            String socialProvider = socialLogin.getSocialProvider();
            String socialId = socialLogin.getSocialId();

            boolean resetData = false;

            if (!UserIDGenerator.getGamedockUserId(context).equals(receivedUid)) {
                resetData = true;
                UserIDGenerator.manualUUIDGeneration(context, receivedUid);
            }

            GamedockSDK.getInstance(context).setGamedockToken(spilToken);
            this.socialId = socialId;
            this.socialProvider = socialProvider;
            GamedockSDK.getInstance(context).setUserId(socialProvider, socialId);

            unauthorized = false;
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginSuccessful(resetData, socialProvider, socialId, false);
        } else {
            GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
        }
    }

    /**
     * Method used to process the response from the backend if an error occurred during the login process.
     *
     * @param socialLoginJSON The JSON information sent by the backend.
     */
    public void processUserLoginError(String socialLoginJSON) {
        try {
            JSONObject userLoginError = new JSONObject(socialLoginJSON);

            if (userLoginError.has("authError")) {
                int authError = userLoginError.getInt("authError");

                switch (authError) {
                    case 1:
                        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.InvalidSpilTokenError);
                        break;
                    case 3:
                        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.InvalidSocialTokenError);
                        break;
                    case 4:
                        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.UserAlreadyLinkedError);
                        break;
                    case 5:
                        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.SocialIdAlreadyLinkedError);
                        break;
                    default:
                        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
                        break;
                }
            } else {
                GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
            }

            return;
        } catch (JSONException e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).setUserId(null, null);
        GamedockSDK.getInstance(context).getSocialCallbacks().LoginFailed(ErrorCodes.GamedockLoginServerError);
    }

    /**
     * Method used internally to initiate the Social Logout event.
     *
     * @param global The flag that informs the backend if a global logout should be done.
     */
    public void userLogout(boolean global) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        SocialLoginEvent socialLoginEvent = new SocialLoginEvent(context);
        socialLoginEvent.setUserLogout();

        newUid = UUID.randomUUID().toString();

        socialLoginEvent.addCustomData("global", global);
        socialLoginEvent.addCustomData("newUid", newUid);

        GamedockSDK.getInstance(context).trackEvent(socialLoginEvent, null);

        if (!global) {
            GamedockSDK.getInstance(context).setUserId(null, null);
            UserIDGenerator.manualUUIDGeneration(context, newUid);
            newUid = null;
            GamedockSDK.getInstance(context).setGamedockToken(null);
            GamedockSDK.getInstance(context).getSocialCallbacks().LogoutSuccessful();
        }
    }

    /**
     * Method used to process the response in case of a global logout.
     */
    public void processUserLogout() {
        GamedockSDK.getInstance(context).setUserId(null, null);
        UserIDGenerator.manualUUIDGeneration(context, newUid);
        newUid = null;
        GamedockSDK.getInstance(context).setGamedockToken(null);
        GamedockSDK.getInstance(context).getSocialCallbacks().LogoutSuccessful();
    }

    /**
     * Method used internally to inform the backend that the user has chosen to play as a guest.
     */
    public void userPlayAsGuest() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        SocialLoginEvent socialLoginEvent = new SocialLoginEvent(context);
        socialLoginEvent.setUserPlayAsGuest();

        String newUid = UUID.randomUUID().toString();
        socialLoginEvent.addCustomData("newUid", newUid);

        GamedockSDK.getInstance(context).trackEvent(socialLoginEvent, null);

        UserIDGenerator.manualUUIDGeneration(context, newUid);

        unauthorized = false;
        GamedockSDK.getInstance(context).setGamedockToken(null);
        GamedockSDK.getInstance(context).getSocialCallbacks().LoginSuccessful(true, null, null, true);
    }

    /**
     * Method used to process the unauthorized 401 responses from the server.
     * Called from the {@link NetworkCall} class when response code is 401.
     *
     * @param errorJSONString The JSON error information received from the backend.
     */
    public synchronized void processUnauthorizedResponse(String errorJSONString) {
        if (unauthorized || !GamedockSDK.getInstance(context).isAppRunning()) {
            return;
        }

        unauthorized = true;
        try {
            JSONObject errorJSON = new JSONObject(errorJSONString);

            if (errorJSON.has("authError")) {
                int authError = errorJSON.getInt("authError");

                switch (authError) {
                    case 1:
                        GamedockSDK.getInstance(context).getSocialCallbacks().AuthenticationError(ErrorCodes.InvalidSpilTokenError);
                        break;
                    case 2:
                        GamedockSDK.getInstance(context).getSocialCallbacks().AuthenticationError(ErrorCodes.RequiresLoginError);
                        break;
                    default:
                        GamedockSDK.getInstance(context).getSocialCallbacks().AuthenticationError(ErrorCodes.GamedockLoginServerError);
                        break;
                }
            } else {
                GamedockSDK.getInstance(context).getSocialCallbacks().AuthenticationError(ErrorCodes.GamedockLoginServerError);
            }

            return;
        } catch (JSONException e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).getSocialCallbacks().AuthenticationError(ErrorCodes.GamedockLoginServerError);
    }

    /**
     * Method used to show the unauthorized dialog in which the user selects if he wants to re-login or to play as guest.
     *
     * @param title           The title of the dialog.
     * @param message         The message to be displayed by the dialog.
     * @param loginText       The text for the login button.
     * @param playAsGuestText The text for the play as guest button.
     */
    public void showUnauthorizedDialog(final String title, final String message, final String loginText, final String playAsGuestText) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        final int dialogHeader;
        int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

        if (resourceId != 0) {
            dialogHeader = resourceId;
        } else {
            dialogHeader = R.drawable.permission_header;
        }

        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(dialogHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(loginText)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                GamedockSDK.getInstance(context).getSocialCallbacks().RequestLogin();
                                materialDialog.dismiss();
                            }
                        })
                        .setNeutralText(playAsGuestText)
                        .onNeutral(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction which) {
                                userPlayAsGuest();
                                materialDialog.dismiss();
                            }
                        });

                builder.show();
            }
        });
    }

    public String getSocialId() {
        return socialId;
    }

    public void setSocialId(String socialId) {
        this.socialId = socialId;
    }

    public String getSocialProvider() {
        return socialProvider;
    }

    public void setSocialProvider(String socialProvider) {
        this.socialProvider = socialProvider;
    }

    public void resetSocialManager() {
        mInstance = null;
    }
}
