package io.gamedock.sdk.social;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Social Login Callbacks.
 * Notifies the application when the Social Login operations have been done.
 * Informs the application if an error occurred while processing the Social Login.
 */
public class SocialCallbacks {

    private OnSocialListener socialListener;

    public SocialCallbacks() {
        socialListener = null;
    }

    public SocialCallbacks(OnSocialListener socialListener) {
        this.socialListener = socialListener;
    }

    /**
     * Method called when the Social Login was successful.
     *
     * @param resetData      Flag that tells if the developer should reset the data and restart the game.
     * @param socialProvider The social provider with which the user has logged in.
     * @param socialId       The social id of the user.
     * @param isGuest        Flag that tells if the user logged in as a guest.
     */
    public void LoginSuccessful(boolean resetData, String socialProvider, String socialId, boolean isGuest) {
        if (socialListener != null) {
            socialListener.OnLoginSuccessful(resetData, socialProvider, socialId, isGuest);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("resetData", resetData);
                jsonObject.put("socialProvider", socialProvider);
                jsonObject.put("socialId", socialId);
                jsonObject.put("isGuest", isGuest);

                UnityPlayer.UnitySendMessage("GamedockSDK", "LoginSuccessful", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when an error occurred during the login process.
     *
     * @param error Uses the {@link ErrorCodes} class do define the error.
     */
    public void LoginFailed(ErrorCodes error) {
        if (socialListener != null) {
            socialListener.OnLoginFailed(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "LoginFailed", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the login process is requested by the user through the unauthorized dialog.
     */
    public void RequestLogin() {
        if (socialListener != null) {
            socialListener.OnRequestLogin();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "RequestLogin", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the logout process was successful.
     */
    public void LogoutSuccessful() {
        if (socialListener != null) {
            socialListener.OnLogoutSuccessful();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "LogoutSuccessful", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when an error occurred during the logout procedure.
     *
     * @param error Uses the {@link ErrorCodes} class do define the error.
     */
    public void LogoutFailed(ErrorCodes error) {
        if (socialListener != null) {
            socialListener.OnLogoutFailed(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "LogoutFailed", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when a 401 unauthorized error has occurred.
     *
     * @param error Uses the {@link ErrorCodes} class do define the error.
     */
    public void AuthenticationError(ErrorCodes error) {
        if (socialListener != null) {
            socialListener.OnAuthError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "AuthenticationError", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the SocialCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }
}
