package io.gamedock.sdk.reward;


import android.content.Context;

import com.google.gson.reflect.TypeToken;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.RewardEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.reward.Reward;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.error.ErrorCodes;

public class RewardManager {

    private static final String DeepLink = "deeplink";
    private static final String PushNotification = "pushnotification";

    /**
     * Method that sends an event requesting a claim of a reward with a given token.
     *
     * @param context    The activity context.
     * @param token      The token associated with the reward. Can only be used once.
     * @param rewardType The type of reward that will be requested from Gamedock.
     */
    public static void claimToken(Context context, String token, String rewardType) {
        RewardEvent rewardEvent = new RewardEvent(context);

        rewardEvent.setClaimTokenEvent();
        rewardEvent.setRewardType(rewardType);

        rewardEvent.addCustomData("token", token);

        GamedockSDK.getInstance(context).trackEvent(rewardEvent, null);
    }

    /**
     * Method that processes the reward received from the Gamedock backend.
     *
     * @param context    The activity context.
     * @param rewardList The list of rewards that has been received from Gamedock backend.
     * @param success    Information if the reward was successfully claimed or not.
     * @param rewardType The type of reward that was awarded from Gamedock backend.
     * @param error      The error message in case the reward was not received from Gamedock backend.
     */
    public static void processReward(Context context, ArrayList<Reward> rewardList, boolean success, String rewardType, String error) {
        if (!success) {
            GamedockSDK.getInstance(context).getRewardCallbacks().rewardTokenClaimFailed(rewardType, ErrorCodes.RewardServerError.setMessage(error));
            return;
        }

        String reason = "";

        if (rewardType.toLowerCase().trim().equals(DeepLink)) {
            reason = PlayerDataUpdateReasons.Deeplink;
        } else if (rewardType.toLowerCase().trim().equals(PushNotification)) {
            reason = PlayerDataUpdateReasons.PushNotification;
        }

        ArrayList<Reward> externalReward = null;

        if (rewardList != null) {
            for (int i = 0; i < rewardList.size(); i++) {
                switch (rewardList.get(i).type) {
                    case "CURRENCY":
                        GamedockSDK.getInstance(context).addCurrencyToWallet(rewardList.get(i).id, rewardList.get(i).amount, reason, null, "sdk", null, false, null);
                        break;
                    case "ITEM":
                        Item gameItem = GamedockGameDataManager.getInstance(context).getItem(rewardList.get(i).id);

                        if (gameItem == null) {
                            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemNotFound);
                            return;
                        }

                        if (gameItem.isUnique()) {
                            UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
                            uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
                            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0, 0,0, reason);
                        } else {
                            GamedockSDK.getInstance(context).addItemToInventory(rewardList.get(i).id, rewardList.get(i).amount, reason, null, "sdk", null, false, null);
                        }
                        break;
                    case "EXTERNAL":
                        if (externalReward == null) {
                            externalReward = new ArrayList<>();
                        }
                        externalReward.add(rewardList.get(i));
                        break;
                }
            }

            if (externalReward != null && !externalReward.isEmpty()) {
                sendRewardsToExternal(context, externalReward, rewardType);
            }
        }
    }

    /**
     * Method that send the reward to the external platform (Unity/AIR) if the rewards have the EXTERNAL tag.
     *
     * @param context    The The activity context.
     * @param rewardList The list of rewards that has been received from the backend.
     */
    public static void sendRewardsToExternal(Context context, ArrayList<Reward> rewardList, String rewardType) {
        String listString = GamedockSDK.getInstance(context).getGson().toJson(rewardList, new TypeToken<ArrayList<Reward>>() {
        }.getType());
        JSONArray reward = null;
        try {
            reward = new JSONArray(listString);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        GamedockSDK.getInstance(context).getRewardCallbacks().rewardTokenClaimed(reward, rewardType);
    }
}
