package io.gamedock.sdk.network;

import android.content.Context;

import androidx.annotation.NonNull;

import com.evernote.android.job.Job;
import com.evernote.android.job.JobRequest;

import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.EventActionListener;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.internal.HeartbeatEvent;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Task Job Class that adds a job to the background sending pipeline.
 * The job has a deadline of 1 week before being discarded.
 */
public class NetworkJob {

    /**
     * Method that creates a job that will be added to the Job Scheduler in order to be executed when there is internet connection
     *
     * @param context        The activity context.
     * @param event          The {@link Event} object that needs to be sent to the backend.
     * @param actionListener The {@link EventActionListener} which handles the response if it is not {@code null}.
     */
    public void addJob(final Context context, Event event, EventActionListener actionListener) {
        boolean internetAvailable = NetworkUtil.isInternetAvailable(context);
        //check if network connection is available
        if (internetAvailable) {
            try {
                //Send network call
                NetworkApi senderApi = new NetworkApi(context, event, actionListener);
                senderApi.sendEvent().subscribeOn(Schedulers.computation())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(senderApi.dataSenderObserver);
            } catch (InternalError e) {
                e.printStackTrace();
            }

            //flush persistence list if events are still inside it
            for (int i = 0; i < EventManager.getInstance(context).getPersistenceListSize(); i++) {
                if (!EventManager.getInstance(context).getPersistenceEventsList(i + 1).isEmpty()) {
                    synchronized (EventManager.getInstance(context).getPersistenceEventsList(i + 1)) {
                        LoggingUtil.d("Flushing persistence event list!");

                        ArrayList<Event> temp = new ArrayList<>(EventManager.getInstance(context).getPersistenceEventsList(i + 1));

                        EventManager.getInstance(context).clearPersistenceEventDataList(EventManager.getInstance(context).getPersistenceEventsList(i + 1), i + 1);
                        for (int j = 0; j < temp.size(); j++) {
                            //Send event
                            NetworkApi senderApi = new NetworkApi(context, temp.get(j), null);
                            senderApi.sendEvent().subscribeOn(Schedulers.computation())
                                    .observeOn(AndroidSchedulers.mainThread())
                                    .subscribe(senderApi.dataSenderObserver);
                        }
                    }
                }
            }
        } else if (!internetAvailable && !(event instanceof HeartbeatEvent)) {
            event.setQueued(true);

            boolean check = false;
            int updateGameStateCount = 0;

            for (int i = 0; i < EventManager.getInstance(context).getPersistenceListSize(); i++) {
                ArrayList<Event> persistenceList = EventManager.getInstance(context).getPersistenceEventsList(i + 1);
                for (int j = 0; j < persistenceList.size(); j++) {
                    if ((event.getTimestamp() == persistenceList.get(j).getTimestamp() &&
                            (event.isQueued() == persistenceList.get(j).isQueued()) &&
                            (event.getName().equals(persistenceList.get(j).getName())))) {
                        check = true;
                        LoggingUtil.d("Duplicate Event detected before scheduling: " + event.toString());
                        break;
                    }

                    if (persistenceList.get(j).getName().equals("updateGameState") && event.getName().equals("updateGameState")) {
                        updateGameStateCount++;
                        if (updateGameStateCount > 1) {
                            persistenceList.remove(j);
                        }
                    }


                }
                EventManager.getInstance(context).savePersistenceEventsList(persistenceList, i + 1);
            }

            //check if event is in recent events list
            for (int i = 0; i < EventManager.getInstance(context).getRecentSentEvents().list.size(); i++) {
                if ((event.getTimestamp() == EventManager.getInstance(context).getRecentSentEvents().list.get(i).getTimestamp() &&
                        (event.isQueued() == EventManager.getInstance(context).getRecentSentEvents().list.get(i).isQueued()) &&
                        (event.getName().equals(EventManager.getInstance(context).getRecentSentEvents().list.get(i).getName())))) {
                    check = true;
                    LoggingUtil.d("Duplicate Event detected in recent events: " + event.toString());
                }
            }

            if (!check) {
                //add event to persistence
                EventManager.getInstance(context).addToPersistenceEventDataList(event);

                event.handleNetworkError(context);

                if (EventManager.getInstance(context).getPersistenceEventsList(EventManager.getInstance(context).getPersistenceListSize()).size() < 2) {
                    PersistJob.scheduleJob();
                }
            }
        }
    }

    /**
     * Inner class that handles the creation of a Job for sending Events to the Gamedock backend.
     */
    public static class PersistJob extends Job {
        private Integer size;
        public static final String TAG = "gamedockEvents";

        @NonNull
        @Override
        protected Result onRunJob(@NonNull Params params) {
            try {
                synchronized (size = EventManager.getInstance(getContext()).getPersistenceListSize()) {
                    for (int i = 0; i < size; i++) {
                        ArrayList<Event> persistenceList = EventManager.getInstance(getContext()).getPersistenceEventsList(i + 1);
                        if (persistenceList.isEmpty()) {
                            continue;
                        }

                        ArrayList<Event> temp = new ArrayList<>(persistenceList);

                        EventManager.getInstance(getContext()).clearPersistenceEventDataList(persistenceList, i + 1);
                        CountDownLatch countDownLatch = new CountDownLatch(temp.size());
                        for (int j = 0; j < temp.size(); j++) {
                            final NetworkApi senderApi = new NetworkApi(getContext(), temp.get(j), null);
                            senderApi.setCountDownLatch(countDownLatch);
                            //Send
                            senderApi.sendEvent().subscribeOn(Schedulers.computation())
                                    .observeOn(AndroidSchedulers.mainThread())
                                    .subscribe(senderApi.dataSenderObserver);
                        }

                        try {
                            countDownLatch.await(1, TimeUnit.MINUTES);
                        } catch (InterruptedException ignored) {
                        }
                    }
                    return Result.SUCCESS;
                }
            } catch (RuntimeException e) {
                return Result.FAILURE;
            }
        }

        public static void scheduleJob() {
            try {
                new JobRequest.Builder(PersistJob.TAG)
                        .setExecutionWindow(10_000L, 10000_000L)
                        .setRequiredNetworkType(JobRequest.NetworkType.CONNECTED)
                        .setRequirementsEnforced(true)
                        .build()
                        .schedule();
            } catch (Exception e) {
                LoggingUtil.e("Job could not be scheduled.");
            }
        }
    }
}