package io.gamedock.sdk.network;

import android.content.Context;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.Locale;
import java.util.Scanner;

import javax.net.ssl.HttpsURLConnection;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.encryption.message.MessagePH;
import io.gamedock.sdk.social.SocialManager;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.other.GZIPCompression;

/**
 * Class that handles the sending and receiving of events and their responses from the Gamedock Backend.
 */

public class NetworkCall {
    /**
     * Method that handles the sending and response of the messages.
     *
     * @return The response from the backend.
     */
    static String makeNetworkCall(Context context, NetworkApi api) {
        String response = "";
        HttpsURLConnection connection = null;
        String errorResponse = null;
        int responseCode = 0;
        try {

            URL url = new URL(api.getApiUrl());
            connection = (HttpsURLConnection) url.openConnection();
            connection.setRequestMethod("POST");
            connection.setReadTimeout(8000);
            connection.setConnectTimeout(4000);
            connection.setDoInput(true);
            connection.setDoOutput(true);

            String payload = api.getEventPayload();

            String ph = MessagePH.generate(payload, GamedockSDK.getInstance(context).getApplicationPackageName(), GamedockSDK.getInstance(context).getGamedockUID());

            connection.setRequestProperty("Ph", ph);

            if (GamedockConfigManager.getInstance(context).spil.gzipCompression && payload.getBytes("UTF-8").length > GamedockConfigManager.getInstance(context).spil.compressionThreshold) {
                LoggingUtil.d("Compression used for event.");

                connection.setRequestProperty("Content-Encoding", "gzip");

                BufferedOutputStream outputStream = new BufferedOutputStream(connection.getOutputStream());
                outputStream.write(GZIPCompression.compress(payload));
                outputStream.flush();
                outputStream.close();
            } else {
                OutputStream outputStream = connection.getOutputStream();

                BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream, "UTF-8"));
                writer.write(payload);
                writer.flush();
                writer.close();
                outputStream.close();
            }

            responseCode = connection.getResponseCode();

            if (responseCode == HttpsURLConnection.HTTP_OK) {
                response = readInputStreamToString(connection);

                if (GamedockSDK.getInstance(context).isUnitTesting()) {
                    response = readMockResponse(context, response, api.getEvent().getName());
                }

                if (connection.getHeaderField("Ph") != null) {
                    String sph = connection.getHeaderField("Ph");
                    boolean verified = MessagePH.verify(response, sph);
                    if (!verified) {
                        throw new GamedockHttpException("Server security failed. Failed verification");
                    }
                } else {
                    throw new GamedockHttpException("Server security failed. No information was sent from the server");
                }

                try {
                    if (connection.getHeaderField("x-spil-segments") != null) {
                        String segmentData = ", \"segments\":\"" + connection.getHeaderField("x-spil-segments") + "\"";
                        response = new StringBuffer(response).insert(response.length() - 1, segmentData).toString();
                    }

                    if (connection.getHeaderField("x-spil-data") != null) {
                        String service = ", \"serviceData\":\"" + connection.getHeaderField("x-spil-data") + "\"";
                        response = new StringBuffer(response).insert(response.length() - 1, service).toString();
                    }
                } catch (OutOfMemoryError | Exception ignore) {
                }
            } else {
                InputStream stream = connection.getErrorStream();
                Scanner scanner = new Scanner(stream);
                scanner.useDelimiter("\\Z");
                errorResponse = scanner.next();

                if (responseCode == HttpsURLConnection.HTTP_UNAUTHORIZED) {
                    LoggingUtil.w("Response error: " + errorResponse);
                    SocialManager.getInstance(context).processUnauthorizedResponse(errorResponse);
                    return response;
                } else if (responseCode == HttpURLConnection.HTTP_BAD_GATEWAY
                        || responseCode == HttpURLConnection.HTTP_UNAVAILABLE
                        || responseCode == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
                    throw new ConnectException("\n" + errorResponse);
                } else if (responseCode == HttpsURLConnection.HTTP_GONE) {
                    GamedockSDK.getInstance(context).setGameUnpublished(true);
                    throw new GamedockHttpException("\n" + errorResponse);
                } else {
                    throw new GamedockHttpException("\n" + errorResponse);
                }
            }
        } catch (UnknownHostException | SocketTimeoutException | SocketException e) {
            //If the connection is too slow and the event has not already been queued create a new queued event

            if (e instanceof SocketTimeoutException) {
                api.getEvent().setRetryReason("sockettimeout");
            } else if (e instanceof ConnectException) {
                api.getEvent().setRetryReason(String.format(Locale.ENGLISH, "http_%d", responseCode));
            } else if (e instanceof UnknownHostException) {
                api.getEvent().setRetryReason("unknownhostexception");
            } else {
                api.getEvent().setRetryReason("socketexception");
            }

            //Add to recent events
            api.addToRecentEvents();

            if (!api.getEvent().isQueued()) {
                api.getEvent().handleNetworkError(context);
                LoggingUtil.d("Event will be queued due to slow connection: " + api.getEvent().toString());
                GamedockSDK.getInstance(context).trackEventQueued(api.getEvent(), api.getActionListener());
            }
        } catch (SecurityException | IOException | GamedockHttpException | ArrayIndexOutOfBoundsException e) {
            LoggingUtil.w("Response Code: " + errorResponse + "\nStacktrace: ");
            e.printStackTrace();
            api.getEvent().handleNetworkError(context);
        } catch (NullPointerException e) {
            e.printStackTrace();
            api.getEvent().handleNetworkError(context);
        } finally {
            try {
                assert connection != null;
                connection.disconnect();
            } catch (Exception e) {
                e.printStackTrace(); //If you want further info on failure...
            }
        }
        return response;
    }

    private static class GamedockHttpException extends Exception {
        GamedockHttpException(String s) {
            super(s);
        }
    }

    /**
     * Method that reads the response from the backend and converts it into a {@link String}
     *
     * @param connection The HTTP connection.
     * @return Returns the response from the backend.
     */
    private static String readInputStreamToString(HttpURLConnection connection) {
        String response = null;
        StringBuilder sb = new StringBuilder();
        InputStream is = null;

        try {
            is = new BufferedInputStream(connection.getInputStream());
            BufferedReader br = new BufferedReader(new InputStreamReader(is));
            String inputLine;
            while ((inputLine = br.readLine()) != null) {
                sb.append(inputLine);
            }
            response = sb.toString();
        } catch (Exception e) {
            LoggingUtil.i("Error reading InputStream");
            response = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    LoggingUtil.i("Error closing InputStream");
                }
            }
        }
        return response;
    }

    private static String readMockResponse(Context context, String response, String eventName) {
        String mockResponse = null;

        try {
            InputStream is = context.getAssets().open("mock/" + eventName + ".json");
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            mockResponse = new String(buffer, "UTF-8");

        } catch (IOException ex) {
            LoggingUtil.w("No mock response was found. Using live data retrieved from the server.");
        }

        if (mockResponse != null) {
            return mockResponse;
        } else {
            return response;
        }
    }
}
