package io.gamedock.sdk.mission;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Mission Configuration Callbacks.
 * Notifies the application when the Mission Configuration has been received and processed by the SDK.
 * Informs the application if an error occurred while processing the Mission Configuration values.
 */
public class MissionConfigurationCallbacks {
    private OnMissionConfigurationListener missionConfigurationListener;

    public MissionConfigurationCallbacks() {
        this.missionConfigurationListener = null;
    }

    public MissionConfigurationCallbacks(OnMissionConfigurationListener missionConfigurationListener) {
        this.missionConfigurationListener = missionConfigurationListener;
    }

    /**
     * Method that is called when the mission configuration has been process and is available to be retrieved.
     */
    public void missionConfigurationAvailable(String containersJSON, String missionJSON) {
        if (this.missionConfigurationListener != null) {
            this.missionConfigurationListener.MissionConfigurationAvailable(containersJSON, missionJSON);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("containers", containersJSON);
                jsonObject.put("missions", missionJSON);

                UnityPlayer.UnitySendMessage("GamedockSDK", "MissionConfigurationAvailable", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the MissionConfigurationCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when the mission configuration is not available (list is empty).
     */
    public void missionConfigurationNotAvailable() {
        if (this.missionConfigurationListener != null) {
            this.missionConfigurationListener.MissionConfigurationNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "MissionConfigurationNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the MissionConfigurationCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while processing the Mission Configuration.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void missionConfigurationError(ErrorCodes error) {
        if (this.missionConfigurationListener != null) {
            this.missionConfigurationListener.MissionConfigurationError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "MissionConfigurationError", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the MissionConfigurationCallbacks in order to receive information or something went wrong with Unity");
            }
        }
    }
}
