package io.gamedock.sdk.localization;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class LocalizationCallbacks {

    private OnLocalizationListener localizationListener;

    public LocalizationCallbacks() {
        localizationListener = null;
    }

    public LocalizationCallbacks(OnLocalizationListener localizationListener) {
        this.localizationListener = localizationListener;
    }

    /**
     * Method that is called when the localization is available.
     */
    public void localizationAvailable(String locale, boolean isDefaultLocale) {
        if (localizationListener != null) {
            localizationListener.LocalizationAvailable(locale, isDefaultLocale);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("locale", locale);
                jsonObject.put("isDefaultLocale", isDefaultLocale);

                UnityPlayer.UnitySendMessage("GamedockSDK", "LocalizationAvailable", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when the localization is not available.
     */
    public void localizationNotAvailable() {
        if (localizationListener != null) {
            localizationListener.LocalizationNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "LocalizationNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called if an error has occured while trying to retrieve the game configuration.
     *
     * @param error The error title and error message describing the problem.
     */
    public void localizationError(ErrorCodes error) {
        if (localizationListener != null) {
            localizationListener.LocalizationError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "LocalizationError", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }
}
