package io.gamedock.sdk.initialization;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;

import com.google.gson.reflect.TypeToken;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.internal.InitializeSDKEvent;
import io.gamedock.sdk.models.feature.Feature;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Class that handles sdk initialisation operations from the backend.
 */
public class InitializationManager {

    private static JSONObject uidChange;

    /**
     * Method that send a initialisation sdk event towards the backend.
     *
     * @param context The Activity context.
     */
    public static void initializeSDK(Context context) {
        if (!NetworkUtil.isInternetAvailable(context)) {
            GamedockSDK.getInstance(context).initFeatures();
        }

        InitializeSDKEvent initializeSDKEvent = new InitializeSDKEvent(context);
        initializeSDKEvent.setInitializeSDK();

        GamedockSDK.getInstance(context).trackEvent(initializeSDKEvent, null);
    }

    /**
     * Method used to process the user id change request from the backend.
     *
     * @param context       The Activity context.
     * @param uidChangeJSON The information sent from the backend.
     */
    public static void processUidChange(final Context context, JSONObject uidChangeJSON) {
        uidChange = uidChangeJSON;

        if (uidChange.has("targetUid")) {
            Handler handler = new Handler(Looper.getMainLooper());
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    try {
                        String newUid = uidChange.getString("targetUid");

                        GamedockSDK.getInstance(context).getInitializationDataCallbacks().userIdChangeRequest(newUid);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
            }, 10000);
        }
    }

    /**
     * Method used to process the confirmation sent from the game regarding user id change.
     *
     * @param context The Activity context.
     */
    public static void confirmUserIdChange(Context context) {
        if (uidChange == null) {
            return;
        }

        try {
            String newUid = uidChange.getString("targetUid");
            String oldUid = GamedockSDK.getInstance(context).getGamedockUID();

            UserIDGenerator.manualUUIDGeneration(context, newUid);
            GamedockSDK.getInstance(context).resetData();

            InitializeSDKEvent userChangedEvent = new InitializeSDKEvent(context);
            userChangedEvent.setUserChanged();

            userChangedEvent.addCustomData("oldUid", oldUid);
            userChangedEvent.addCustomData("newUid", newUid);

            GamedockSDK.getInstance(context).trackEvent(userChangedEvent, null);

            GamedockSDK.getInstance(context).getInitializationDataCallbacks().userIdChangeCompleted();

            uidChange = null;
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to process the game version update response.
     *
     * @param context           The Activity context.
     * @param gameVersionStatus The game version update information.
     */
    public static void processGameVersionStatus(final Context context, final JSONObject gameVersionStatus) {
        if (gameVersionStatus.has("update")) {
            try {
                String newStatus = gameVersionStatus.getString("update");

                GamedockSDK.getInstance(context).getInitializationDataCallbacks().gameVersionStatus(newStatus);
            } catch (JSONException e) {
                e.printStackTrace();
            }

        }
    }

    public static void processFeaturesInitialisation(Context context, JSONArray featuresJSON) {
        for (int i = 0; i < featuresJSON.length(); i++) {
            try {
                Feature feature = GamedockSDK.getInstance(context).getGson().fromJson(featuresJSON.getJSONObject(i).toString(), Feature.class);
                FeaturesUtil.getFeatures().put(feature.getName(), feature);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        String featuresToSave = GamedockSDK.getInstance(context).getGson().toJson(FeaturesUtil.getFeatures());
        GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.FeaturesInfo, featuresToSave);
    }

    public static void processWhitelistedEvents(Context context, JSONArray whitelistedEventsJSON) {
        Type listType = new TypeToken<ArrayList<String>>(){}.getType();
        ArrayList<String> whitelistedEvents = GamedockSDK.getInstance(context).getGson().fromJson(whitelistedEventsJSON.toString(), listType);

        EventManager.getInstance(context).getWhitelistedEvents().addAll(whitelistedEvents);
    }
}
