package io.gamedock.sdk.gamedata.promotions;


import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.ArrayList;
import java.util.HashMap;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.PromotionEvent;
import io.gamedock.sdk.gamedata.packages.PackageManager;
import io.gamedock.sdk.models.gamedata.packages.Package;
import io.gamedock.sdk.models.gamedata.promotion.AffectedEntity;
import io.gamedock.sdk.models.gamedata.promotion.Promotion;
import io.gamedock.sdk.models.gamedata.shop.Entry;
import io.gamedock.sdk.models.gamedata.shop.Tab;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.web.WebViewActivity;

/**
 * Class that handles all the Promotion configuration logic.
 */
public class PromotionsManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "promotions";

    public static final String Id = "id";
    public static final String Name = "name";

    private static volatile PromotionsManager mInstance = null;
    private Context context;

    private Gson gson;
    private StorageUtil storageUtil;

    private HashMap<Integer, Promotion> promotions = new HashMap<>();

    private PromotionsManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
        storageUtil = GamedockSDK.getInstance(context).getStorageUtil();

        String promotionsJSON = storageUtil.getString(StorageUtil.Keys.SpilPromotions, "{}");
        promotions = gson.fromJson(promotionsJSON, new TypeToken<HashMap<Integer, Promotion>>() {
        }.getType());
    }

    public static PromotionsManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new PromotionsManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to send a request towards the server to retrieve the Promotions configuration.
     * If no internet is available, it notifies the game if any locally stored configurations are available.
     */
    public void requestPromotions() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        PromotionEvent promotionEvent = new PromotionEvent(context);
        promotionEvent.setRequestPromotions();

        GamedockSDK.getInstance(context).trackEvent(promotionEvent, null);

        if (!NetworkUtil.isInternetAvailable(context)) {
            if (!promotions.isEmpty()) {
                GamedockSDK.getInstance(context).getPromotionsCallbacks().promotionsAvailable(getPromotions());
            } else {
                GamedockSDK.getInstance(context).getPromotionsCallbacks().promotionsNotAvailable();
            }
        }
    }

    /**
     * Method used to process the Promotions configuration response from the backend.
     *
     * @param receivedPromotions The list of Promotions sent from the backend.
     */
    public void processPromotions(HashMap<Integer, Promotion> receivedPromotions) {
        if (receivedPromotions == null) {
            return;
        }

        promotions = receivedPromotions;
        storageUtil.putString(StorageUtil.Keys.SpilPromotions, gson.toJson(promotions));

        GamedockSDK.getInstance(context).getPromotionsCallbacks().promotionsAvailable(getPromotions());
    }

    /**
     * Method used to send an event requesting the server to send a splash screen for a specific promotion.
     *
     * @param promotionId The id of the promotion for which the splash screen needs to be shown.
     */
    public void showPromotionScreen(int promotionId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        PromotionEvent promotionEvent = new PromotionEvent(context);
        promotionEvent.setShowPromotionScreen();

        promotionEvent.addCustomData(Id, promotionId);

        GamedockSDK.getInstance(context).trackEvent(promotionEvent, null);
    }

    /**
     * Method that processes the bought promotion base on a bundle id.
     *
     * @param bundleId The id of the bundle for which the promotion was applied.
     */
    public void sendBoughtPromotion(int bundleId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        int promotionId = 0;

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity affectedEntity : promotion.getAffectedEntities()) {
                if (affectedEntity.getId() == bundleId) {
                    promotionId = promotion.getId();
                }
            }
        }

        if (promotionId == 0) {
            LoggingUtil.d("Bundle with id: " + bundleId + " not found in promotions. Bought Promotion event will not be sent");
            return;
        }

        sendBoughtPromotionEvent(promotionId);
    }

    /**
     * Method that processes the bought promotion base on a package id.
     *
     * @param packageId The id of the package for which the promotion was applied.
     */
    public void sendBoughtPromotion(String packageId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        int promotionId = 0;

        Package packageEntry = PackageManager.getInstance(context).getPackageObject(packageId);

        if (packageEntry == null) {
            LoggingUtil.d("Package with id: " + packageId + " not found. Bought Promotion event will not be sent");
            return;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity affectedEntity : promotion.getAffectedEntities()) {
                if (affectedEntity.getId() == packageEntry.getId()) {
                    promotionId = promotion.getId();
                }
            }
        }

        if (promotionId == 0) {
            LoggingUtil.d("Package with id: " + packageId + " not found in promotions. Bought Promotion event will not be sent");
            return;
        }

        sendBoughtPromotionEvent(promotionId);
    }

    /**
     * Method used to send the event which notifies the backend a certain promotion has been bought, either for a bundle or a package.
     *
     * @param promotionId The id of the promotion that was bought.
     */
    private void sendBoughtPromotionEvent(int promotionId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        PromotionEvent promotionEvent = new PromotionEvent(context);
        promotionEvent.setBoughtPromotion();

        promotionEvent.addCustomData(Id, promotionId);
        promotionEvent.addCustomData(Name, promotions.get(promotionId).getName());

        GamedockSDK.getInstance(context).trackEvent(promotionEvent, null);

        updatePromotions();
    }

    /**
     * Method used to process the response of the bought promotion event from the backend.
     *
     * @param promotion The promotion that was bought.
     */
    public void processBoughtPromotionResponse(Promotion promotion) {
        if (promotion == null) {
            return;
        }
        promotions.get(promotion.getId()).setAmountPurchased(promotion.getAmountPurchased());

        boolean maxReached = false;
        if (promotion.getAmountPurchased() == promotion.getMaxPurchase() || promotion.getAmountPurchased() > promotion.getMaxPurchase()) {
            //return callback
            maxReached = true;
            promotions.remove(promotion.getId());
        }

        updatePromotions();
        GamedockSDK.getInstance(context).getPromotionsCallbacks().promotionAmountBought(promotion.getId(), promotion.getAmountPurchased(), maxReached);
    }

    /**
     * Method used to retrieve a bundle Promotion object based on the bundle id.
     *
     * @param bundleId The id of the bundle associated with the promotion.
     * @return The Promotion object.
     */
    public Promotion getBundlePromotionObject(int bundleId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity entity : promotion.getAffectedEntities()) {
                if (entity.getType().equals(PlayerDataManager.BundleCheck) && entity.getId() == bundleId) {
                    return promotion;
                }
            }
        }

        return null;
    }

    /**
     * Method used to retrieve a bundle Promotion JSON string based on the bundle id.
     *
     * @param bundleId The id of the bundle associated with the promotion.
     * @return The Promotion JSON string.
     */
    public String getBundlePromotion(int bundleId) {
        return gson.toJson(getBundlePromotionObject(bundleId));
    }

    /**
     * Method used to send the Promotion information to a splash screen.
     *
     * @param bundleId The id of the bundle associated with the promotion.
     */
    public void getBundlePromotionForWeb(int bundleId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (WebViewActivity.getActivity() == null) {
            LoggingUtil.d("No Splash Screen active. No message will be sent.");
            return;
        }

        WebViewActivity.getActivity().javascriptBridge.nativeMessage("getBundlePromotion", getBundlePromotion(bundleId));
    }

    /**
     * Method used to retrieve a bundle Promotion object based on the package id.
     *
     * @param packageId The id of the package associated with the promotion.
     * @return The Promotion object.
     */
    public Promotion getPackagePromotionObject(String packageId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity entity : promotion.getAffectedEntities()) {
                if (entity.getType().equals(PlayerDataManager.PackageCheck)) {
                    Package packageEntry = PackageManager.getInstance(context).getPackageObject(entity.getId());

                    if (packageEntry != null && packageEntry.getPackageId().equals(packageId)) {
                        return promotion;
                    }
                }
            }
        }

        return null;
    }

    /**
     * Method used to retrieve a bundle Promotion JSON string based on the package id.
     *
     * @param packageId The id of the package associated with the promotion.
     * @return The Promotion JSON string.
     */
    public String getPackagePromotion(String packageId) {
        return gson.toJson(getPackagePromotionObject(packageId));
    }

    /**
     * Method used to send the Promotion information to a splash screen.
     *
     * @param packageId The id of the package associated with the promotion.
     */
    public void getPackagePromotionForWeb(String packageId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (WebViewActivity.getActivity() == null) {
            LoggingUtil.d("No Splash Screen active. No message will be sent.");
            return;
        }

        WebViewActivity.getActivity().javascriptBridge.nativeMessage("getBundlePromotion", getPackagePromotion(packageId));
    }

    /**
     * Retrieves the promotion list information.
     *
     * @return The JSON {@link String} for the Promotions.
     */
    public String getPromotions() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return gson.toJson(new ArrayList<Promotion>());
        }

        ArrayList<Promotion> promotionList = new ArrayList<>();
        promotionList.addAll(promotions.values());

        return gson.toJson(promotionList);
    }

    public HashMap<Integer, Promotion> getPromotionList() {
        return promotions;
    }

    /**
     * Method used to check if a promotion is available based on a bundle id.
     *
     * @param bundleId The id of the bundle associated with the promotion.
     * @return
     */
    public boolean hasBundlePromotion(int bundleId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity affectedEntity : promotion.getAffectedEntities()) {
                if (affectedEntity.getType().equals(PlayerDataManager.BundleCheck) && affectedEntity.getId() == bundleId) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Method used to check if a promotion is available based on a package id.
     *
     * @param id The id of the package associated with the promotion.
     * @return
     */
    public boolean hasPackagePromotion(int id) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        Package packageEntry = PackageManager.getInstance(context).getPackageObject(id);

        if (packageEntry == null) {
            return false;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity affectedEntity : promotion.getAffectedEntities()) {
                if (affectedEntity.getType().equals(PlayerDataManager.PackageCheck) && affectedEntity.getId() == packageEntry.getId()) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Method used to check if a promotion is available based on a package id (Google).
     *
     * @param packageId The id (Google) of the package associated with the promotion.
     * @return
     */
    public boolean hasPackagePromotion(String packageId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        Package packageEntry = PackageManager.getInstance(context).getPackageObject(packageId);

        if (packageEntry == null) {
            return false;
        }

        for (Promotion promotion : promotions.values()) {
            for (AffectedEntity affectedEntity : promotion.getAffectedEntities()) {
                if (affectedEntity.getId() == packageEntry.getId()) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Method used to check if a promotion is available based on a shop entry.
     *
     * @param entry The entry associated with the promotion.
     * @return
     */
    public boolean hasActiveEntryPromotion(Entry entry) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        if (entry.getType().equals(PlayerDataManager.BundleCheck)) {
            return hasBundlePromotion(entry.getId());
        } else if (entry.getType().equals(PlayerDataManager.PackageCheck)) {
            return hasPackagePromotion(entry.getId());
        }

        return false;
    }

    /**
     * Method used to check if a promotion is available in a shop tab.
     *
     * @param tab The shop tab associated with the promotion.
     * @return
     */
    public boolean hasActiveTabPromotion(Tab tab) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        for (Entry entry : tab.getEntries()) {
            if (entry.getType().equals(PlayerDataManager.BundleCheck)) {
                if (hasBundlePromotion(entry.getId())) {
                    return true;
                }
            } else if (entry.getType().equals(PlayerDataManager.PackageCheck)) {
                if (hasPackagePromotion(entry.getId())) {
                    return true;
                }
            }
        }

        return false;
    }

    private void updatePromotions() {
        storageUtil.putString(StorageUtil.Keys.SpilPromotions, gson.toJson(promotions));
    }

    public void resetPromotionData() {
        promotions.clear();
        storageUtil.remove(StorageUtil.Keys.SpilPromotions);
        mInstance = null;
    }
}
