package io.gamedock.sdk.gamedata.promotions;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Promotiobns Callbacks.
 * Notifies the application when the Packages configuration has been received and processed by the SDK as well as updates on the state of how many promotions of a certain type have been bought.
 */
public class PromotionsCallbacks {

    private OnPromotionListener promotionListener;

    public PromotionsCallbacks() {
        promotionListener = null;
    }

    public PromotionsCallbacks(OnPromotionListener promotionListener) {
        this.promotionListener = promotionListener;
    }

    /**
     * Method that is called when the Promotions configuration has been process and is available to be retrieved.
     */
    public void promotionsAvailable(String promotionsJSON) {
        if (promotionListener != null) {
            promotionListener.PromotionsAvailable(promotionsJSON);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "PromotionsAvailable", promotionsJSON);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when the Promotions configuration is not available.
     */
    public void promotionsNotAvailable() {
        if (promotionListener != null) {
            promotionListener.PromotionsNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "PromotionsNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called in order to notify the game that a promotion has been bought and if the max amount of purchases has been reached.
     * @param promotionId The id of the bought promotion.
     * @param amountPurchased The current amount of purchased promotions of this type.
     * @param maxAmountReached If the max amount of purchases has been reached.
     */
    public void promotionAmountBought(int promotionId, int amountPurchased, boolean maxAmountReached) {
        if (promotionListener != null) {
            promotionListener.PromotionAmountBought(promotionId, amountPurchased, maxAmountReached);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("promotionId", promotionId);
                jsonObject.put("amountPurchased", amountPurchased);
                jsonObject.put("maxAmountReached", maxAmountReached);

                UnityPlayer.UnitySendMessage("GamedockSDK", "PromotionAmountBought", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

}
