package io.gamedock.sdk.gamedata;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Game Data Callbacks.
 * Notifies the application when the Game Data has been received and processed by the SDK.
 * Informs the application if an error occurred while processing the Game Data values.
 */
public class GamedockGameDataCallbacks {
    private OnGameDataListener gameDataListener;

    public GamedockGameDataCallbacks() {
        this.gameDataListener = null;
    }

    public GamedockGameDataCallbacks(OnGameDataListener gameDataListener) {
        this.gameDataListener = gameDataListener;
    }

    /**
     * Method that is called when the Game Data has been process and is available to be retrieved.
     */
    public void gameDataAvailable(String gameDataJSON) {
        if (this.gameDataListener != null) {
            this.gameDataListener.GameDataAvailable(gameDataJSON);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "GamedockGameDataAvailable", gameDataJSON);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the GamedockGameDataCallback in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while processing the Game Data.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void gameDataError(ErrorCodes error) {
        if (this.gameDataListener != null) {
            this.gameDataListener.GameDataError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "GamedockGameDataError", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the GamedockGameDataCallback in order to receive information or something went wrong with Unity");
            }
        }
    }
}
