package io.gamedock.sdk.events.response;

import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;

import io.gamedock.sdk.models.userdata.UserDataMeta;
import io.gamedock.sdk.models.userdata.UserDataVersion;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.MissionData;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that processes information regarding user data responses.
 */
public class UserDataResponseEvent extends ResponseEvent {

    private Wallet walletObject = null;
    private Inventory inventoryObject = null;
    private MissionData missionDataObject = null;
    private boolean externalChange;
    private String gameStateData = null;
    private ArrayList<UserDataVersion> userDataVersions = new ArrayList<>();
    private UserDataMeta userDataMeta = null;

    UserDataResponseEvent(ResponseEvent responseEvent) {
        super();

        try {
            Gson gson = new Gson();
            if (responseEvent.responseData != null) {
                if (responseEvent.responseData.has("playerData")) {
                    JSONObject playerDataJSON = responseEvent.responseData.getJSONObject("playerData");
                    if (playerDataJSON.has("wallet")) {
                        JSONObject walletJSON = playerDataJSON.getJSONObject("wallet");

                        walletObject = new Wallet();

                        if (walletJSON.has("currencies")) {
                            JSONArray currenciesJSON = walletJSON.getJSONArray("currencies");

                            ConcurrentHashMap<Integer, PlayerCurrency> currencyHashMap = new ConcurrentHashMap<>();

                            for (int i = 0; i < currenciesJSON.length(); i++) {
                                JSONObject jsonObject = currenciesJSON.getJSONObject(i);

                                PlayerCurrency currency = new PlayerCurrency();
                                currency.setId(jsonObject.getInt("id"));
                                currency.setCurrentBalance(jsonObject.getInt("currentBalance"));

                                currency.setDelta(jsonObject.getInt("delta"));

                                currencyHashMap.put(currency.getId(), currency);
                            }
                            walletObject.setCurrenciesMap(currencyHashMap);
                        }


                        if (walletJSON.has("offset")) {
                            walletObject.setOffset(walletJSON.getLong("offset"));
                        }

                        if (walletJSON.has("logic")) {
                            walletObject.setLogic(walletJSON.getString("logic"));
                        }
                    }

                    if (playerDataJSON.has("inventory")) {
                        JSONObject inventoryJSON = playerDataJSON.getJSONObject("inventory");

                        inventoryObject = new Inventory();

                        if (inventoryJSON.has("items")) {
                            JSONArray itemsJSON = inventoryJSON.getJSONArray("items");

                            ConcurrentHashMap<Integer, PlayerItem> itemsHashMap = new ConcurrentHashMap<>();

                            for (int i = 0; i < itemsJSON.length(); i++) {
                                JSONObject jsonObject = itemsJSON.getJSONObject(i);

                                PlayerItem item = new PlayerItem();
                                item.setId(jsonObject.getInt("id"));
                                item.setAmount(jsonObject.getInt("amount"));

                                item.setDelta(jsonObject.getInt("delta"));

                                itemsHashMap.put(item.getId(), item);
                            }
                            inventoryObject.setItemsMap(itemsHashMap);
                        }

                        if (inventoryJSON.has("uniqueItems")) {
                            JSONArray uniqueItemsJSON = inventoryJSON.getJSONArray("uniqueItems");

                            ConcurrentHashMap<String, UniquePlayerItem> uniqueItemsHashMap = new ConcurrentHashMap<>();

                            JsonParser jsonParser = new JsonParser();

                            for (int i = 0; i < uniqueItemsJSON.length(); i++) {
                                UniquePlayerItem uniquePlayerItem = gson.fromJson(uniqueItemsJSON.getJSONObject(i).toString(), UniquePlayerItem.class);
                                if (uniqueItemsJSON.getJSONObject(i).has("uniqueProperties")) {
                                    JsonObject properties = (JsonObject) jsonParser.parse(uniqueItemsJSON.getJSONObject(i).getJSONObject("uniqueProperties").toString());
                                    uniquePlayerItem.setUniqueProperties(properties);
                                }

                                uniqueItemsHashMap.put(uniquePlayerItem.getUniqueId(), uniquePlayerItem);
                            }
                            inventoryObject.setUniqueItemsMap(uniqueItemsHashMap);
                        }

                        if (inventoryJSON.has("offset")) {
                            inventoryObject.setOffset(inventoryJSON.getLong("offset"));
                        }

                        if (inventoryJSON.has("logic")) {
                            inventoryObject.setLogic(inventoryJSON.getString("logic"));
                        }
                    }
                }

                if (responseEvent.responseData.has("gameState")) {
                    this.gameStateData = responseEvent.responseData.getJSONObject("gameState").toString();
                }

                if (responseEvent.responseData.has("missionData")) {
                    if (responseEvent.responseData.getJSONObject("missionData").length() == 0) {
                        missionDataObject = null;
                    } else {
                        missionDataObject = gson.fromJson(responseEvent.responseData.getJSONObject("missionData").toString(), MissionData.class);

                        JsonParser jsonParser = new JsonParser();
                        for (int i = 0; i < responseEvent.responseData.getJSONObject("missionData").getJSONArray("missionProgress").length(); i++) {
                            JsonObject progress = (JsonObject) jsonParser.parse(responseEvent.responseData.getJSONObject("missionData").getJSONArray("missionProgress").getJSONObject(i).getJSONObject("progress").toString());
                            missionDataObject.getMissionProgress().get(i).setProgress(progress);
                        }
                    }
                }

                if (responseEvent.responseData.has("externalChange")) {
                    externalChange = responseEvent.responseData.getBoolean("externalChange");
                }

                if (responseEvent.responseData.has("deviceVersions")) {
                    JSONObject deviceVersionsJSON = responseEvent.responseData.getJSONObject("deviceVersions");

                    if (deviceVersionsJSON.length() > 0) {
                        for (int i = 0; i < deviceVersionsJSON.names().length(); i++) {
                            UserDataVersion userDataVersion = new UserDataVersion();

                            userDataVersion.setDeviceId(deviceVersionsJSON.names().getString(i));
                            userDataVersion.setVersion(deviceVersionsJSON.getInt(deviceVersionsJSON.names().getString(i)));

                            userDataVersions.add(userDataVersion);
                        }
                    }
                }

                if (responseEvent.responseData.has("metaData")) {
                    JSONObject metaDataJSON = responseEvent.responseData.getJSONObject("metaData");

                    userDataMeta = new UserDataMeta();

                    if (metaDataJSON.has("serverTime")) {
                        userDataMeta.serverTime = metaDataJSON.getLong("serverTime");
                    }

                    if (metaDataJSON.has("clientTime")) {
                        userDataMeta.clientTime = metaDataJSON.getLong("clientTime");
                    }

                    if (metaDataJSON.has("timezoneOffset")) {
                        userDataMeta.timezoneOffset = metaDataJSON.getInt("timezoneOffset");
                    }

                    if (metaDataJSON.has("deviceModel")) {
                        userDataMeta.deviceModel = metaDataJSON.getString("deviceModel");
                    }

                    if (metaDataJSON.has("appVersion")) {
                        userDataMeta.appVersion = metaDataJSON.getString("appVersion");
                    }

                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }

        setName(responseEvent.getName());
        setType(responseEvent.getType());
        setAction(responseEvent.getAction());
        this.responseData = responseEvent.responseData;
    }

    @Override
    public void processData(Context context) {
        LoggingUtil.d("Processing data for UserDataResponseEvent");

        String action = getAction().toLowerCase().trim();

        switch (action) {
            case "request":
                UserDataManager.getInstance(context).processRequestUserData(walletObject, inventoryObject, missionDataObject, externalChange, gameStateData, userDataVersions, userDataMeta);
                break;
            case "mergeconflict":
                UserDataManager.getInstance(context).processMergeConflict(walletObject, inventoryObject, missionDataObject, gameStateData, userDataVersions, userDataMeta);
                break;
            case "merge":
                UserDataManager.getInstance(context).processMergeUserData(userDataVersions);
                break;
            case "lockerror":
                UserDataManager.getInstance(context).processLockError();
                break;
            case "syncerror":
                UserDataManager.getInstance(context).processSyncError();
                break;
        }
    }
}