package io.gamedock.sdk.events.response;

import android.content.Context;
import android.util.Log;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that holds all the attributes of a response.
 * Handles the generation of different types of response events.
 * It can create the following responses:
 * - Advertisement Response Event
 * - Overlay Response Event ({@link android.webkit.WebView} overlay)
 * - Game Configuration Response Event
 * - IAP Packages Response Event
 * - Notification Response Event
 * - Player Data Response Event
 * - Game Data Response Event
 * - Game State Response Event
 * - Reward Response Event
 */
public class ResponseEvent extends Event {

    private static String TAG = "ResponseEvent";
    public JSONObject responseData = new JSONObject();
    public JSONObject ids = new JSONObject();
    private String action = "";
    private String type = "";
    private String message = "";
    private String segments = "";
    private String serviceData = "";

    public ResponseEvent() {
        super();
    }

    public static ResponseEvent Build(String jsonString) {
        ResponseEvent responseEvent = null;

        try {
            JSONObject rawJSON = new JSONObject(jsonString);

            responseEvent = new ResponseEvent();
            responseEvent.setName(rawJSON.getString("name"));

            if (rawJSON.has("type")) {
                responseEvent.setType(rawJSON.getString("type"));
            }

            if (rawJSON.has("action")) {
                responseEvent.setAction(rawJSON.getString("action"));
            }

            if (rawJSON.has("message")) {
                responseEvent.setMessage(rawJSON.getString("message"));
            }

            if (rawJSON.has("data")) {
                responseEvent.responseData = rawJSON.getJSONObject("data");
            }

            if (rawJSON.has("ids")) {
                responseEvent.setIds(rawJSON.getJSONObject("ids"));
            }

            if (rawJSON.has("segments")) {
                responseEvent.setSegments(rawJSON.getString("segments"));
            }

            if (rawJSON.has("serviceData")) {
                responseEvent.setServiceData(rawJSON.getString("serviceData"));
            }

            switch (responseEvent.getType().toLowerCase().trim()) {
                case "sdk":
                    responseEvent = new InitializeSDKResponseEvent(responseEvent);
                    break;
                case "advertisement":
                    responseEvent = new AdvertisementResponseEvent(responseEvent);
                    break;
                case "overlay":
                    responseEvent = new OverlayResponseEvent(responseEvent);
                    break;
                case "gameconfig":
                    responseEvent = new ConfigResponseEvent(responseEvent);
                    break;
                case "notification":
                    responseEvent = new NotificationResponseEvent(responseEvent);
                    break;
                case "userdata":
                    responseEvent = new UserDataResponseEvent(responseEvent);
                    break;
                case "playerdata":
                    responseEvent = new PlayerDataResponseEvent(responseEvent);
                    break;
                case "gamestate":
                    responseEvent = new GameStateResponseEvent(responseEvent);
                    break;
                case "missiondata":
                    responseEvent = new MissionDataResponseEvent(responseEvent);
                    break;
                case "gamedata":
                    responseEvent = new GameDataResponseEvent(responseEvent);
                    break;
                case "missions":
                    responseEvent = new MissionResponseEvent(responseEvent);
                    break;
                case "promotions":
                    responseEvent = new PromotionsResponseEvent(responseEvent);
                    break;
                case "iap":
                    responseEvent = new IAPResponseEvent(responseEvent);
                    break;
                case "reward":
                    responseEvent = new RewardResponseEvent(responseEvent);
                    break;
                case "serverdata":
                    responseEvent = new ServerDataResponseEvent(responseEvent);
                    break;
                case "tieredevent":
                    responseEvent = new TieredResponseEvent(responseEvent);
                    break;
                case "assetbundles":
                    responseEvent = new AssetBundlesResponseEvent(responseEvent);
                    break;
                case "localization":
                    responseEvent = new LocalizationResponseEvent(responseEvent);
                    break;
                case "moreapps":
                    responseEvent = new MoreAppsResponseEvent(responseEvent);
                    break;
                case "authentication":
                    responseEvent = new SocialLoginResponseEvent(responseEvent);
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            LoggingUtil.e("Error building response event: " + e.getMessage() + "\nFrom response value: " + ((jsonString != null && !jsonString.isEmpty()) ? jsonString : "Null"));
        }
        return responseEvent;
    }

    @Override
    public void addData(String key, JSONObject value) {
        try {
            responseData.put(key, value);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @Override
    public String toString() {
        return "name=" + getName() + "; action=" + getAction() + "; type=" + getType() + "; ids=" + getIds() + "; segments=" + getSegments() + "; serviceData=" + getServiceData() + "; data=" + responseData.toString();
    }

    public String toStringNotification() {
        try {
            return "{" + '"' + "name" + '"' + ":" + '"' + getName() + '"' + "," + '"' + "action" + '"' + ":" + '"' + getAction() + '"' + "," + '"' + "type" + '"' + ":" + '"' + getType() + '"' + "," + '"' + "data" + '"' + ":" + (responseData.get("eventData") != null ? "{\"eventData\":" + responseData.get("eventData").toString() + "}" : "{}") + "}";
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return "";
    }

    public String toJSONString(boolean readable) {
        try {
            JSONObject jo = new JSONObject();
            jo.put("name", getName());
            jo.put("action", getAction());
            jo.put("type", getType());
            jo.put("data", responseData);

            if (readable) {
                return jo.toString(1);
            } else {
                return jo.toString();
            }

        } catch (Exception e) {
            Log.e(TAG, "cannot parse response event to json " + e.getMessage());
        }
        return "";
    }

    public String getAction() {
        return action;
    }

    public void setAction(String action) {
        this.action = action;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public String getSegments() {
        return segments;
    }

    public void setSegments(String segments) {
        this.segments = segments;
    }

    public String getServiceData() {
        return serviceData;
    }

    public void setServiceData(String serviceData) {
        this.serviceData = serviceData;
    }

    public JSONObject getIds() {
        return ids;
    }

    public void setIds(JSONObject ids) {
        this.ids = ids;

        if (ids.length() == 0) {
            return;
        }

        try {
            for(int i = 0; i< ids.names().length(); i++){
                FeaturesUtil.setFeatureVersionId(ids.names().getString(i), ids.getLong(ids.names().getString(i)));
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Dummy method.
     * Check implementation of this method in each ResponseEvent object
     *
     * @param context The activity context.
     */
    public void processData(Context context) {

    }
}