package io.gamedock.sdk.events.response;

import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.models.ads.admob.AdMobObject;
import io.gamedock.sdk.models.ads.internal.PriorityAdConfig;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that processes information regarding advertisement responses.
 * Builds the response event object based on the information received from the response.
 * Can create the following:
 * - Ads initialization event
 * - will initialise Chartboost or Ad Mob
 */
public class AdvertisementResponseEvent extends ResponseEvent {

    private AdMobObject adMobObject = null;

    private String notAvailableAdType = "";
    private long nextTimestampInterstitial;

    AdvertisementResponseEvent(ResponseEvent responseEvent) {
        super();

        try {
            if (responseEvent.getAction().toLowerCase().trim().equals("init")) {

                //Populate the priorty ads configuration list (if it exists)
                ArrayList<PriorityAdConfig> priorityAdConfigs = null;
                if(responseEvent.responseData.has("priorityAdsConfig")) {
                    Type listType = new TypeToken<ArrayList<PriorityAdConfig>>(){}.getType();
                    priorityAdConfigs = new Gson().fromJson(responseEvent.responseData.getJSONArray("priorityAdsConfig").toString(), listType);
                }

                JSONObject providers = responseEvent.responseData.getJSONObject("providers");
                if (providers.has(AdManager.AdMob)) {
                    String appId;
                    String bannerAdUnitId = null;
                    String interstitialAdUnitId = null;
                    String rewardVideoAdUnitId = null;
                    Map<String, String> interstitialCustomTargeting = null;
                    Map<String, String> rewardVideoCustomTargeting = null;
                    AdMobMediationNetworks adMobMediationNetworks = null;
                    ArrayList<PriorityAdConfig> adMobPriorityAdConfigs = new ArrayList<>();

                    appId = providers.getJSONObject(AdManager.AdMob).getString("appId");

                    if (providers.getJSONObject(AdManager.AdMob).has(AdManager.BANNER)) {
                        bannerAdUnitId = providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.BANNER).getString("adUnitId");
                    }

                    if (providers.getJSONObject(AdManager.AdMob).has(AdManager.INTERSTITIAL)) {
                        interstitialAdUnitId = providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.INTERSTITIAL).getString("adUnitId");

                        if (providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.INTERSTITIAL).has("targeting")) {

                            interstitialCustomTargeting = new HashMap<>();
                            JSONObject jsonObject = providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.INTERSTITIAL).getJSONObject("targeting");

                            Iterator<String> it = responseData.keys();
                            while (it.hasNext()) {
                                String key = it.next();
                                String value = jsonObject.getString(key);

                                interstitialCustomTargeting.put(key, value);
                            }
                        }
                    }

                    if (providers.getJSONObject(AdManager.AdMob).has(AdManager.REWARD_VIDEO)){
                        rewardVideoAdUnitId = providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.REWARD_VIDEO).getString("adUnitId");

                        if (providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.REWARD_VIDEO).has("targeting")) {

                            rewardVideoCustomTargeting = new HashMap<>();
                            JSONObject jsonObject = providers.getJSONObject(AdManager.AdMob).getJSONObject(AdManager.REWARD_VIDEO).getJSONObject("targeting");

                            Iterator<String> it = responseData.keys();
                            while (it.hasNext()) {
                                String key = it.next();
                                String value = jsonObject.getString(key);

                                rewardVideoCustomTargeting.put(key, value);
                            }
                        }
                    }

                    //Select from the priority ad config list the configs belonging to AdMob
                    if (priorityAdConfigs != null) {
                        for (PriorityAdConfig adConfig: priorityAdConfigs) {
                            if (adConfig.getProvider().equals(AdManager.AdMob)) {
                                adMobPriorityAdConfigs.add(adConfig);
                            }
                        }
                    }

                    adMobObject = new AdMobObject(appId, bannerAdUnitId, interstitialAdUnitId, rewardVideoAdUnitId, null, 0, interstitialCustomTargeting, rewardVideoCustomTargeting, adMobMediationNetworks, adMobPriorityAdConfigs);
                }
            } else if (responseEvent.getAction().toLowerCase().trim().equals("show")) {
                //Check if the response informs the SDK if the ad can be shown (based on conditions from the backend)
                boolean available = true;
                if (responseEvent.responseData.has("available")) {
                    available = responseEvent.responseData.getBoolean("available");
                    if (!available) {
                        if (responseEvent.responseData.has("status")) {
                            String status = responseEvent.responseData.getString("status");
                            LoggingUtil.d(status);
                        }

                        notAvailableAdType = responseEvent.responseData.getString("adType");

                        if (responseEvent.responseData.has("secondsToInterstitial")) {
                            LoggingUtil.d("The timer for interstitials has been set to: " + responseEvent.responseData.getString("secondsToInterstitial") + " seconds");
                        }


                        if (responseEvent.responseData.has("nextTimestamp")) {
                            LoggingUtil.d("Interstitial not available due to backend timer");
                            nextTimestampInterstitial = responseEvent.responseData.getLong("nextTimestamp");
                        }
                    }
                }

                if (available) {
                    String provider = responseEvent.responseData.getString("provider");
                    switch (provider) {
                        case AdManager.AdMob: {
                            String adType = responseEvent.responseData.getString("adType");
                            int conditionId = 0;
                            if (responseEvent.responseData.has("conditionId")) {
                                conditionId = responseEvent.responseData.getInt("conditionId");
                            }
                            adMobObject = new AdMobObject(null, null, null,null, adType, conditionId,null, null, null, null);
                            break;
                        }
                    }
                }
            }

            setName(responseEvent.getName());
            setType(responseEvent.getType());
            setAction(responseEvent.getAction());
            setIds(responseEvent.getIds());
            this.responseData = responseEvent.responseData;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void processData(Context context) {
        LoggingUtil.d("Processing data for AdvertisementResponseEvent");

        String action = getAction().toLowerCase().trim();

        if (action.equals("init")) {
            if (adMobObject != null) {
                GamedockSDK.getInstance(context).startAdMob(adMobObject.appId, adMobObject.bannerAdUnitId, adMobObject.interstitialAdUnitId, adMobObject.interstitialCustomTargeting, adMobObject.rewardVideoAdUnitId, adMobObject.rewardVideoCustomTargeting, adMobObject.adMobMediationNetworks, adMobObject.priorityAdConfigs);
            }
        } else if (action.equals("show")) {
            if (adMobObject != null) {
                GamedockSDK.getInstance(context).requestAd(AdManager.AdMob, adMobObject.adType, adMobObject.conditionId);
            } else {
                AdManager.getInstance().nextTimestampInterstitial = nextTimestampInterstitial;
                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(notAvailableAdType);
                notAvailableAdType = null;
            }
        }
    }

    public AdMobObject getAdMobObject() {
        return adMobObject;
    }

}
