package io.gamedock.sdk.config.internal;

import android.content.Context;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;
import android.widget.Toast;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.models.config.AppsFlyer;
import io.gamedock.sdk.models.config.Chartboost;
import io.gamedock.sdk.models.config.Firebase;
import io.gamedock.sdk.models.config.GoogleServices;
import io.gamedock.sdk.models.config.Lotame;
import io.gamedock.sdk.models.config.Spil;
import io.gamedock.sdk.utils.device.DeviceUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Class that handles the initialisation and all the logic behind the SDK config values.
 * Uses the Game Config to retrieve the SDK Config values.
 * Checks if Debug Mode was enabled.
 * Offers manual setting of the subdomain.
 */
public class GamedockConfigManager {
    private static final Object lock = new Object();

    private static volatile GamedockConfigManager mInstance = null;

    public AppsFlyer appsFlyer = new AppsFlyer();
    public Lotame lotame = new Lotame();
    public Chartboost chartboost = new Chartboost();
    public Firebase firebase = new Firebase();
    public GoogleServices googleServices = new GoogleServices();
    public Spil spil = new Spil();

    private String googleAdvertisingId = null;

    private boolean debugModeEnabled;

    public static GamedockConfigManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new GamedockConfigManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Constructor method that handles the creation of the Gamedock Config object.
     * Hold the configuration information for the following features: Chartboost, Google Analytics, Adjust, Zendesk, Debug Flag and Google Cloud Subdomain.
     *
     * @param context The activity context.
     */
    private GamedockConfigManager(final Context context) {
        try {

            JSONObject sdkConfig = new JSONObject(GamedockSDK.getInstance(context).getConfigValue("androidSdkConfig"));

            if (sdkConfig.has("version")) {
                int version = sdkConfig.getInt("version");
                LoggingUtil.w("New Gamedock Config Structure detected! Using version: " + version);

                init(context, sdkConfig);
            } else {
                LoggingUtil.e("Old Gamedock Config detected! Using old initialisation! Please update your Gamedock Configuration to the new format!");
                deprecatedInit(context, sdkConfig);
            }

        } catch (NullPointerException | JSONException e) {
            Uri link = Uri.parse("https://goo.gl/zHzmXJ");
            LoggingUtil.e("Could not retrieve the 'androidSdkConfig' json object from the 'defaultGameConfig.json' file. Please make sure to include this file into your application and include the json object following a similar structure as found here: " + link);
        }

        retrieveGoogleAdvertisingId(context);
    }

    /**
     * Method used to initialise the Gamedock configuration part of the game config. It uses the new standard.
     *
     * @param context   The Activity context.
     * @param sdkConfig The {@link JSONObject} containing the Gamedock configuration that needs to be created.
     */
    private void init(Context context, JSONObject sdkConfig) {

        Gson gson = GamedockSDK.getInstance(context).getGson();

        try {
            if (sdkConfig.has("appsFlyer")) {
                try {
                    String appsflyerObject = sdkConfig.getJSONObject("appsFlyer").toString();
                    appsFlyer = gson.fromJson(appsflyerObject, AppsFlyer.class);
                } catch (JsonSyntaxException e) {
                    e.printStackTrace();
                }
            } else {
                LoggingUtil.e("AppsFlyer Configuration is missing from 'defaultGameConfig.json'!");
            }

            if (sdkConfig.has("lotame")) {
                try {
                    String lotameObject = sdkConfig.getJSONObject("lotame").toString();
                    lotame = gson.fromJson(lotameObject, Lotame.class);
                } catch (JsonSyntaxException e) {
                    e.printStackTrace();
                }
            } else {
                LoggingUtil.e("Lotame Configuration is missing from 'defaultGameConfig.json'!");
            }

            if (sdkConfig.has("chartboost")) {
                String chartboostObject = sdkConfig.getJSONObject("chartboost").toString();
                chartboost = gson.fromJson(chartboostObject, Chartboost.class);
            } else {
                LoggingUtil.e("Chartboost Configuration is missing from 'defaultGameConfig.json'!");
            }

            if (sdkConfig.has("firebase")) {
                String firebaseObject = sdkConfig.getJSONObject("firebase").toString();
                firebase = gson.fromJson(firebaseObject, Firebase.class);
            } else {
                LoggingUtil.e("Firebase Configuration is missing from 'defaultGameConfig.json'!");
            }

            if (sdkConfig.has("googleServices")) {
                String googleServicesObject = sdkConfig.getJSONObject("googleServices").toString();
                googleServices = gson.fromJson(googleServicesObject, GoogleServices.class);
            } else {
                LoggingUtil.e("Google Services Configuration are missing from 'defaultGameConfig.json'!");
            }

            if (sdkConfig.has("spil")) {
                String spilObject = sdkConfig.getJSONObject("spil").toString();
                spil = gson.fromJson(spilObject, Spil.class);
            } else {
                LoggingUtil.e("Gamedock Configuration is missing from 'defaultGameConfig.json'!");
            }

            setDebugMode(context);
        } catch (JSONException e) {
            e.printStackTrace();
        }

    }

    /**
     * Method used to initialise the Gamedock configuration part of the game config. It uses the old standard and it is used for backwards compatibility.
     *
     * @param context   The Activity context.
     * @param sdkConfig The {@link JSONObject} containing the Gamedock configuration that needs to be created.
     */
    private void deprecatedInit(Context context, JSONObject sdkConfig) {
        chartboost = new Chartboost();
        googleServices = new GoogleServices();
        spil = new Spil();

        try {
            if (sdkConfig.has("chartBoostAppId")) {
                chartboost.appId = sdkConfig.getString("chartBoostAppId");
            } else {
                LoggingUtil.e("Missing 'chartBoostAppId' key-value from 'defaultGameConfig.json'");
            }

            if (sdkConfig.has("chartBoostAppSignature")) {
                chartboost.appSignature = sdkConfig.getString("chartBoostAppSignature");
            } else {
                LoggingUtil.e("Missing 'chartBoostAppSignature' key-value from 'defaultGameConfig.json'");
            }

            if (sdkConfig.has("googleIAPKey")) {
                googleServices.IAPKey = sdkConfig.getString("googleIAPKey");
            } else {
                LoggingUtil.e("Missing 'googleIAPKey' key-value from 'defaultGameConfig.json'");
            }

            if (sdkConfig.has("googleAnalyticsId")) {
                googleServices.analyticsId = sdkConfig.getString("googleAnalyticsId");
            } else {
                LoggingUtil.e("Missing 'googleAnalyticsId' key-value from 'defaultGameConfig.json'");
            }

            if (GamedockSDK.getInstance(context).getStorageUtil().getBoolean(StorageUtil.Keys.SpilSubdomainOverride, false)) {
                spil.subDomain = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.SpilSubdomain, null);
            } else if (sdkConfig.has("spilSubdomain")) {
                spil.subDomain = sdkConfig.getString("spilSubdomain");
                if (!spil.subDomain.equals("")) {
                    spil.subDomain = spil.subDomain + ".";
                }
            } else {
                spil.subDomain = "";
                LoggingUtil.i("No Gamedock Custom Domain set. Defaulting to the regular domain.");
            }

            setDebugMode(context);

        } catch (JSONException e) {
            Uri link = Uri.parse("https://goo.gl/zHzmXJ");
            LoggingUtil.e("Could not retrieve the 'androidSdkConfig' json object from the 'defaultGameConfig.json' file. Please make sure to include this file into your application and include the json object following a similar structure as found here: " + link);
        }
    }

    public String getGoogleAdvertisingId() {
        try {
            return googleAdvertisingId;
        } catch (NullPointerException e) {
            return "";
        }
    }

    public boolean isDebugModeEnabled() {
        return debugModeEnabled;
    }

    /**
     * Method used to set the debug flag retrieved from the Sample App.
     *
     * @param context The context of the Activity.
     */
    public void setDebugMode(final Context context) {
        debugModeEnabled = DeviceUtil.getDebugMode(context);

        if (debugModeEnabled) {
            LoggingUtil.d("GamedockSDK Debug Mode On!");

            new Handler(Looper.getMainLooper()).post(new Runnable() {
                @Override
                public void run() {
                    if (GamedockSDK.getInstance(context).isAppRunning()) {
                        Toast.makeText(context, "GamedockSDK Debug Mode On!", Toast.LENGTH_SHORT).show();
                    }
                }
            });


            LoggingUtil.fullLogging = true;
        }
    }

    /**
     * Method used to retrieve from the system the Google Advertising Id.
     *
     * @param context The context of the Activity.
     */
    public void retrieveGoogleAdvertisingId(final Context context) {
        try {
            googleAdvertisingId = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.GoogleAdvertisingId, null);

            if (googleAdvertisingId == null) {
                if (GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS) {
                    //Google Play Services are available
                    AsyncTask<Void, Void, String> googleAdIdTask = new AsyncTask<Void, Void, String>() {
                        @Override
                        protected String doInBackground(Void... voids) {
                            try {
                                AdvertisingIdClient.Info idInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                                return idInfo.getId();
                            } catch (NullPointerException | GooglePlayServicesNotAvailableException | GooglePlayServicesRepairableException | IOException e) {
                                e.printStackTrace();
                            }

                            return null;
                        }

                        @Override
                        protected void onPostExecute(String result) {
                            googleAdvertisingId = result;
                            if (result != null) {
                                GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.GoogleAdvertisingId, googleAdvertisingId);
                            }
                        }
                    };
                    googleAdIdTask.execute();
                }
            }
        } catch (NoClassDefFoundError | Exception e) {
            LoggingUtil.e("Something went wrong retrieving the Google Advertising ID. Exception was: " + e.getClass().getSimpleName());
            e.printStackTrace();
        }
    }

    public void resetGamedockConfigManager() {
        mInstance = null;
    }
}
