package io.gamedock.sdk.assetbundles;

import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.AssetBundlesEvent;
import io.gamedock.sdk.models.assetbundles.AssetBundle;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Manager class used to handle the Unity Asset Bundle configurations received from the Gamedock backend.
 */
public class AssetBundlesManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "assetBundles";

    private static volatile AssetBundlesManager mInstance = null;
    private Context context;

    private Gson gson;
    private StorageUtil storageUtil;

    private ArrayList<AssetBundle> assetBundles;

    private AssetBundlesManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
        storageUtil = GamedockSDK.getInstance(context).getStorageUtil();

        String assetBundlesJSON = storageUtil.getString(StorageUtil.Keys.AssetBundles, "[]");
        assetBundles = gson.fromJson(assetBundlesJSON, new TypeToken<ArrayList<AssetBundle>>() {
        }.getType());
    }

    public static AssetBundlesManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new AssetBundlesManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to send a request event to fetch the asset bundle configurations from the backend.
     * If no internet is available will check if any locally stored configurations are available and inform the game accordingly.
     */
    public void requestAssetBundles() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        AssetBundlesEvent assetBundlesEvent = new AssetBundlesEvent(context);
        assetBundlesEvent.setRequestAssetBundles();

        GamedockSDK.getInstance(context).trackEvent(assetBundlesEvent, null);

        if (!NetworkUtil.isInternetAvailable(context)) {
            if (!assetBundles.isEmpty()) {
                GamedockSDK.getInstance(context).getAssetBundlesCallbacks().assetBundlesAvailable(gson.toJson(assetBundles));
            } else {
                GamedockSDK.getInstance(context).getAssetBundlesCallbacks().assetBundlesNotAvailable();
            }
        }
    }

    /**
     * Method used to process the response from the backend for asset bundle configurations.
     *
     * @param receivedAssetBundles The list of asset bundle configurations.
     */
    public void processAssetBundles(ArrayList<AssetBundle> receivedAssetBundles) {
        if (receivedAssetBundles == null) {
            return;
        }

        assetBundles = receivedAssetBundles;
        storageUtil.putString(StorageUtil.Keys.AssetBundles, gson.toJson(assetBundles));

        if (!assetBundles.isEmpty()) {
            GamedockSDK.getInstance(context).getAssetBundlesCallbacks().assetBundlesAvailable(gson.toJson(assetBundles));
        } else {
            GamedockSDK.getInstance(context).getAssetBundlesCallbacks().assetBundlesNotAvailable();
        }
    }

    public String getAssetBundlesJSON() {
        return gson.toJson(assetBundles);
    }

    public ArrayList<AssetBundle> getAssetBundles() {
        return assetBundles;
    }

    public void resetAssetBundlesData() {
        assetBundles.clear();
        storageUtil.remove(StorageUtil.Keys.AssetBundles);
        mInstance = null;
    }
}
