package io.gamedock.sdk.ads.internal;

import android.content.Context;
import android.content.Intent;

import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.core.rewarded.RewardedVideoAd;
import io.gamedock.sdk.ads.core.rewarded.RewardedVideoAdListener;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class GamedockAdsRewardedVideo {
    private GamedockAdsManager manager;

    private boolean isRewardVideoModuleEnabled;
    public boolean isVideoEnabled;

    private RewardedVideoAd rewardedVideoAd;
    private String rewardVideoRequestId;
    private String videoLocation;
    private String videoRewardType;
    private RewardedVideoAdListener rewardedVideoAdLoadListener;

    private String placementId;
    private AdProvider currentProvider;
    private boolean earnedReward;

    public void initialise(final Context context, BaseAdManagerImplementation baseAdManagerImplementation) {
        this.manager = (GamedockAdsManager) baseAdManagerImplementation;
        this.isRewardVideoModuleEnabled = true;

        rewardedVideoAdLoadListener = new RewardedVideoAdListener() {
            @Override
            public void onAdLoaded() {
                LoggingUtil.d("Rewarded Video Ad Module: Ad loaded callback");

                LoggingUtil.d("Ad is available");
                isVideoEnabled = true;
                GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.REWARD_VIDEO);
                AdMobManager.getInstance().resetPriorityAdConfigsFillSettings(AdManager.REWARD_VIDEO);
                AdEvents.rewardVideoAvailable(context);
            }

            @Override
            public void onAdFailedToLoad(ErrorCodes error) {
                LoggingUtil.d("Rewarded Video Ad Module: Ad failed to load callback: " + error.getName());

                AdEvents.rewardVideoNotAvailable(context);

                if (AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig() != null) {
                    LoggingUtil.d("Failed to load priority rewarded video ad.");

                    AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().setHasFill(false);
                    AdMobManager.getInstance().requestAdMobRewardVideo(context);
                } else {
                    isVideoEnabled = false;
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
                }
            }

            @Override
            public void onAdDisplayed() {
                LoggingUtil.d("Rewarded Video Ad Module: Ad displayed callback");

                AdManager.getInstance().isFullScreenAdShowing = true;

                if (AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig() == null) {
                    LoggingUtil.d("Default rewarded video displayed");
                    AdEvents.rewardVideoDidDisplay(context, placementId, false);
                } else {
                    LoggingUtil.d("Priority rewarded video displayed");
                    AdEvents.rewardVideoDidDisplay(context, placementId, true);
                    AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().setImpressions(AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().getImpressions() - 1);
                }
            }

            @Override
            public void onAdFailedToDisplay(ErrorCodes error) {
                LoggingUtil.d("Rewarded Video Ad Module: Ad failed to display callback: " + error.getName());

                AdManager.getInstance().isFullScreenAdShowing = false;

                if (AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig() != null) {
                    AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().setImpressions(AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().getImpressions() - 1);
                }

                AdEvents.rewardVideoDidNotDisplay(context);
                GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.GamedockAds, AdManager.REWARD_VIDEO, "dismiss");

                if (GamedockAdsAdActivity.activity != null) {
                    GamedockAdsAdActivity.activity.finish();
                }
            }

            @Override
            public void onAdClicked() {
                LoggingUtil.d("Rewarded Video Ad Module: Ad clicked callback");

                AdEvents.rewardVideoDidClick(context);
            }

            @Override
            public void onAdClosed() {
                LoggingUtil.d("Rewarded Video Ad Module: Ad closed callback");

                AdManager.getInstance().isFullScreenAdShowing = false;

                if (!earnedReward) {
                    AdEvents.rewardVideoDidDismiss(context);
                    GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.GamedockAds, AdManager.REWARD_VIDEO, "dismiss");
                } else {
                    AdEvents.rewardVideoDidClose(context);
                    GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.GamedockAds, AdManager.REWARD_VIDEO, "close");
                    earnedReward = false;
                }
            }

            @Override
            public void onAdRewardEarned() {
                LoggingUtil.d("Rewarded Video Ad Module: Ad Reward Earned callback");
                earnedReward = true;
            }
        };
    }

    public void request(final Context context, AdProvider provider, String placementId, boolean isPriority, boolean fromAdMob) {
        if (!isRewardVideoModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.REWARD_VIDEO)) {
            LoggingUtil.d("Requesting a rewarded video while one is already showing is not allowed");
            return;
        }

        this.placementId = placementId;
        this.currentProvider = provider;
        this.earnedReward = false;

        rewardVideoRequestId = UUID.randomUUID().toString();

        rewardedVideoAd = new RewardedVideoAd(provider, placementId, rewardedVideoAdLoadListener);

        AdEvents.rewardedVideoRequested(context, placementId, isPriority);

        rewardedVideoAd.load();
    }

    public void show(final Context context) {
        if (!isRewardVideoModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.REWARD_VIDEO)) {
            LoggingUtil.d("Showing a rewarded video while one is already showing is not allowed");
            return;
        }

        if (rewardedVideoAd.isLoaded()) {
            GamedockSDK.getInstance(context).isShowingChildActivity = true;

            Intent intent = new Intent(context, GamedockAdsAdActivity.class);
            intent.putExtra("adType", AdManager.REWARD_VIDEO);
            intent.putExtra("adProvider", AdManager.GamedockAds);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

            GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.REWARD_VIDEO);
            AdManager.getInstance().isFullScreenAdShowing = true;
            context.startActivity(intent);
        } else {
            LoggingUtil.d("No rewarded video ad loaded. Stopping current show.");
            isRewardVideoModuleEnabled = false;
            AdEvents.rewardVideoNotAvailable(context);

            if (AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig() != null) {
                AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().setHasFill(false);
            }
        }
    }

    public String getRewardVideoRequestId() {
        return rewardVideoRequestId;
    }

    public String getVideoLocation() {
        return videoLocation;
    }

    public void setVideoLocation(String videoLocation) {
        this.videoLocation = videoLocation;
    }

    public String getVideoRewardType() {
        return videoRewardType;
    }

    public void setVideoRewardType(String videoRewardType) {
        this.videoRewardType = videoRewardType;
    }

    public RewardedVideoAd getRewardedVideoAd() {
        return rewardedVideoAd;
    }

    public boolean hasEarnedReward() {
        return earnedReward;
    }

    public void setEarnedReward(boolean earnedReward) {
        this.earnedReward = earnedReward;
    }

    public AdProvider getCurrentProvider() {
        return currentProvider;
    }
}
