package io.gamedock.sdk.ads.internal;

import android.content.Context;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalProvider;
import io.gamedock.sdk.ads.providers.init.BaseAdProviderInit;
import io.gamedock.sdk.ads.providers.init.InitializationStatus;
import io.gamedock.sdk.ads.providers.init.OnGamedockAdsInitializationCompleted;
import io.gamedock.sdk.ads.utils.settings.Settings;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class GamedockAdsManager extends BaseAdManagerImplementation {
    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "ads";

    private static volatile GamedockAdsManager mInstance;
    public boolean initialised;

    private GamedockAdsBanner gamedockAdsBanner = new GamedockAdsBanner();
    private GamedockAdsInterstitial gamedockAdsInterstitial = new GamedockAdsInterstitial();
    private GamedockAdsRewardedVideo gamedockAdsRewardedVideo = new GamedockAdsRewardedVideo();

    public static GamedockAdsManager getInstance() {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new GamedockAdsManager();
                }
            }
        }
        return mInstance;
    }

    public void initialise(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        try {
            Settings settings = new Settings();
            settings.setWithUserConsent(AdManager.getInstance().checkPersonalisedAdsPermission(context));
            settings.setDebugModeEnabled(true);

            ArrayList<BaseAdProviderInit> adNetworkInits = new ArrayList<>();
            ImproveDigitalProvider.Init improveDigitalInit = new ImproveDigitalProvider.Init();
            adNetworkInits.add(improveDigitalInit);

            GamedockAds.initialize(context, adNetworkInits, settings, new OnGamedockAdsInitializationCompleted() {
                @Override
                public void onInitializationCompleted(ArrayList<InitializationStatus> status) {
                    GamedockAdsManager.getInstance().initialised = true;

                    gamedockAdsBanner.initialise(context, getInstance());
                    gamedockAdsInterstitial.initialise(context, getInstance());
                    gamedockAdsRewardedVideo.initialise(context, getInstance());
                }
            });
        } catch (NoClassDefFoundError | Exception e) {
            LoggingUtil.e("Gamedock Ads Module not included! If you want to use Gamedock Internal Ads please include the gamedock-sdk-ads dependency");
        }
    }

    public void requestGamedockAdsBanner(Context context, AdProvider provider, String placementId, String position, String adSize, boolean isPriority, boolean fromAdMob) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Request Gamedock Ads Banner");

        gamedockAdsBanner.request(context, provider, placementId, position, adSize, isPriority, fromAdMob);
    }

    public void showGamedockAdsBanner(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Show Gamedock Ads Banner");

        gamedockAdsBanner.show(context);
    }

    public void hideGamedockAdsBanner(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Hide Gamedock Ads Banner");

        gamedockAdsBanner.hide(context);
    }

    public void requestGamedockAdsInterstitial(final Context context, AdProvider provider, String videoPlacementId, String staticPlacementId, boolean isPriority, boolean fromAdMob) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        LoggingUtil.d("Request Gamedock Ads Interstitial");

        gamedockAdsInterstitial.request(context, provider, videoPlacementId, staticPlacementId, isPriority, fromAdMob);
    }

    public void showGamedockAdsInterstitial(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        LoggingUtil.d("Show Gamedock Ads Interstitial ");

        gamedockAdsInterstitial.show(context);
    }

    public void requestGamedockAdsRewardVideo(final Context context, AdProvider provider, String placementId, boolean isPriority, boolean fromAdMob) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.REWARD_VIDEO);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        LoggingUtil.d("Request Gamedock Ads Reward Video");

        gamedockAdsRewardedVideo.request(context, provider, placementId, isPriority, fromAdMob);
    }

    /**
     * Method used to display a AdMob reward video.
     *
     * @param context The context of the Activity.
     */
    public void showAdMobRewardVideo(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.REWARD_VIDEO);
            return;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        LoggingUtil.d("Show Gamedock Ads Reward Video");

        gamedockAdsRewardedVideo.show(context);
    }

    public boolean isAdAvailable(final Context context, String adType) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        if (!initialised) {
            LoggingUtil.e("Gamedock Ads was not initialized. Make sure to include the gamedock-sdk-ads module and call initialize before requesting ads.");
            return false;
        }

        if (adType != null) {
            switch (adType) {
                case io.gamedock.sdk.ads.AdManager.BANNER:
                    if (gamedockAdsBanner == null) {
                        return false;
                    }
                    return (gamedockAdsBanner.isBannerEnabled) && NetworkUtil.isInternetAvailable(context);
                case io.gamedock.sdk.ads.AdManager.INTERSTITIAL:
                    if (gamedockAdsInterstitial == null || gamedockAdsInterstitial.getInterstitialAd() == null) {
                        return false;
                    }
                    boolean isAvailableBasedOnTimer = (System.currentTimeMillis() / 1000) > AdManager.getInstance().nextTimestampInterstitial;
                    return gamedockAdsInterstitial.getInterstitialAd().isLoaded() && isAvailableBasedOnTimer && NetworkUtil.isInternetAvailable(context);
                case io.gamedock.sdk.ads.AdManager.REWARD_VIDEO:
                    if (gamedockAdsRewardedVideo == null || gamedockAdsRewardedVideo.getRewardedVideoAd() == null) {
                        return false;
                    }
                    return gamedockAdsRewardedVideo.getRewardedVideoAd().isLoaded() && NetworkUtil.isInternetAvailable(context);
            }
        }

        return false;
    }

    public GamedockAdsBanner getGamedockAdsBanner() {
        return gamedockAdsBanner;
    }

    public GamedockAdsInterstitial getGamedockAdsInterstitial() {
        return gamedockAdsInterstitial;
    }

    public GamedockAdsRewardedVideo getGamedockAdsRewardedVideo() {
        return gamedockAdsRewardedVideo;
    }
}
