package io.gamedock.sdk.ads.internal;

import android.content.Context;
import android.content.Intent;

import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.core.interstitial.InterstitialAd;
import io.gamedock.sdk.ads.core.interstitial.InterstitialAdListener;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class GamedockAdsInterstitial {
    private GamedockAdsManager manager;

    private boolean isInterstitialModuleEnabled;
    public boolean isInterstitialEnabled;

    private InterstitialAd interstitialAd;
    private String staticPlacementId;
    private String videoPlacementId;
    private String interstitialRequestId;
    private int conditionId;
    private InterstitialAdListener interstitialAdListener;

    private AdProvider currentProvider;

    public void initialise(final Context context, BaseAdManagerImplementation baseAdManagerImplementation) {
        this.manager = (GamedockAdsManager) baseAdManagerImplementation;
        this.isInterstitialModuleEnabled = true;

        interstitialAdListener = new InterstitialAdListener() {
            @Override
            public void onAdLoaded() {
                LoggingUtil.d("Interstitial Ad Module: Ad loaded callback");

                isInterstitialEnabled = true;

                AdEvents.interstitialAvailable(context, conditionId);
                AdMobManager.getInstance().resetPriorityAdConfigsFillSettings(AdManager.INTERSTITIAL);
            }

            @Override
            public void onAdFailedToLoad(ErrorCodes error) {
                LoggingUtil.d("Interstitial Ad Module: Ad failed to load callback: " + error.getName());

                AdManager.getInstance().isFullScreenAdShowing = false;

                AdEvents.interstitialNotAvailable(context, conditionId);

                if (AdMobManager.getInstance().getSelectedInterstitialPriorityConfig() != null) {
                    LoggingUtil.d("Failed to load priority interstitial ad.");

                    AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().setHasFill(false);
                    AdMobManager.getInstance().requestAdMobInterstitial(context);
                } else {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
                    conditionId = 0;
                }
            }

            @Override
            public void onAdDisplayed() {
                LoggingUtil.d("Interstitial Ad Module: Ad displayed callback");

                AdManager.getInstance().isFullScreenAdShowing = true;

                LoggingUtil.d("Priority interstitial displayed");
                if (AdMobManager.getInstance().getSelectedInterstitialPriorityConfig() != null) {
                    AdEvents.interstitialDidDisplay(context, conditionId, videoPlacementId + "-" + staticPlacementId, true);
                    AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().setImpressions(AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().getImpressions() - 1);
                } else {
                    AdEvents.interstitialDidDisplay(context, conditionId, videoPlacementId + "-" + staticPlacementId, false);
                }

            }

            @Override
            public void onAdFailedToDisplay(ErrorCodes error) {
                LoggingUtil.d("Interstitial Ad Module: Ad failed to display callback: " + error.getName());

                AdManager.getInstance().isFullScreenAdShowing = false;

                if (AdMobManager.getInstance().getSelectedInterstitialPriorityConfig() != null) {
                    AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().setImpressions(AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().getImpressions() - 1);
                    AdMobManager.getInstance().requestAdMobInterstitial(context);
                }

                AdEvents.interstitialDidNotDisplay(context, conditionId);
                GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.GamedockAds, AdManager.INTERSTITIAL, "close");

                if (GamedockAdsAdActivity.activity != null) {
                    GamedockAdsAdActivity.activity.finish();
                }
            }

            @Override
            public void onAdClicked() {
                LoggingUtil.d("Interstitial Ad Module: Ad clicked callback");

                AdEvents.interstitialDidClick(context, conditionId);
            }

            @Override
            public void onAdClosed() {
                LoggingUtil.d("Interstitial Ad Module: Ad closed callback");

                AdManager.getInstance().isFullScreenAdShowing = false;

                AdEvents.interstitialDidClose(context, conditionId);
                GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.GamedockAds, AdManager.INTERSTITIAL, "close");
                conditionId = 0;
                AdMobManager.getInstance().requestAdMobInterstitial(context);
            }
        };

    }

    public void request(final Context context, AdProvider provider, String videoPlacementId, String staticPlacementId, boolean isPriority, boolean fromAdMob) {
        if (!isInterstitialModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.INTERSTITIAL)) {
            LoggingUtil.d("Requesting an interstitial while one is already showing is not allowed");
            return;
        }

        this.videoPlacementId = videoPlacementId;
        this.staticPlacementId = staticPlacementId;
        this.currentProvider = provider;
        interstitialRequestId = UUID.randomUUID().toString();

        interstitialAd = new InterstitialAd(provider, videoPlacementId, staticPlacementId, interstitialAdListener);

        AdEvents.interstitialRequested(context, conditionId, videoPlacementId + "-" + staticPlacementId, isPriority);

        interstitialAd.load();
    }

    public void show(final Context context) {
        if (!isInterstitialModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.INTERSTITIAL)) {
            LoggingUtil.d("Showing an interstitial while one is already showing is not allowed");
            return;
        }


        if (interstitialAd.isLoaded()) {
            GamedockSDK.getInstance(context).isShowingChildActivity = true;

            Intent intent = new Intent(context, GamedockAdsAdActivity.class);
            intent.putExtra("adType", AdManager.INTERSTITIAL);
            intent.putExtra("adProvider", AdManager.GamedockAds);
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

            GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.INTERSTITIAL);
            AdManager.getInstance().isFullScreenAdShowing = true;
            context.startActivity(intent);
        } else {
            LoggingUtil.d("No interstitial ad loaded. Stopping current show.");
            isInterstitialEnabled = false;
            AdEvents.interstitialNotAvailable(context, conditionId);
            conditionId = 0;
            if (AdMobManager.getInstance().getSelectedInterstitialPriorityConfig() != null) {
                AdMobManager.getInstance().getSelectedInterstitialPriorityConfig().setHasFill(false);
            }
            AdManager.getInstance().requestInterstitial(context);
        }
    }

    public InterstitialAd getInterstitialAd() {
        return interstitialAd;
    }

    public int getConditionId() {
        return conditionId;
    }

    public void setConditionId(int conditionId) {
        this.conditionId = conditionId;
    }

    public String getInterstitialRequestId() {
        return interstitialRequestId;
    }

    public AdProvider getCurrentProvider() {
        return currentProvider;
    }
}
