package io.gamedock.sdk.ads.internal;

import android.content.Context;

import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.core.banner.BannerAd;
import io.gamedock.sdk.ads.core.banner.BannerAdListener;
import io.gamedock.sdk.ads.core.banner.BannerAdPosition;
import io.gamedock.sdk.ads.core.banner.BannerAdSize;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class GamedockAdsBanner {

    private GamedockAdsManager manager;

    public boolean isBannerEnabled;
    private boolean isBannerModuleEnabled;
    private BannerAd bannerAd;
    private BannerAdListener bannerAdListener;
    private String bannerRequestId;

    private String bannerSize;
    private String bannerPosition;

    private String placementId;

    private boolean fromAdMob;

    private AdProvider currentProvider;

    public void initialise(final Context context, BaseAdManagerImplementation manager) {
        this.manager = (GamedockAdsManager) manager;
        isBannerModuleEnabled = true;

        bannerAdListener = new BannerAdListener() {
            @Override
            public void onAdLoaded() {
                LoggingUtil.d("Banner Ad Module: Ad loaded callback");

                bannerSize = null;
                bannerPosition = null;
                isBannerEnabled = true;

                if (bannerAd != null && !bannerAd.isShowing()) {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.BANNER);
                    AdEvents.bannerAvailable(context);
                    AdMobManager.getInstance().resetPriorityAdConfigsFillSettings(AdManager.BANNER);
                }
            }

            @Override
            public void onAdFailedToLoad(ErrorCodes error) {
                LoggingUtil.d("Banner Ad Module: Ad failed to load callback: " + error.getName());

                AdEvents.bannerNotAvailable(context);
                bannerRequestId = null;
                isBannerEnabled = false;

                hide(context);

                if (fromAdMob) {
                    if (AdMobManager.getInstance().getSelectedBannerPriorityConfig() != null) {
                        AdMobManager.getInstance().getSelectedBannerPriorityConfig().setHasFill(false);
                    }

                    AdMobManager.getInstance().getAdMobBanner().request(context, bannerPosition, bannerSize);
                } else {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                }
            }

            @Override
            public void onAdDisplayed() {
                LoggingUtil.d("Banner Ad Module: Ad displayed callback");
                AdEvents.bannerDidDisplay(context, placementId, true);
                GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.BANNER);

                if (fromAdMob) {
                    if (AdMobManager.getInstance().getSelectedBannerPriorityConfig() != null) {
                        AdMobManager.getInstance().getSelectedBannerPriorityConfig().setImpressions(AdMobManager.getInstance().getSelectedBannerPriorityConfig().getImpressions() - 1);
                    }
                }
            }

            @Override
            public void onAdHidden() {
                LoggingUtil.d("Banner Ad Module: Ad hidden callback");
                AdEvents.bannerDidHide(context);
            }

            @Override
            public void onAdFailedToDisplay(ErrorCodes error) {
                LoggingUtil.d("Banner Ad Module: Ad failed to display callback: " + error.getName());
                AdEvents.bannerDidNotDisplay(context);
                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            }

            @Override
            public void onAdClicked() {
                LoggingUtil.d("Banner Ad Module: Ad clicked callback");

                AdEvents.bannerDidClick(context);
            }

            @Override
            public void onAdClosed() {
                LoggingUtil.d("Banner Ad Module: Ad closed callback");
            }
        };
    }

    public void request(Context context, AdProvider provider, String placementId, String position, String adSize, boolean isPriority, boolean fromAdMob) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        try {
            bannerRequestId = UUID.randomUUID().toString();
            this.placementId = placementId;
            this.fromAdMob = fromAdMob;
            this.currentProvider = provider;
            isBannerEnabled = false;

            if (bannerAd != null) {
                bannerAd.destroy();
            }

            bannerAd = new BannerAd(provider, placementId, bannerAdListener);

            this.bannerPosition = position;
            this.bannerSize = adSize;

            BannerAdSize bannerAdSize = BannerAdSize.SMART_BANNER;

            if (adSize != null) {
                switch (adSize) {
                    case "BANNER":
                        bannerAdSize = BannerAdSize.BANNER;
                        break;
                    case "LEADERBOARD":
                        bannerAdSize = BannerAdSize.LEADERBOARD;
                        break;
                    case "FULL_BANNER":
                        bannerAdSize = BannerAdSize.FULL_BANNER;
                        break;
                    case "LARGE_BANNER":
                        bannerAdSize = BannerAdSize.LARGE_BANNER;
                        break;
                    case "MEDIUM_RECTANGLE":
                        bannerAdSize = BannerAdSize.MEDIUM_RECTANGLE;
                        break;
                    case "SMART_BANNER":
                    default:
                        break;
                }
            }

            bannerAd.setAdSize(bannerAdSize);

            BannerAdPosition bannerAdPosition = BannerAdPosition.BOTTOM;

            if (position != null) {
                switch (position) {
                    case "TOP":
                        bannerAdPosition = BannerAdPosition.TOP;
                        break;
                    case "BOTTOM":
                    default:
                        bannerAdPosition = BannerAdPosition.BOTTOM;
                        break;
                }
            }

            bannerAd.setAdPosition(bannerAdPosition);

            AdEvents.bannerRequested(context, placementId, isPriority);
            bannerAd.load();
        } catch (Exception e) {
            e.printStackTrace();
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
        }
    }


    public void show(Context context) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        if (bannerAd == null) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        if (isBannerEnabled) {
            bannerAd.show();
        } else {
            AdEvents.bannerDidNotDisplay(context);
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
        }
    }


    public void hide(Context context) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        if (bannerAd == null) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        bannerAd.hide();

    }

    public String getBannerRequestId() {
        return bannerRequestId;
    }

    public AdProvider getCurrentProvider() {
        return currentProvider;
    }

    public BannerAd getBannerAd() {
        return bannerAd;
    }
}
