package io.gamedock.sdk.ads.admob;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import androidx.annotation.NonNull;

import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdValue;
import com.google.android.gms.ads.FullScreenContentCallback;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.OnPaidEventListener;
import com.google.android.gms.ads.OnUserEarnedRewardListener;
import com.google.android.gms.ads.admanager.AdManagerAdRequest;
import com.google.android.gms.ads.rewarded.RewardItem;
import com.google.android.gms.ads.rewarded.RewardedAd;
import com.google.android.gms.ads.rewarded.RewardedAdLoadCallback;
import com.google.android.gms.ads.rewarded.ServerSideVerificationOptions;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class AdMobRewardedVideo {

    private AdMobManager adMobManager;

    private boolean isRewardVideoModuleEnabled;
    public boolean isVideoEnabled;

    private RewardedAd rewardedVideoAd;
    private String rewardVideoAdUnitId;
    private String rewardVideoRequestId;
    private JSONObject videoReward;
    private String videoLocation;
    private String videoRewardType;
    private RewardedAdLoadCallback rewardedVideoAdLoadListener;
    private FullScreenContentCallback fullScreenContentCallback;
    private OnUserEarnedRewardListener userEarnedRewardListener;
    private OnPaidEventListener rewardedVideoPaidEventListener;

    private Map<String, String> adMobRewardVideoCustomTargeting;

    public void initialise(final Context context, BaseAdManagerImplementation baseAdManagerImplementation) {
        this.adMobManager = (AdMobManager) baseAdManagerImplementation;

        rewardedVideoAdLoadListener = new RewardedAdLoadCallback() {
            @Override
            public void onAdLoaded(@NonNull RewardedAd receivedRewardedAd) {
                LoggingUtil.d("Ad is available");
                rewardedVideoAd = receivedRewardedAd;

                JSONObject serverValidationParameters = configureServerValidationParameters(context);
                ServerSideVerificationOptions serverSideVerificationOptions = new ServerSideVerificationOptions.Builder()
                        .setUserId(GamedockSDK.getInstance(context).getGamedockUID())
                        .setCustomData(serverValidationParameters.toString())
                        .build();
                rewardedVideoAd.setServerSideVerificationOptions(serverSideVerificationOptions);
                rewardedVideoAd.setFullScreenContentCallback(fullScreenContentCallback);
                rewardedVideoAd.setOnPaidEventListener(rewardedVideoPaidEventListener);

                isVideoEnabled = true;
                GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.REWARD_VIDEO);
                adMobManager.resetPriorityAdConfigsFillSettings(AdManager.REWARD_VIDEO);
                AdEvents.rewardVideoAvailable(context);
            }

            @Override
            public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
                AdManager.getInstance().isFullScreenAdShowing = false;

                LoggingUtil.d("No ad available: reward video. Error: " + loadAdError.toString());

                AdEvents.rewardVideoNotAvailable(context);

                if (adMobManager.getSelectedRewardVideoPriorityConfig() != null) {
                    LoggingUtil.d("Failed to load priority rewarded video ad.");

                    adMobManager.getSelectedRewardVideoPriorityConfig().setHasFill(false);
                    adMobManager.requestAdMobRewardVideo(context);
                } else {
                    isVideoEnabled = false;
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
                }
            }
        };

        fullScreenContentCallback = new FullScreenContentCallback() {
            @Override
            public void onAdShowedFullScreenContent() {
                AdManager.getInstance().isFullScreenAdShowing = true;

                if (adMobManager.getSelectedRewardVideoPriorityConfig() == null) {
                    LoggingUtil.d("Default rewarded video displayed");
                    AdEvents.rewardVideoDidDisplay(context, rewardVideoAdUnitId, false);
                } else {
                    LoggingUtil.d("Priority rewarded video displayed");
                    AdEvents.rewardVideoDidDisplay(context, adMobManager.getSelectedRewardVideoPriorityConfig().getAdUnitId(), true);
                    adMobManager.getSelectedRewardVideoPriorityConfig().setImpressions(adMobManager.getSelectedRewardVideoPriorityConfig().getImpressions() - 1);
                }
            }

            @Override
            public void onAdFailedToShowFullScreenContent(AdError adError) {
                LoggingUtil.d("Ad failed to display with error code: " + adError.toString());
                AdEvents.rewardVideoDidNotDisplay(context);
                if (AdMobAdActivity.activity != null) {
                    AdMobAdActivity.activity.finish();
                }
            }

            @Override
            public void onAdDismissedFullScreenContent() {
                AdManager.getInstance().isFullScreenAdShowing = false;

                rewardedVideoAd = null;

                if (videoReward == null) {
                    AdEvents.rewardVideoDidDismiss(context);
                    GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.AdMob, AdManager.REWARD_VIDEO, "dismiss");
                } else {
                    AdEvents.rewardVideoDidClose(context);
                    GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.AdMob, AdManager.REWARD_VIDEO, "close");
                }
            }

            @Override
            public void onAdImpression() {
                super.onAdImpression();
            }
        };

        userEarnedRewardListener = new OnUserEarnedRewardListener() {
            @Override
            public void onUserEarnedReward(@NonNull RewardItem rewardItem) {
                videoReward = new JSONObject();
                try {
                    videoReward.put("currencyName", rewardItem.getType());
                    videoReward.put("currencyId", rewardItem.getType());
                    videoReward.put("reward", String.valueOf(rewardItem.getAmount()));
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        };

        rewardedVideoPaidEventListener = new OnPaidEventListener() {
            @Override
            public void onPaidEvent(AdValue adValue) {
                AdEvents.rewardVideoDidMonetize(context, adValue.getCurrencyCode(), adValue.getPrecisionType(), adValue.getValueMicros());
            }
        };
    }

    public void start(final Context context, final String adUnitId, final Map<String, String> adMobCustomTargeting, AdMobMediationNetworks adMobMediationNetworks) {
        try {
            adMobRewardVideoCustomTargeting = adMobCustomTargeting;
            rewardVideoAdUnitId = adUnitId;
            isRewardVideoModuleEnabled = true;
        } catch (NoClassDefFoundError | Exception e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
        }
    }

    public void request(final Context context) {
        if (!isRewardVideoModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.REWARD_VIDEO)) {
            LoggingUtil.d("Requesting a rewarded video while one is already showing is not allowed");
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            LoggingUtil.e("Failed to request AdMob Reward Video. Context doesn't belong to an activity.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        try {
            adMobManager.getHandler().post(new Runnable() {
                @Override
                public void run() {
                    try {
                        String adUnitId = "";
                        String displayNetwork = "";
                        boolean isPriority = false;

                        AdRequest adRequest = null;
                        AdManagerAdRequest adManagerAdRequest = null;

                        adMobManager.setSelectedRewardVideoPriorityConfig(adMobManager.getPriorityAdConfig(AdManager.REWARD_VIDEO));

                        if (adMobManager.getSelectedRewardVideoPriorityConfig() == null) {
                            LoggingUtil.d("No priority rewarded video configuration found. Using default configuration");
                            adMobManager.setSelectedRewardVideoPriorityConfig(null);
                            adUnitId = rewardVideoAdUnitId;
                            adRequest = adMobManager.getAdMobAdRequest(context, adMobRewardVideoCustomTargeting);

                            displayNetwork = AdManager.AdMob;
                        } else if (adMobManager.getSelectedRewardVideoPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                            LoggingUtil.d("Priority rewarded video configuration found for AdMob. AdUnitId: " + adMobManager.getSelectedRewardVideoPriorityConfig().getAdUnitId());
                            adUnitId = adMobManager.getSelectedRewardVideoPriorityConfig().getAdUnitId();
                            adRequest = adMobManager.getAdMobAdRequest(context, adMobRewardVideoCustomTargeting);

                            displayNetwork = AdManager.AdMob;
                            isPriority = true;
                        } else if (adMobManager.getSelectedRewardVideoPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                            LoggingUtil.d("Priority rewarded video configuration found for AdManager. AdUnitId: " + adMobManager.getSelectedRewardVideoPriorityConfig().getAdUnitId());
                            adUnitId = adMobManager.getSelectedRewardVideoPriorityConfig().getAdUnitId();
                            adManagerAdRequest = adMobManager.getAdManagerAdRequest(context, adMobRewardVideoCustomTargeting);

                            displayNetwork = AdManager.AdManager;
                            isPriority = true;
                        } else if (adMobManager.getSelectedRewardVideoPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                            LoggingUtil.d("Priority rewarded video configuration found for Improve Digital. Placement Id: " + adMobManager.getSelectedRewardVideoPriorityConfig().getVideoPlacementId());
                            GamedockAdsManager.getInstance().requestGamedockAdsRewardVideo(context, AdProvider.IMPROVE_DIGITAL, adMobManager.getSelectedRewardVideoPriorityConfig().getVideoPlacementId(), true, true);

                            return;
                        }

                        final String finalDisplayNetwork = displayNetwork;
                        final AdRequest finalAdRequest = adRequest;
                        final AdManagerAdRequest finalAdManagerAdRequest = adManagerAdRequest;
                        final String finalAdUnitId = adUnitId;
                        final boolean finalIsPriority = isPriority;

                        activity.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (rewardedVideoAd != null) {
                                    LoggingUtil.d("Ad is available");
                                    isVideoEnabled = true;
                                    GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.REWARD_VIDEO);
                                } else {
                                    if (finalDisplayNetwork.equals(AdManager.AdMob)) {
                                        RewardedAd.load(context, finalAdUnitId, finalAdRequest, rewardedVideoAdLoadListener);
                                        AdEvents.rewardedVideoRequested(context, finalAdUnitId, finalIsPriority);
                                    } else if (finalDisplayNetwork.equals(AdManager.AdManager)) {
                                        RewardedAd.load(context, finalAdUnitId, finalAdManagerAdRequest, rewardedVideoAdLoadListener);
                                        AdEvents.rewardedVideoRequested(context, finalAdUnitId, finalIsPriority);
                                    } else {
                                        GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
                                    }
                                }
                            }
                        });
                    } catch (NoClassDefFoundError | Exception e) {
                        e.printStackTrace();
                        LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                    }
                }
            });
        } catch (NoClassDefFoundError | Exception e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
        }
    }

    public void show(final Context context) {
        if (!isRewardVideoModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.REWARD_VIDEO)) {
            LoggingUtil.d("Showing a rewarded video while one is already showing is not allowed");
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            LoggingUtil.e("Failed to show AdMob Reward Video. Context doesn't belong to an activity.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
            return;
        }

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    if (adMobManager.getSelectedRewardVideoPriorityConfig() != null) {
                        if (adMobManager.getSelectedRewardVideoPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                            GamedockAdsManager.getInstance().showAdMobRewardVideo(context);

                            return;
                        }
                    }

                    if (rewardedVideoAd != null) {
                        LoggingUtil.d("Opening AdMob Reward Video Activity");
                        AdManager.getInstance().isFullScreenAdShowing = true;
                        //Special case for Unity Ads (Unity Ads cannot be shown if it is not from the original activity with which AdMob was initialized)
                        if (rewardedVideoAd != null && rewardedVideoAd.getResponseInfo() != null && rewardedVideoAd.getResponseInfo().getMediationAdapterClassName() != null && rewardedVideoAd.getResponseInfo().getMediationAdapterClassName().equals("com.google.ads.mediation.unity.UnityAdapter")) {
                            rewardedVideoAd.show(activity, userEarnedRewardListener);
                        } else {
                            GamedockSDK.getInstance(context).isShowingChildActivity = true;

                            Intent intent = new Intent(context, AdMobAdActivity.class);
                            intent.putExtra("adType", AdManager.REWARD_VIDEO);
                            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                            GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.REWARD_VIDEO);

                            context.startActivity(intent);
                        }
                    } else {
                        GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
                        AdEvents.rewardVideoDidNotDisplay(context);
                        if (AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig() != null) {
                            AdMobManager.getInstance().getSelectedRewardVideoPriorityConfig().setHasFill(false);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    AdEvents.rewardVideoDidNotDisplay(context);
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.REWARD_VIDEO);
                }
            }
        });
    }

    private JSONObject configureServerValidationParameters(Context context) {
        JSONObject customData = new JSONObject();
        try {
            if (!GamedockSDK.getInstance(context).isCoppaEnabled()) {
                customData.put("deviceId", GamedockSDK.getInstance(context).getDeviceId());
            }
            customData.put("uid", GamedockSDK.getInstance(context).getGamedockUID());
            customData.put("packageName", GamedockSDK.getInstance(context).getApplicationPackageName());
            customData.put("os", "Android");

            JSONArray externalIdsList = new JSONArray();
            for (Map.Entry<String, String> entry : GamedockSDK.getInstance(context).externalIds.entrySet()) {
                JSONObject externalIdObject = new JSONObject();
                externalIdObject.put("provider", entry.getKey());
                externalIdObject.put("id", entry.getValue());

                externalIdsList.put(externalIdObject);
            }
            customData.put("externalIds", externalIdsList);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return customData;
    }

    public String getRewardVideoRequestId() {
        return rewardVideoRequestId;
    }

    public void setRewardVideoRequestId(String rewardVideoRequestId) {
        this.rewardVideoRequestId = rewardVideoRequestId;
    }

    public String getRewardVideoAdUnitId() {
        return rewardVideoAdUnitId;
    }

    public Map<String, String> getAdMobRewardVideoCustomTargeting() {
        return adMobRewardVideoCustomTargeting;
    }


    public void setAdMobRewardVideoCustomTargeting(Map<String, String> adMobRewardVideoCustomTargeting) {
        this.adMobRewardVideoCustomTargeting = adMobRewardVideoCustomTargeting;
    }

    public JSONObject getVideoReward() {
        return videoReward;
    }

    public void setVideoReward(JSONObject videoReward) {
        this.videoReward = videoReward;
    }

    public String getVideoLocation() {
        return videoLocation;
    }

    public void setVideoLocation(String videoLocation) {
        this.videoLocation = videoLocation;
    }

    public String getVideoRewardType() {
        return videoRewardType;
    }

    public void setVideoRewardType(String videoRewardType) {
        this.videoRewardType = videoRewardType;
    }

    public RewardedAd getRewardedVideoAd() {
        return rewardedVideoAd;
    }

    public OnUserEarnedRewardListener getUserEarnedRewardListener() {
        return userEarnedRewardListener;
    }
}
