package io.gamedock.sdk.ads.admob;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;

import com.adcolony.sdk.AdColony;
import com.adcolony.sdk.AdColonyAppOptions;
import com.applovin.sdk.AppLovinPrivacySettings;
import com.facebook.ads.AdSettings;
import com.fyber.inneractive.sdk.external.InneractiveAdManager;
import com.google.ads.mediation.adcolony.AdColonyMediationAdapter;
import com.google.ads.mediation.inmobi.InMobiConsent;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdView;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.RequestConfiguration;
import com.google.android.gms.ads.admanager.AdManagerAdRequest;
import com.google.android.gms.ads.admanager.AdManagerAdView;
import com.google.android.gms.ads.admanager.AdManagerInterstitialAd;
import com.google.android.gms.ads.initialization.AdapterStatus;
import com.google.android.gms.ads.initialization.InitializationStatus;
import com.google.android.gms.ads.initialization.OnInitializationCompleteListener;
import com.google.android.gms.ads.interstitial.InterstitialAd;
import com.google.android.gms.ads.mediation.admob.AdMobExtras;
import com.google.android.gms.ads.rewarded.RewardedAd;
import com.inmobi.sdk.InMobiSdk;
import com.ironsource.mediationsdk.IronSource;
import com.my.target.ads.MyTargetView;
import com.my.target.common.MyTargetPrivacy;
import com.tapjoy.TJPrivacyPolicy;
import com.tapjoy.Tapjoy;
import com.unity3d.ads.UnityAds;
import com.unity3d.ads.metadata.MetaData;
import com.vungle.mediation.VungleConsent;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.models.ads.internal.MediationInfo;
import io.gamedock.sdk.models.ads.internal.PriorityAdConfig;
import io.gamedock.sdk.utils.device.NetworkUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;

import static com.google.android.gms.ads.RequestConfiguration.TAG_FOR_UNDER_AGE_OF_CONSENT_FALSE;
import static com.google.android.gms.ads.RequestConfiguration.TAG_FOR_UNDER_AGE_OF_CONSENT_TRUE;

//import com.chartboost.sdk.Chartboost;
//import com.google.ads.mediation.chartboost.ChartboostAdapter;

/**
 * Utility class used for requesting and displaying AdMob ads.
 * Used for showing AdMob interstitials and reward videos.
 * Displays the ads in a container Activity {@link AdMobAdActivity}.
 */
public class AdMobManager extends BaseAdManagerImplementation {
    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "ads";

    private static volatile AdMobManager mInstance;
    public boolean initialised;

    private AdMobBanner adMobBanner = new AdMobBanner();
    private AdMobInterstitial adMobInterstitial = new AdMobInterstitial();
    private AdMobRewardedVideo adMobRewardedVideo = new AdMobRewardedVideo();

    private ArrayList<PriorityAdConfig> priorityAdConfigs = new ArrayList<>();

    private ArrayList<MediationInfo> mediationInfoList = new ArrayList<>();

    private PriorityAdConfig selectedBannerPriorityConfig;
    private PriorityAdConfig selectedInterstitialPriorityConfig;
    private PriorityAdConfig selectedRewardVideoPriorityConfig;

    private Handler mainHandler;

    private HandlerThread handlerThread;
    private Handler handler;

    public static AdMobManager getInstance() {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new AdMobManager();
                }
            }
        }
        return mInstance;
    }

    public AdMobManager() {
        mainHandler = new Handler(Looper.getMainLooper());

        handlerThread = new HandlerThread("AdMobHandlerThread");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());
    }

    public InterstitialAd getInterstitialAd() {
        return adMobInterstitial.getInterstitialAd();
    }

    public AdManagerInterstitialAd getAdManagerInterstitialAd() {
        return adMobInterstitial.getAdManagerInterstitialAd();
    }

    public AdView getBannerAd() {
        return adMobBanner.getBannerAd();
    }

    public AdManagerAdView getAdManagerBannerAd() {
        return adMobBanner.getAdManagerBannerAd();
    }

    public RewardedAd getRewardedVideoAd() {
        return adMobRewardedVideo.getRewardedVideoAd();
    }

    /**
     * Method used to initialise the AdMob SDK.
     *
     * @param context The context of the Activity.
     * @param appId   The AdMob appId received from the Gamedock backend.
     */
    public void initialise(final Context context, final String appId, final String bannerAdUnitId, final String interstitialAdUnitId, final Map<String, String> interstitialCustomTargeting, final String rewardVideoAdUnitId, final Map<String, String> rewardVideoCustomTargeting, final AdMobMediationNetworks adMobMediationNetworks, final ArrayList<PriorityAdConfig> priorityAdConfigs) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (!initialised && appId != null && context != null) {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    try {
                        MobileAds.initialize(context, new OnInitializationCompleteListener() {
                            @Override
                            public void onInitializationComplete(InitializationStatus initializationStatus) {
                                for (Map.Entry<String, AdapterStatus> adapter : initializationStatus.getAdapterStatusMap().entrySet()) {
                                    MediationInfo mediationInfo = new MediationInfo(adapter.getKey(), adapter.getValue().getLatency(), adapter.getValue().getInitializationState().toString());
                                    mediationInfoList.add(mediationInfo);
                                }

                                if (GamedockAds.getInstance() != null && GamedockAds.getInstance().isInitialized()) {
                                    for (int i = 0; i < GamedockAds.getInstance().getInitializationStatus().size(); i++) {
                                        MediationInfo mediationInfo = new MediationInfo(GamedockAds.getInstance().getInitializationStatus().get(i).getAdProvider().toString(), 0, GamedockAds.getInstance().getInitializationStatus().get(i).getStatus().toString());
                                        mediationInfoList.add(mediationInfo);
                                    }
                                }

                                LoggingUtil.d("AdMob initialization mediations status: " + GamedockSDK.getInstance(context).getGson().toJson(mediationInfoList));

                                initialised = true;

                                if (priorityAdConfigs != null) {
                                    AdMobManager.getInstance().priorityAdConfigs.addAll(priorityAdConfigs);
                                }

                                adMobBanner.initialise(context, getInstance());
                                adMobInterstitial.initialise(context, getInstance());
                                adMobRewardedVideo.initialise(context, getInstance());

                                if (!(context instanceof Activity)) {
                                    return;
                                }

                                AdEvents.advertisementInitialized(context, mediationInfoList);
                                GamedockSDK.getInstance(context).getAdCallbacks().AdsInitialized();

                                if (bannerAdUnitId != null) {
                                    AdMobManager.getInstance().startAdMobBanner(context, bannerAdUnitId);
                                } else {
                                    LoggingUtil.d("No ad unit id found for AdMob banner. Skipping AdMob banner initialization!");
                                }

                                if (interstitialAdUnitId != null) {
                                    AdMobManager.getInstance().startAdMobInterstitial(context, interstitialAdUnitId, interstitialCustomTargeting, adMobMediationNetworks);
                                } else {
                                    LoggingUtil.d("No ad unit id found for AdMob interstitial. Skipping AdMob interstitial initialization!");
                                }

                                if (rewardVideoAdUnitId != null) {
                                    AdMobManager.getInstance().startAdMobRewardVideo(context, rewardVideoAdUnitId, rewardVideoCustomTargeting);
                                } else {
                                    LoggingUtil.d("No ad unit id found for AdMob reward video. Skipping AdMob reward video initialization!");
                                }
                            }
                        });
                    } catch (NoClassDefFoundError e) {
                        LoggingUtil.e("AdMob dependency not found. AdMob module will not initialize!");
                    }
                }
            });
        } else {
            LoggingUtil.d("AdMob SDK already initialised! Skipping initialisation.");
        }
    }

    /**
     * Method used to initialise the AdMob banner module.
     * Creates a bannerAd view that can later be used to show or hide banner ads.
     *
     * @param context  The context of the Activity.
     * @param adUnitId The adUnitId for the AdMob banner.
     */
    public void startAdMobBanner(final Context context, String adUnitId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        LoggingUtil.d("Start AdMob Banner");

        adMobBanner.start(context, adUnitId, null, null);
    }

    /**
     * Method used to request a banner ad from AdMob.
     * If position or adSize are different from the default, the bannerAd view will be recreated with the new specification.
     *
     * @param context  The context of the Activity.
     * @param position The position of the banner. Can be TOP or BOTTOM.
     * @param adSize   The ad size of the banner. Can be: BANNER, LEADERBOARD, FULL_BANNER, LARGE_BANNER, SMART_BANNER, MEDIUM_RECTANGLE.
     */
    public void requestAdMobBanner(final Context context, String position, final String adSize) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        if (!initialised) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Request AdMob Banner");

        adMobBanner.request(context, position, adSize);
    }

    /**
     * Method used to show the banner ad.
     * The visibility of the banner ad object is toggled on.
     *
     * @param context The context of the Activity.
     */
    public void showAdMobBanner(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Show AdMob Banner");

        adMobBanner.show(context);
    }

    /**
     * Method used to hide the banner ad.
     * The visibility of the banner ad object is toggled off.
     *
     * @param context The context of the Activity.
     */
    public void hideAdMobBanner(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.BANNER);
            return;
        }

        LoggingUtil.d("Hide AdMob Banner");

        adMobBanner.hide(context);
    }

    /**
     * Method used to initialise AdMob.
     * This method is called internally when a correct configuration is received from the Gamedock Backend.
     * Requires an "adUnityId".
     * Registers the listeners for the AdMob.
     *
     * @param context                          The activity context.
     * @param adUnitId                         The id for the specific application. It is configured in the Gamedock Console.
     * @param adMobInterstitialCustomTargeting The defined custom targeting.
     */
    public void startAdMobInterstitial(final Context context, String adUnitId, Map<String, String> adMobInterstitialCustomTargeting, AdMobMediationNetworks adMobMediationNetworks) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        LoggingUtil.d("Start AdMob Interstitial");

        adMobInterstitial.start(context, adUnitId, adMobInterstitialCustomTargeting, adMobMediationNetworks);
    }

    /**
     * Method used to request a AdMob interstitial ad.
     */
    public void requestAdMobInterstitial(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        if (!initialised) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        LoggingUtil.d("Request AdMob Interstitial");

        adMobInterstitial.request(context);
    }

    /**
     * Method used to display a AdMob ad.
     * Uses the {@link AdMobAdActivity} as container.
     *
     * @param context The activity context.
     */
    public void showAdMobInterstitial(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.INTERSTITIAL);
            return;
        }

        LoggingUtil.d("Show AdMob Interstitial ");

        adMobInterstitial.show(context);
    }

    /**
     * Method used to initialise the reward video module for AdMob.
     *
     * @param context         The context of the Activity.
     * @param adUnitId        The ad unit id used for the reward video module of AdMob.
     * @param customTargeting Custom Targeting map used to pass specific information to AdMob, as received from the Gamedock backend.
     */
    public void startAdMobRewardVideo(final Context context, final String adUnitId, final Map<String, String> customTargeting) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        LoggingUtil.d("Start AdMob Reward Video");

        adMobRewardedVideo.start(context, adUnitId, customTargeting, null);
    }

    /**
     * Method used to request a reward video from the AdMob servers.
     *
     * @param context The context of the Activity.
     */
    public void requestAdMobRewardVideo(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.REWARD_VIDEO);
            return;
        }

        if (!initialised) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.REWARD_VIDEO);
            return;
        }

        LoggingUtil.d("Request AdMob Reward Video");

        adMobRewardedVideo.request(context);
    }

    /**
     * Method used to display a AdMob reward video.
     *
     * @param context The context of the Activity.
     */
    public void showAdMobRewardVideo(final Context context) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(io.gamedock.sdk.ads.AdManager.REWARD_VIDEO);
            return;
        }

        LoggingUtil.d("Show AdMob Reward Video");

        adMobRewardedVideo.show(context);
    }

    public AdRequest getAdMobAdRequest(Context context, Map<String, String> customTargeting) {
        Bundle bundle = new Bundle();
        final AdRequest adRequest;
        final RequestConfiguration requestConfiguration;
        final AdRequest.Builder builder = new AdRequest.Builder();
        final RequestConfiguration.Builder requestConfigurationBuilder = MobileAds.getRequestConfiguration().toBuilder();

        boolean withPersonalisedAds = AdManager.getInstance().checkPersonalisedAdsPermission(context);

        configureMediationConsent(context, withPersonalisedAds);

        if (customTargeting == null) {
            customTargeting = new HashMap<>();
        }

        customTargeting.remove("npa");

        if (withPersonalisedAds) {
            customTargeting.put("npa", "0");
        } else {
            customTargeting.put("npa", "1");
            customTargeting.put("rdp", "1");
        }

        for (Map.Entry<String, String> entry : customTargeting.entrySet()) {
            bundle.putString(entry.getKey(), entry.getValue());
        }

        builder.addNetworkExtras(new AdMobExtras(bundle));

//        Bundle cbBundle = new ChartboostAdapter.ChartboostExtrasBundleBuilder()
//                .setFramework(Chartboost.CBFramework.CBFrameworkOther, "4.2.0")
//                .build();
//        builder.addNetworkExtrasBundle(ChartboostAdapter.class, cbBundle);

        if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
            requestConfigurationBuilder.setTagForChildDirectedTreatment(TAG_FOR_UNDER_AGE_OF_CONSENT_TRUE);
        } else {
            requestConfigurationBuilder.setTagForChildDirectedTreatment(TAG_FOR_UNDER_AGE_OF_CONSENT_FALSE);
        }

        if (AdManager.testDeviceId != null) {
            requestConfigurationBuilder.setTestDeviceIds(Collections.singletonList(AdManager.testDeviceId));
            configureMediationTestMode();
        }

        requestConfiguration = requestConfigurationBuilder.build();
        MobileAds.setRequestConfiguration(requestConfiguration);
        adRequest = builder.build();
        return adRequest;
    }

    public AdManagerAdRequest getAdManagerAdRequest(Context context, Map<String, String> customTargeting) {
        final AdManagerAdRequest adRequest;
        final RequestConfiguration requestConfiguration;
        final AdManagerAdRequest.Builder builder = new AdManagerAdRequest.Builder();
        final RequestConfiguration.Builder requestConfigurationBuilder = MobileAds.getRequestConfiguration().toBuilder();

        boolean withPersonalisedAds = AdManager.getInstance().checkPersonalisedAdsPermission(context);

        configureMediationConsent(context, withPersonalisedAds);

        if (customTargeting == null) {
            customTargeting = new HashMap<>();
        }

        customTargeting.remove("npa");

        if (withPersonalisedAds) {
            customTargeting.put("npa", "0");
        } else {
            customTargeting.put("npa", "1");
            customTargeting.put("rdp", "1");
        }

        for (Map.Entry<String, String> entry : customTargeting.entrySet()) {
            builder.addCustomTargeting(entry.getKey(), entry.getValue());

        }

        if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
            requestConfigurationBuilder.setTagForChildDirectedTreatment(TAG_FOR_UNDER_AGE_OF_CONSENT_TRUE);
        } else {
            requestConfigurationBuilder.setTagForChildDirectedTreatment(TAG_FOR_UNDER_AGE_OF_CONSENT_FALSE);
        }

        if (AdManager.testDeviceId != null) {
            requestConfigurationBuilder.setTestDeviceIds(Collections.singletonList(AdManager.testDeviceId));
            configureMediationTestMode();
        }

        requestConfiguration = requestConfigurationBuilder.build();
        MobileAds.setRequestConfiguration(requestConfiguration);
        adRequest = builder.build();
        return adRequest;
    }

    public boolean isAdAvailable(final Context context, String adType) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return false;
        }

        Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            return false;
        }

        if (adType != null) {
            switch (adType) {
                case io.gamedock.sdk.ads.AdManager.BANNER:
                    return (adMobBanner.isBannerEnabled ) && NetworkUtil.isInternetAvailable(context);
                case io.gamedock.sdk.ads.AdManager.INTERSTITIAL:
                    if (adMobInterstitial.getInterstitialAd() != null) {
                        try {
                            final boolean[] isLoaded = {false};
                            final CountDownLatch latch = new CountDownLatch(1);
                            activity.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    try {
                                        boolean isAvailableBasedOnTimer = (System.currentTimeMillis() / 1000) > AdManager.getInstance().nextTimestampInterstitial;
                                        isLoaded[0] = (adMobInterstitial.getInterstitialAd() != null || adMobInterstitial.getAdManagerInterstitialAd() != null) && isAvailableBasedOnTimer;
                                    } catch (NoClassDefFoundError | Exception e) {
                                        LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                                    }
                                    latch.countDown();
                                }
                            });
                            latch.await(5, TimeUnit.SECONDS);

                            return isLoaded[0] && NetworkUtil.isInternetAvailable(context);
                        } catch (NoClassDefFoundError | Exception e) {
                            e.printStackTrace();
                            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                            return false;
                        }
                    } else {
                        return false;
                    }
                case io.gamedock.sdk.ads.AdManager.REWARD_VIDEO:
                    if (adMobRewardedVideo.getRewardedVideoAd() != null) {
                        try {
                            final boolean[] isLoaded = {false};
                            final CountDownLatch latch = new CountDownLatch(1);
                            activity.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    try {
                                        isLoaded[0] = adMobRewardedVideo.getRewardedVideoAd().isLoaded();
                                    } catch (NoClassDefFoundError | Exception e) {
                                        LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                                    }
                                    latch.countDown();
                                }
                            });
                            latch.await(5, TimeUnit.SECONDS);

                            return isLoaded[0] && NetworkUtil.isInternetAvailable(context);
                        } catch (NoClassDefFoundError | Exception e) {
                            e.printStackTrace();
                            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                            return false;
                        }
                    } else {
                        return false;
                    }
            }
        }

        return false;
    }

    /**
     * Method used to configure the GDPR consent for the various ad mediation networks that are implemented in the SDK.
     *
     * @param context             The context of the Activity.
     * @param withPersonalisedAds Describes if personalised ads should be displayed by the mediation networks.
     */
    private void configureMediationConsent(Context context, boolean withPersonalisedAds) {
        try {
            //Unity (COPPA is set in dashboard)
            MetaData gdprData = new MetaData(context);
            gdprData.set("gdpr.consent", withPersonalisedAds);
            gdprData.commit();

            //AdColony
            String adColonyConsentString = withPersonalisedAds ? "1" : "0";
            AdColonyAppOptions appOptions = AdColonyMediationAdapter.getAppOptions();
            appOptions.setPrivacyFrameworkRequired(AdColonyAppOptions.GDPR, true);
            appOptions.setPrivacyConsentString(AdColonyAppOptions.GDPR, adColonyConsentString);
            appOptions.setPrivacyFrameworkRequired(AdColonyAppOptions.CCPA, true);
            appOptions.setPrivacyConsentString(AdColonyAppOptions.CCPA, adColonyConsentString);
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                appOptions.setPrivacyFrameworkRequired(AdColonyAppOptions.CCPA, true);
            }

            //Vungle (COPPA is set in dashboard)
            if (withPersonalisedAds) {
                VungleConsent.updateConsentStatus(com.vungle.warren.Vungle.Consent.OPTED_IN, "1.0.0");
            } else {
                VungleConsent.updateConsentStatus(com.vungle.warren.Vungle.Consent.OPTED_OUT, "1.0.0");
            }

            //AppLovin
            AppLovinPrivacySettings.setHasUserConsent(withPersonalisedAds, context);
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                AppLovinPrivacySettings.setIsAgeRestrictedUser(true, context);
            }

            //Facebook
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                AdSettings.setMixedAudience(true);
            }

            //InMobi
            JSONObject consentObject = new JSONObject();
            try {
                consentObject.put(InMobiSdk.IM_GDPR_CONSENT_AVAILABLE, withPersonalisedAds);
                consentObject.put("gdpr", "1");
            } catch (JSONException exception) {
                exception.printStackTrace();
            }
            InMobiConsent.updateGDPRConsent(consentObject);
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                InMobiSdk.setAgeGroup(InMobiSdk.AgeGroup.BELOW_18);
            }

            //Tapjoy
            String tapjoyConsentString = withPersonalisedAds ? "1" : "0";
            TJPrivacyPolicy.getInstance().setUserConsent(tapjoyConsentString);
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                TJPrivacyPolicy.getInstance().setBelowConsentAge(true);
            }

            //Ironsource
            IronSource.setConsent(withPersonalisedAds);

            //MyTarget
            MyTargetPrivacy.setUserConsent(withPersonalisedAds);
            if (GamedockSDK.getInstance(context).isCoppaEnabled()) {
                MyTargetPrivacy.setUserAgeRestricted(true);
            }

            //Chartboost
//            DataUseConsent chartboostDataUseConsent;
//            if (withPersonalisedAds) {
//                chartboostDataUseConsent = new GDPR(GDPR.GDPR_CONSENT.BEHAVIORAL);
//            } else {
//                chartboostDataUseConsent = new GDPR(GDPR.GDPR_CONSENT.NON_BEHAVIORAL);
//            }
//            Chartboost.addDataUseConsent(context, chartboostDataUseConsent);

            //Fyber
            InneractiveAdManager.setGdprConsent(withPersonalisedAds);

            //Ogury Not able to set consent at this point
//            Integer[] nonIabVendorsAccepted = new Integer[0];
//            OguryChoiceManagerExternal.TcfV2.setConsent(context, "ASSET_KEY", "NO-IAB-STRING-GENERATED", nonIabVendorsAccepted);
        } catch (NoClassDefFoundError e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
        }

    }

    private void configureMediationTestMode() {
        //Unity
        UnityAds.setDebugMode(true);

        //AdColony
        AdColony.getAppOptions().setTestModeEnabled(true);

        //Vungle
        //Configured in the Vungle console

        //AppLovin
        //Configured in the AppLovin console

        //Facebook
        AdSettings.setTestMode(true);

        //InMobi
        //Configured in the InMobi console

        //Tapjoy
        Tapjoy.setDebugEnabled(true);

        //Ironsource
        //Configured in the Ironsource console

        //MyTarget
        MyTargetView.setDebugMode(true);

        //Chartboost
        //Configured in the Chartboost console

        //Fyber
        //Configured in the Fyber console
    }

    public PriorityAdConfig getPriorityAdConfig(String adType) {
        int highestPriority = 0;
        PriorityAdConfig selectedConfig = null;
        for (PriorityAdConfig priorityAdConfig : priorityAdConfigs) {
            if (priorityAdConfig.getAdType().equals(adType) && priorityAdConfig.hasFill()) {
                if (highestPriority < priorityAdConfig.getPriority() && priorityAdConfig.getImpressions() > 0) {
                    highestPriority = priorityAdConfig.getPriority();
                    selectedConfig = priorityAdConfig;
                }
            }
        }

        return selectedConfig;
    }

    public void resetPriorityAdConfigsFillSettings(String adType) {
        for (PriorityAdConfig priorityAdConfig : priorityAdConfigs) {
            if (priorityAdConfig.getAdType().equals(adType)) {
                priorityAdConfig.setHasFill(true);
            }
        }
    }

    public PriorityAdConfig getSelectedBannerPriorityConfig() {
        return selectedBannerPriorityConfig;
    }

    public void setSelectedBannerPriorityConfig(PriorityAdConfig selectedBannerPriorityConfig) {
        this.selectedBannerPriorityConfig = selectedBannerPriorityConfig;
    }

    public PriorityAdConfig getSelectedInterstitialPriorityConfig() {
        return selectedInterstitialPriorityConfig;
    }

    public void setSelectedInterstitialPriorityConfig(PriorityAdConfig selectedInterstitialPriorityConfig) {
        this.selectedInterstitialPriorityConfig = selectedInterstitialPriorityConfig;
    }

    public PriorityAdConfig getSelectedRewardVideoPriorityConfig() {
        return selectedRewardVideoPriorityConfig;
    }

    public void setSelectedRewardVideoPriorityConfig(PriorityAdConfig selectedRewardVideoPriorityConfig) {
        this.selectedRewardVideoPriorityConfig = selectedRewardVideoPriorityConfig;
    }

    public Handler getMainHandler() {
        return mainHandler;
    }

    public HandlerThread getHandlerThread() {
        return handlerThread;
    }

    public Handler getHandler() {
        return handler;
    }

    public AdMobBanner getAdMobBanner() {
        return adMobBanner;
    }

    public AdMobInterstitial getAdMobInterstitial() {
        return adMobInterstitial;
    }

    public AdMobRewardedVideo getAdMobRewardedVideo() {
        return adMobRewardedVideo;
    }

    public void resetAdMobManager(Context context) {
        hideAdMobBanner(context);
    }

    public static boolean isTablet(Context context) {
        return (context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) >= Configuration.SCREENLAYOUT_SIZE_LARGE;
    }
}
