package io.gamedock.sdk.ads.admob;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.util.Log;

import androidx.annotation.NonNull;

import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdValue;
import com.google.android.gms.ads.FullScreenContentCallback;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.OnPaidEventListener;
import com.google.android.gms.ads.admanager.AdManagerAdRequest;
import com.google.android.gms.ads.admanager.AdManagerInterstitialAd;
import com.google.android.gms.ads.admanager.AdManagerInterstitialAdLoadCallback;
import com.google.android.gms.ads.interstitial.InterstitialAd;
import com.google.android.gms.ads.interstitial.InterstitialAdLoadCallback;

import java.util.Map;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class AdMobInterstitial {

    private AdMobManager adMobManager;

    private boolean isInterstitialModuleEnabled;
    public boolean isInterstitialEnabled;

    private AdManagerInterstitialAd adManagerInterstitialAd;
    private InterstitialAd interstitialAd;
    private String interstitialAdUnitId;
    private String interstitialRequestId;
    private int conditionId;
    private InterstitialAdLoadCallback interstitialAdLoadCallback;
    private AdManagerInterstitialAdLoadCallback adManagerInterstitialAdLoadCallback;
    private FullScreenContentCallback fullScreenContentCallback;
    private OnPaidEventListener interstitialPaidEventListener;

    private Map<String, String> adMobInterstitialCustomTargeting;

    public void initialise(final Context context, BaseAdManagerImplementation baseAdManagerImplementation) {
        this.adMobManager = (AdMobManager) baseAdManagerImplementation;

        interstitialAdLoadCallback = new InterstitialAdLoadCallback() {
            @Override
            public void onAdLoaded(@NonNull com.google.android.gms.ads.interstitial.InterstitialAd receivedInterstitialAd) {
                interstitialAd = receivedInterstitialAd;

                interstitialAd.setFullScreenContentCallback(fullScreenContentCallback);
                interstitialAd.setOnPaidEventListener(interstitialPaidEventListener);

                isInterstitialEnabled = true;

                AdEvents.interstitialAvailable(context, conditionId);
                adMobManager.resetPriorityAdConfigsFillSettings(AdManager.INTERSTITIAL);
            }

            @Override
            public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
                AdManager.getInstance().isFullScreenAdShowing = false;

                AdEvents.interstitialNotAvailable(context, conditionId);

                if (adMobManager.getSelectedInterstitialPriorityConfig() != null) {
                    LoggingUtil.d("Failed to load priority interstitial ad. Error: " + loadAdError.toString());

                    adMobManager.getSelectedInterstitialPriorityConfig().setHasFill(false);
                    adMobManager.requestAdMobInterstitial(context);
                } else {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
                    conditionId = 0;
                }
            }
        };

        adManagerInterstitialAdLoadCallback = new AdManagerInterstitialAdLoadCallback() {
            @Override
            public void onAdLoaded(@NonNull AdManagerInterstitialAd receivedAdManagerInterstitialAd) {
                adManagerInterstitialAd = receivedAdManagerInterstitialAd;

                adManagerInterstitialAd.setFullScreenContentCallback(fullScreenContentCallback);
                adManagerInterstitialAd.setOnPaidEventListener(interstitialPaidEventListener);

                isInterstitialEnabled = true;

                AdEvents.interstitialAvailable(context, conditionId);
                adMobManager.resetPriorityAdConfigsFillSettings(AdManager.INTERSTITIAL);
            }

            @Override
            public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
                super.onAdFailedToLoad(loadAdError);
            }
        };

        fullScreenContentCallback = new FullScreenContentCallback() {
            @Override
            public void onAdDismissedFullScreenContent() {
                AdManager.getInstance().isFullScreenAdShowing = false;

                AdEvents.interstitialDidClose(context, conditionId);
                GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.AdMob, AdManager.INTERSTITIAL, "close");

                interstitialAd = null;
                adManagerInterstitialAd = null;

                conditionId = 0;
                adMobManager.requestAdMobInterstitial(context);
            }

            @Override
            public void onAdFailedToShowFullScreenContent(AdError adError) {
                super.onAdFailedToShowFullScreenContent(adError);

                LoggingUtil.e("Ad failed to display with error: " + adError.toString());

                AdManager.getInstance().isFullScreenAdShowing = false;

                AdEvents.interstitialDidClose(context, conditionId);
                GamedockSDK.getInstance(context).getAdCallbacks().AdFinished(AdManager.AdMob, AdManager.INTERSTITIAL, "close");

                interstitialAd = null;
                adManagerInterstitialAd = null;

                conditionId = 0;
                adMobManager.requestAdMobInterstitial(context);
            }

            @Override
            public void onAdShowedFullScreenContent() {
                AdManager.getInstance().isFullScreenAdShowing = true;

                if (adMobManager.getSelectedInterstitialPriorityConfig() == null) {
                    LoggingUtil.d("Default interstitial displayed");
                    AdEvents.interstitialDidDisplay(context, conditionId, interstitialAdUnitId, false);
                } else {
                    LoggingUtil.d("Priority interstitial displayed");
                    AdEvents.interstitialDidDisplay(context, conditionId, adMobManager.getSelectedInterstitialPriorityConfig().getAdUnitId(), true);
                    adMobManager.getSelectedInterstitialPriorityConfig().setImpressions(adMobManager.getSelectedInterstitialPriorityConfig().getImpressions() - 1);
                }
            }

            @Override
            public void onAdImpression() {
                super.onAdImpression();
                Log.d("Sebi", "Hello?");
            }
        };

        interstitialPaidEventListener = new OnPaidEventListener() {
            @Override
            public void onPaidEvent(AdValue adValue) {
                AdEvents.interstitialDidMonetize(context, adValue.getCurrencyCode(), adValue.getPrecisionType(), adValue.getValueMicros());
            }
        };
    }

    public void start(Context context, String adUnitId, Map<String, String> adMobCustomTargeting, AdMobMediationNetworks adMobMediationNetworks) {
        try {
            //AdMob
            interstitialAdUnitId = adUnitId;
            isInterstitialModuleEnabled = true;
            this.adMobInterstitialCustomTargeting = adMobCustomTargeting;

            adMobManager.requestAdMobInterstitial(context);
        } catch (NoClassDefFoundError | Exception e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
        }
    }

    public void request(final Context context) {
        if (!isInterstitialModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.INTERSTITIAL)) {
            LoggingUtil.d("Requesting an interstitial while one is already showing is not allowed");
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            LoggingUtil.e("Failed to request AdMob Interstitial. Context doesn't belong to an activity.");
            return;
        }

        try {
            adMobManager.getHandler().post(new Runnable() {
                @Override
                public void run() {
                    try {
                        String displayNetwork = "";

                        AdRequest adRequest = null;
                        AdManagerAdRequest adManagerAdRequest = null;

                        String adUnitId = null;
                        boolean isPriority = false;

                        adMobManager.setSelectedInterstitialPriorityConfig(adMobManager.getPriorityAdConfig(AdManager.INTERSTITIAL));

                        if (adMobManager.getSelectedInterstitialPriorityConfig() == null) {
                            LoggingUtil.d("No priority interstitial configuration found. Using default configuration");
                            adMobManager.setSelectedInterstitialPriorityConfig(null);
                            adUnitId = interstitialAdUnitId;
                            adRequest = adMobManager.getAdMobAdRequest(context, adMobInterstitialCustomTargeting);

                            displayNetwork = AdManager.AdMob;
                        } else if (adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                            LoggingUtil.d("Priority interstitial configuration found for AdMob. AdUnitId: " + adMobManager.getSelectedInterstitialPriorityConfig().getAdUnitId());
                            adUnitId = adMobManager.getSelectedInterstitialPriorityConfig().getAdUnitId();
                            adRequest = adMobManager.getAdMobAdRequest(context, adMobInterstitialCustomTargeting);

                            displayNetwork = AdManager.AdMob;
                            isPriority = true;
                        } else if (adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                            LoggingUtil.d("Priority interstitial configuration found for AdManager. AdUnitId: " + adMobManager.getSelectedInterstitialPriorityConfig().getAdUnitId());
                            adUnitId = adMobManager.getSelectedInterstitialPriorityConfig().getAdUnitId();
                            adManagerAdRequest = adMobManager.getAdManagerAdRequest(context, adMobInterstitialCustomTargeting);

                            displayNetwork = AdManager.AdManager;
                            isPriority = true;
                        } else if (adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                            LoggingUtil.d("Priority interstitial ad configuration found for Improve Digital. Placement Ids: " + adMobManager.getSelectedInterstitialPriorityConfig().getVideoPlacementId() + "-" + adMobManager.getSelectedInterstitialPriorityConfig().getStaticPlacementId());
                            GamedockAdsManager.getInstance().requestGamedockAdsInterstitial(context, AdProvider.IMPROVE_DIGITAL, adMobManager.getSelectedInterstitialPriorityConfig().getVideoPlacementId(), adMobManager.getSelectedInterstitialPriorityConfig().getStaticPlacementId(), true, true);

                            return;
                        }

                        interstitialRequestId = UUID.randomUUID().toString();

                        final String finalDisplayNetwork = displayNetwork;
                        final AdRequest finalAdRequest = adRequest;
                        final AdManagerAdRequest finalAdManagerAdRequest = adManagerAdRequest;
                        final String finalAdUnitToSend = adUnitId;
                        final boolean finalIsPriority = isPriority;

                        activity.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (finalDisplayNetwork.equals(AdManager.AdMob)) {
                                    if (interstitialAd != null) {
                                        LoggingUtil.d("Gamedock Interstitial already loaded. Skipping new request. Will play cached interstitial.");
                                        GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.INTERSTITIAL);
                                        AdEvents.interstitialAvailable(context, conditionId);
                                    } else {
                                        AdEvents.interstitialRequested(context, conditionId, finalAdUnitToSend, finalIsPriority);
                                        InterstitialAd.load(context, finalAdUnitToSend, finalAdRequest, interstitialAdLoadCallback);
                                    }
                                } else if (finalDisplayNetwork.equals(AdManager.AdManager)) {
                                    if (adManagerInterstitialAd != null) {
                                        LoggingUtil.d("Gamedock Interstitial already loaded. Skipping new request. Will play cached interstitial.");
                                        GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.INTERSTITIAL);
                                        AdEvents.interstitialAvailable(context, conditionId);
                                    } else {
                                        AdEvents.interstitialRequested(context, conditionId, finalAdUnitToSend, finalIsPriority);
                                        AdManagerInterstitialAd.load(context, finalAdUnitToSend, finalAdManagerAdRequest, adManagerInterstitialAdLoadCallback);
                                    }
                                } else {
                                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
                                }
                            }
                        });
                    } catch (NoClassDefFoundError | Exception e) {
                        e.printStackTrace();
                        GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
                        LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
                    }
                }
            });
        } catch (NoClassDefFoundError | Exception e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock AdMob Module not included! If you want to use AdMob please include the gamedock-sdk-admob dependency");
        }
    }

    public void show(final Context context) {
        if (!isInterstitialModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            return;
        }

        if (AdManager.getInstance().checkAdIsShowing(AdManager.INTERSTITIAL)) {
            LoggingUtil.d("Showing an interstitial while one is already showing is not allowed");
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;

            if (activity == null) {
                throw new NullPointerException();
            }
        } catch (NullPointerException | ClassCastException e) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
            LoggingUtil.e("Failed to show AdMob Interstitial. Context doesn't belong to an activity.");
            return;
        }

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    boolean shouldStartInterstitial = false;
                    String adProvider = "";

                    if (adMobManager.getSelectedInterstitialPriorityConfig() == null) {
                        if (interstitialAd != null) {
                            shouldStartInterstitial = true;
                            adProvider = AdManager.AdMob;
                        }
                    } else if (adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                        if (interstitialAd != null) {
                            shouldStartInterstitial = true;
                            adProvider = AdManager.AdMob;
                        }
                    } else if (adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                        if (adManagerInterstitialAd != null) {
                            shouldStartInterstitial = true;
                            adProvider = AdManager.AdManager;
                        }
                    } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null && adMobManager.getSelectedInterstitialPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                        GamedockAdsManager.getInstance().showGamedockAdsInterstitial(context);
                        return;
                    }

                    if (shouldStartInterstitial) {
                        LoggingUtil.d("Opening AdMob Interstitial Activity");
                        AdManager.getInstance().isFullScreenAdShowing = true;
                        //Special case for Unity Ads (Unity Ads cannot be shown if it is not from the original activity with which AdMob was initialized)
                        if (interstitialAd != null && interstitialAd.getResponseInfo() != null && interstitialAd.getResponseInfo().getMediationAdapterClassName() != null && interstitialAd.getResponseInfo().getMediationAdapterClassName().equals("com.google.ads.mediation.unity.UnityAdapter")) {
                            interstitialAd.show(activity);
                        } else {
                            GamedockSDK.getInstance(context).isShowingChildActivity = true;

                            Intent intent = new Intent(context, AdMobAdActivity.class);
                            intent.putExtra("adType", AdManager.INTERSTITIAL);
                            intent.putExtra("adProvider", adProvider);
                            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                            GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.INTERSTITIAL);

                            context.startActivity(intent);
                        }
                    } else {
                        LoggingUtil.d("No interstitial ad loaded. Stopping current show.");
                        isInterstitialEnabled = false;
                        AdEvents.interstitialNotAvailable(context, conditionId);
                        conditionId = 0;
                        if (adMobManager.getSelectedInterstitialPriorityConfig() != null) {
                            adMobManager.getSelectedInterstitialPriorityConfig().setHasFill(false);
                        }
                        request(context);
                    }
                } catch (Exception e) {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.INTERSTITIAL);
                    AdEvents.interstitialDidNotDisplay(context, conditionId);
                    e.printStackTrace();
                }
            }
        });
    }

    public String getInterstitialRequestId() {
        return interstitialRequestId;
    }

    public void setInterstitialRequestId(String interstitialRequestId) {
        this.interstitialRequestId = interstitialRequestId;
    }

    public InterstitialAd getInterstitialAd() {
        return interstitialAd;
    }

    public AdManagerInterstitialAd getAdManagerInterstitialAd() {
        return adManagerInterstitialAd;
    }

    public int getConditionId() {
        return conditionId;
    }

    public void setConditionId(int conditionId) {
        this.conditionId = conditionId;
    }

    public String getInterstitialAdUnitId() {
        return interstitialAdUnitId;
    }
}
