package io.gamedock.sdk.ads.admob;

import android.app.Activity;
import android.content.Context;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdSize;
import com.google.android.gms.ads.AdValue;
import com.google.android.gms.ads.AdView;
import com.google.android.gms.ads.OnPaidEventListener;
import com.google.android.gms.ads.admanager.AdManagerAdRequest;
import com.google.android.gms.ads.admanager.AdManagerAdView;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.ads.AdManager;
import io.gamedock.sdk.ads.BaseAdManagerImplementation;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.utils.logging.LoggingUtil;

public class AdMobBanner {

    private AdMobManager adMobManager;

    private boolean isBannerModuleEnabled;
    public boolean isBannerEnabled;
    private AdManagerAdView adManagerBannerAd;
    private int publisherBannerAdViewId;
    private AdView bannerAd;
    private int bannerAdViewId;
    private String bannerAdUnitId;
    private AdListener bannerAdListener;
    private OnPaidEventListener bannerPaidEventListener;
    private String bannerRequestId;

    private Map<String, String> adMobBannerCustomTargeting;

    private CountDownLatch countDownLatch;

    private String bannerSize;
    private String bannerPosition;

    public void initialise(final Context context, BaseAdManagerImplementation baseAdManagerImplementation) {
        this.adMobManager = (AdMobManager) baseAdManagerImplementation;
        bannerAdListener = new AdListener() {
            @Override
            public void onAdFailedToLoad(int i) {
                AdEvents.bannerNotAvailable(context);

                if (adMobManager.getSelectedBannerPriorityConfig() != null) {
                    LoggingUtil.d("Failed to load priority banner ad.");

                    if (adManagerBannerAd != null) {
                        adManagerBannerAd.setVisibility(View.GONE);
                    }

                    adMobManager.getSelectedBannerPriorityConfig().setHasFill(false);
                    request(context, bannerPosition, bannerSize);
                } else if (bannerAd != null) {
                    LoggingUtil.d("Failed to load default banner ad.");
                    isBannerEnabled = false;
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                    bannerAd.setVisibility(View.GONE);
                } else {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                }
            }

            @Override
            public void onAdLoaded() {
                isBannerEnabled = true;
                bannerSize = null;
                bannerPosition = null;

                if (bannerAd != null && (!bannerAd.isShown() || !adManagerBannerAd.isShown())) {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.BANNER);
                    AdEvents.bannerAvailable(context);
                    adMobManager.resetPriorityAdConfigsFillSettings(AdManager.BANNER);
                }
            }

            @Override
            public void onAdLeftApplication() {
                AdEvents.bannerDidClick(context);
            }
        };

        bannerPaidEventListener = new OnPaidEventListener() {
            @Override
            public void onPaidEvent(AdValue adValue) {
                AdEvents.bannerDidMonetize(context, adValue.getCurrencyCode(), adValue.getPrecisionType(), adValue.getValueMicros());
            }
        };
    }

    public void start(Context context, String adUnitId, Map<String, String> adMobCustomTargeting, AdMobMediationNetworks adMobMediationNetworks) {
        bannerAdUnitId = adUnitId;
        isBannerModuleEnabled = true;
        this.adMobBannerCustomTargeting = adMobCustomTargeting;

        createDefaultBannerAdViews(context);
    }

    public void request(final Context context, final String position, final String adSize) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        countDownLatch = new CountDownLatch(2);
        createDefaultBannerAdViews(context);

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            LoggingUtil.e("Failed to request AdMob Banner. Context doesn't belong to an activity.");
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        if (adSize != null || position != null) {
            isBannerEnabled = false;
            final ViewGroup viewGroup = (ViewGroup) activity.getWindow().getDecorView().getRootView();

            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    //AdMob
                    if (bannerAd != null) {
                        bannerAd.setVisibility(View.GONE);
                        viewGroup.removeView(bannerAd);
                    }

                    //AdManager
                    if (adManagerBannerAd != null) {
                        adManagerBannerAd.setVisibility(View.GONE);
                        viewGroup.removeView(adManagerBannerAd);
                    }

                    if (adSize != null) {
                        try {
                            //AdMob
                            if (bannerAd != null) {
                                bannerAd.destroy();
                            }

                            bannerAd = new AdView(context);
                            bannerAdViewId = View.generateViewId();
                            bannerAd.setId(bannerAdViewId);

                            //AdManager
                            if (adManagerBannerAd != null) {
                                adManagerBannerAd.destroy();
                            }

                            adManagerBannerAd = new AdManagerAdView(context);
                            publisherBannerAdViewId = View.generateViewId();
                            adManagerBannerAd.setId(publisherBannerAdViewId);

                            AdSize adMobAdSize = AdSize.SMART_BANNER;
                            switch (adSize) {
                                case "BANNER":
                                    adMobAdSize = AdSize.BANNER;
                                    break;
                                case "LEADERBOARD":
                                    adMobAdSize = AdSize.LEADERBOARD;
                                    break;
                                case "FULL_BANNER":
                                    adMobAdSize = AdSize.FULL_BANNER;
                                    break;
                                case "LARGE_BANNER":
                                    adMobAdSize = AdSize.LARGE_BANNER;
                                    break;
                                case "MEDIUM_RECTANGLE":
                                    adMobAdSize = AdSize.MEDIUM_RECTANGLE;
                                    break;
                                case "SMART_BANNER":
                                default:
                                    break;
                            }

                            bannerAd.setAdSize(adMobAdSize);
                            adManagerBannerAd.setAdSizes(adMobAdSize);

                            bannerAd.setAdListener(bannerAdListener);
                            adManagerBannerAd.setAdListener(bannerAdListener);

                            bannerAd.setOnPaidEventListener(bannerPaidEventListener);
                        } catch (Exception e) {
                            e.printStackTrace();
                            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                        }
                    }

                    String finalPosition = null;
                    if (position == null) {
                        finalPosition = "BOTTOM";
                    } else {
                        finalPosition = position;
                    }

                    FrameLayout.LayoutParams bannerParams;
                    switch (finalPosition) {
                        case "TOP":
                            bannerParams = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT, Gravity.CENTER_HORIZONTAL | Gravity.TOP);
                            break;
                        case "BOTTOM":
                        default:
                            bannerParams = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT, Gravity.CENTER_HORIZONTAL | Gravity.BOTTOM);
                            break;
                    }

                    try {
                        if (bannerAd != null) {
                            bannerAd.setLayoutParams(bannerParams);
                            viewGroup.addView(bannerAd, bannerParams);
                            bannerAd.setVisibility(View.GONE);
                        }

                        if (adManagerBannerAd != null) {
                            adManagerBannerAd.setLayoutParams(bannerParams);
                            viewGroup.addView(adManagerBannerAd, bannerParams);
                            adManagerBannerAd.setVisibility(View.GONE);
                        }
                    } catch (Error | Exception e) {
                        e.printStackTrace();
                        GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                    }

                    bannerSize = adSize;
                    bannerPosition = position;

                    if (countDownLatch != null) {
                        countDownLatch.countDown();
                    }
                }
            });
        } else {
            if (countDownLatch != null) {
                countDownLatch.countDown();
            }
        }

        if (isBannerEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(AdManager.BANNER);
            return;
        }

        adMobManager.getHandler().post(new Runnable() {
            @Override
            public void run() {
                try {
                    countDownLatch.await(3, TimeUnit.SECONDS);

                    String displayNetwork = "";

                    AdRequest adRequest = null;
                    AdManagerAdRequest adManagerAdRequest = null;

                    String adUnitToSend = null;
                    boolean isPriority = false;

                    adMobManager.setSelectedBannerPriorityConfig(adMobManager.getPriorityAdConfig(AdManager.BANNER));

                    if (adMobManager.getSelectedBannerPriorityConfig() == null) {
                        LoggingUtil.d("No priority banner ad configuration found. Using default configuration");
                        adMobManager.setSelectedBannerPriorityConfig(null);
                        adUnitToSend = bannerAdUnitId;
                        adRequest = adMobManager.getAdMobAdRequest(context, adMobBannerCustomTargeting);

                        displayNetwork = AdManager.AdMob;
                    } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                        LoggingUtil.d("Priority banner ad configuration found for AdMob. AdUnitId: " + adMobManager.getSelectedBannerPriorityConfig().getAdUnitId());
                        adUnitToSend = adMobManager.getSelectedBannerPriorityConfig().getAdUnitId();
                        adRequest = adMobManager.getAdMobAdRequest(context, adMobBannerCustomTargeting);

                        displayNetwork = AdManager.AdMob;
                        isPriority = true;
                    } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                        LoggingUtil.d("Priority banner ad configuration found for AdManager. AdUnitId: " + adMobManager.getSelectedBannerPriorityConfig().getAdUnitId());
                        adUnitToSend = adMobManager.getSelectedBannerPriorityConfig().getAdUnitId();
                        adManagerAdRequest = adMobManager.getAdManagerAdRequest(context, adMobBannerCustomTargeting);

                        displayNetwork = AdManager.AdManager;
                        isPriority = true;
                    } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                        LoggingUtil.d("Priority banner ad configuration found for Improve Digital. Placement Id: " + adMobManager.getSelectedBannerPriorityConfig().getStaticPlacementId());
                        GamedockAdsManager.getInstance().requestGamedockAdsBanner(context, AdProvider.IMPROVE_DIGITAL, adMobManager.getSelectedBannerPriorityConfig().getStaticPlacementId(), position, adSize, true, true);

                        return;
                    }

                    if (GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerAd() != null) {
                        GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerAd().destroy();
                    }
                    bannerRequestId = UUID.randomUUID().toString();

                    final String finalDisplayNetwork = displayNetwork;
                    final AdRequest finalAdRequest = adRequest;
                    final AdManagerAdRequest finalAdManagerAdRequest = adManagerAdRequest;
                    final String finalAdUnitToSend = adUnitToSend;
                    final boolean finalIsPriority = isPriority;

                    activity.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            try {
                                if (finalDisplayNetwork.equals(AdManager.AdMob)) {
                                    if (bannerAd != null) {
                                        bannerAd.setAdUnitId(finalAdUnitToSend);
                                        AdEvents.bannerRequested(context, finalAdUnitToSend, finalIsPriority);
                                        bannerAd.loadAd(finalAdRequest);
                                    }
                                } else if (finalDisplayNetwork.equals(AdManager.AdManager)) {
                                    if (adManagerBannerAd != null) {
                                        adManagerBannerAd.setAdUnitId(finalAdUnitToSend);
                                        AdEvents.bannerRequested(context, finalAdUnitToSend, finalIsPriority);
                                        adManagerBannerAd.loadAd(finalAdManagerAdRequest);
                                    }
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                            }
                        }
                    });
                } catch (InterruptedException | IllegalStateException | NullPointerException e) {
                    e.printStackTrace();
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                }
            }
        });
    }

    public void show(final Context context) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            LoggingUtil.e("Failed to show AdMob Banner. Context doesn't belong to an activity.");
            return;
        }


        try {
            String displayNetwork = "";
            boolean isPriority = false;

            if (adMobManager.getSelectedBannerPriorityConfig() == null) {
                LoggingUtil.d("Default banner displayed");
                displayNetwork = AdManager.AdMob;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                LoggingUtil.d("Priority banner displayed for AdMob");
                displayNetwork = AdManager.AdMob;
                isPriority = true;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                LoggingUtil.d("Priority banner displayed for AdManager");
                displayNetwork = AdManager.AdManager;
                isPriority = true;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                LoggingUtil.d("Priority banner displayed for Improve Digital");
                GamedockAdsManager.getInstance().showGamedockAdsBanner(context);
                return;
            } else {
                AdEvents.bannerDidNotDisplay(context);
                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                return;
            }

            if (isBannerEnabled) {
                final String finalDisplayNetwork = displayNetwork;
                final boolean finalIsPriority = isPriority;
                activity.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if (finalDisplayNetwork.equals(AdManager.AdMob)) {
                            setBannerVisibility(activity, AdManager.AdMob, View.VISIBLE);
                        } else if (finalDisplayNetwork.equals(AdManager.AdManager)) {
                            setBannerVisibility(activity, AdManager.AdManager, View.VISIBLE);
                        }

                        if (finalIsPriority) {
                            AdEvents.bannerDidDisplay(context, adMobManager.getSelectedBannerPriorityConfig().getAdUnitId(), true);
                            adMobManager.getSelectedBannerPriorityConfig().setImpressions(adMobManager.getSelectedBannerPriorityConfig().getImpressions() - 1);
                        } else {
                            AdEvents.bannerDidDisplay(context, bannerAdUnitId, false);
                        }
                        GamedockSDK.getInstance(context).getAdCallbacks().AdStart(AdManager.BANNER);
                    }
                });
            } else {
                AdEvents.bannerDidNotDisplay(context);
                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            }
        } catch (Exception e) {
            e.printStackTrace();
            AdEvents.bannerDidNotDisplay(context);
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
        }
    }

    public void hide(final Context context) {
        if (!isBannerModuleEnabled) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            return;
        }

        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
            LoggingUtil.e("Failed to hide AdMob Banner. Context doesn't belong to an activity.");
            return;
        }

        try {
            String displayNetwork = "";

            if (adMobManager.getSelectedBannerPriorityConfig() == null) {
                LoggingUtil.d("Default banner hidden");
                displayNetwork = AdManager.AdMob;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdMob)) {
                LoggingUtil.d("Priority banner hidden for AdMob");
                displayNetwork = AdManager.AdMob;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.AdManager)) {
                LoggingUtil.d("Priority banner hidden for AdManager");
                displayNetwork = AdManager.AdManager;
            } else if (adMobManager.getSelectedBannerPriorityConfig().getNetwork().equals(AdManager.ImproveDigital)) {
                LoggingUtil.d("Priority banner hidden for Improve Digital");

                GamedockAdsManager.getInstance().hideGamedockAdsBanner(context);
                return;
            }

            final String finalDisplayNetwork = displayNetwork;
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (finalDisplayNetwork.equals(AdManager.AdMob)) {
                        setBannerVisibility(activity, AdManager.AdMob, View.GONE);
                    } else if (finalDisplayNetwork.equals(AdManager.AdManager)) {
                        setBannerVisibility(activity, AdManager.AdManager, View.GONE);
                    }

                    AdEvents.bannerDidHide(context);
                }
            });
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
    }

    private void setBannerVisibility(Activity activity, String network, int visibility) {
        if (network.equals(AdManager.AdMob)) {
            if (activity.findViewById(bannerAdViewId) != null) {
                activity.findViewById(bannerAdViewId).setVisibility(visibility);
            }

            if (bannerAd != null) {
                bannerAd.setVisibility(visibility);
            }
        } else if (network.equals(AdManager.AdManager)) {
            if (activity.findViewById(publisherBannerAdViewId) != null) {
                activity.findViewById(publisherBannerAdViewId).setVisibility(visibility);
            }

            if (adManagerBannerAd != null) {
                adManagerBannerAd.setVisibility(visibility);
            }
        }
    }

    private void createDefaultBannerAdViews(final Context context) {
        final Activity activity;
        try {
            activity = (Activity) context;
        } catch (NullPointerException | ClassCastException e) {
            LoggingUtil.e("Failed to start AdMob Banner. Context doesn't belong to an activity.");
            return;
        }

        try {
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    final ViewGroup viewGroup = (ViewGroup) activity.getWindow().getDecorView().getRootView();
                    final FrameLayout.LayoutParams bannerParams = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT, Gravity.CENTER_HORIZONTAL | Gravity.BOTTOM);

                    //AdMob banner
                    if (bannerAd != null) {
                        bannerAd.destroy();
                        viewGroup.removeView(bannerAd);
                    }

                    bannerAd = new AdView(context);
                    bannerAdViewId = View.generateViewId();
                    bannerAd.setId(bannerAdViewId);

                    bannerAd.setLayoutParams(bannerParams);

                    bannerAd.setAdSize(AdSize.SMART_BANNER);
                    bannerAd.setAdListener(bannerAdListener);
                    bannerAd.setOnPaidEventListener(bannerPaidEventListener);

                    //AdManager banner
                    if (adManagerBannerAd != null) {
                        adManagerBannerAd.destroy();
                        viewGroup.removeView(adManagerBannerAd);
                    }

                    adManagerBannerAd = new AdManagerAdView(context);
                    publisherBannerAdViewId = View.generateViewId();
                    adManagerBannerAd.setId(publisherBannerAdViewId);

                    adManagerBannerAd.setLayoutParams(bannerParams);

                    adManagerBannerAd.setAdSizes(AdSize.SMART_BANNER);
                    adManagerBannerAd.setAdListener(bannerAdListener);
                    if (bannerAd != null) {
                        try {
                            //AdMob
                            viewGroup.addView(bannerAd, bannerParams);
                            bannerAd.bringToFront();
                            bannerAd.setVisibility(View.GONE);

                            //AdManager
                            viewGroup.addView(adManagerBannerAd, bannerParams);
                            adManagerBannerAd.bringToFront();
                            adManagerBannerAd.setVisibility(View.GONE);
                        } catch (Error | Exception e) {
                            e.printStackTrace();
                            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(AdManager.BANNER);
                        }
                    }

                    if (countDownLatch != null) {
                        countDownLatch.countDown();
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public String getBannerRequestId() {
        return bannerRequestId;
    }

    public String getBannerAdUnitId() {
        return bannerAdUnitId;
    }

    public AdView getBannerAd() {
        return bannerAd;
    }

    public AdManagerAdView getAdManagerBannerAd() {
        return adManagerBannerAd;
    }
}
