package io.gamedock.sdk.ads;

import android.content.Context;
import android.view.View;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

import com.lusfold.spinnerloading.SpinnerLoading;

import java.util.ArrayList;
import java.util.Map;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.ads.internal.GamedockMoreAppsManager;
import io.gamedock.sdk.events.internal.AdvertisementEvent;
import io.gamedock.sdk.models.ads.admob.AdMobMediationNetworks;
import io.gamedock.sdk.models.ads.internal.PriorityAdConfig;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

public class AdManager {
    public static final String AdMob = "AdMob";
    public static final String AdManager = "AdManager";
    public static final String GamedockAds = "GamedockAds";
    public static final String ImproveDigital = "ImproveDigital";
    public static final String DFP = "DFP";
    public static final String BANNER = "banner";
    public static final String INTERSTITIAL = "interstitial";
    public static final String REWARD_VIDEO = "rewardVideo";

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "ads";

    private static volatile AdManager mInstance;

    public boolean isBannerShowing;
    public boolean isFullScreenAdShowing;

    public long nextTimestampInterstitial;

    public static String testDeviceId = null;

    private static ProgressBar progressBar;
    private static SpinnerLoading advancedProgressBar;

    public static AdManager getInstance() {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new AdManager();
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to initialise AdMob. Can be manually called for testing purposes.
     * @param appId The adUnitId needed by AdMob to initialise.
     * @param bannerAdUnitId The banner ad unit id.
     * @param interstitialAdUnitId The interstitial ad unit id.
     * @param interstitialCustomTargeting The custom targeting that can be passed into AdMob for interstitials.
     * @param rewardVideoAdUnitId The reward video ad unit id.
     * @param rewardVideoCustomTargeting The custom targeting that can be passed into AdMob for reward videos.
     * @param adMobMediationNetworks The extra mediation networks information.
     */
    public void startAdMob(Context context, String appId, String bannerAdUnitId, String interstitialAdUnitId, Map<String, String> interstitialCustomTargeting, String rewardVideoAdUnitId, Map<String, String> rewardVideoCustomTargeting, AdMobMediationNetworks adMobMediationNetworks, ArrayList<PriorityAdConfig> priorityAdConfigs) {
        if (AdMobManager.getInstance().initialised) {
            return;
        }

        if (priorityAdsContainsGamedockAds(priorityAdConfigs)) {
            GamedockAdsManager.getInstance().initialise(context);
        }

        AdMobManager.getInstance().initialise(context, appId, bannerAdUnitId, interstitialAdUnitId, interstitialCustomTargeting, rewardVideoAdUnitId, rewardVideoCustomTargeting, adMobMediationNetworks, priorityAdConfigs);
    }

    /**
     * Method used to request a reward video from the Gamedock server.
     */
    public void requestRewardVideo(Context context, String location, String rewardType) {
        if (!FeaturesUtil.isFeatureEnabled(AdMobManager.FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(REWARD_VIDEO);
            return;
        }

        AdvertisementEvent event = new AdvertisementEvent(context);
        event.setRequestRewardVideo();

        if (location != null) {
            event.addCustomData("location", location);
            AdMobManager.getInstance().getAdMobRewardedVideo().setVideoLocation(location);
        }

        if (rewardType != null) {
            event.addCustomData("rewardType", rewardType);
            AdMobManager.getInstance().getAdMobRewardedVideo().setVideoRewardType(rewardType);
        }

        String requestId = UUID.randomUUID().toString();
        AdMobManager.getInstance().getAdMobRewardedVideo().setRewardVideoRequestId(requestId);
        event.addCustomData("requestId", requestId);

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method use to request an interstitial ad directly from AdMob (depending on which is enabled).
     */
    public void requestInterstitial(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(AdMobManager.FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(INTERSTITIAL);
            return;
        }

        if (checkAdIsShowing(INTERSTITIAL)) {
            LoggingUtil.d("Requesting an interstitial while one is already showing is not allowed");
            return;
        }

        //Check if time since last interstitial shown has passed (based on secondsToInterstitial from the backend)
        boolean isAvailableBasedOnTimer = (System.currentTimeMillis() / 1000) > nextTimestampInterstitial;
        if (!isAvailableBasedOnTimer) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(INTERSTITIAL);
            return;
        }

        AdvertisementEvent event = new AdvertisementEvent(context);
        event.setRequestInterstitial();

        if (AdMobManager.getInstance().initialised) {
            if (AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            }
        }

        if (GamedockAdsManager.getInstance().initialised) {
            if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            }
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method used internally in order to request an ad from one of the ad provider. Can also be used for manual testing.
     *
     * @param provider The provider from which the ad will be requested. Can be AdMob.
     * @param adType   The type of ad to be requested. Can be interstitial, rewardVideo or moreApps.
     */
    public void requestAd(Context context, String provider, String adType, int conditionId) {
        LoggingUtil.d(provider + " Ad requested! Type = " + adType);

        try {
            if (adType.equals(REWARD_VIDEO)) {
                AdMobManager.getInstance().getAdMobRewardedVideo().isVideoEnabled = false;
            }

            switch (provider) {
                case AdMob:
                    if (adType.equals(INTERSTITIAL)) {
                        //If request doesn't have a condition id, it has been triggered by the developer
                        //In that case look if an ad is actually available from AdMob and respond to the game accordingly
                        if (conditionId != 0) {
                            AdMobManager.getInstance().getAdMobInterstitial().setConditionId(conditionId);
                            GamedockAdsManager.getInstance().getGamedockAdsInterstitial().setConditionId(conditionId);
                            AdMobManager.getInstance().getAdMobInterstitial().isInterstitialEnabled = true;
                            playInterstitial(context);
                        } else {
                            boolean isInterstitialAvailable = AdMobManager.getInstance().isAdAvailable(context, INTERSTITIAL) || GamedockAdsManager.getInstance().isAdAvailable(context, INTERSTITIAL);
                            if (isInterstitialAvailable) {
                                AdMobManager.getInstance().getAdMobInterstitial().isInterstitialEnabled = true;
                                GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(INTERSTITIAL);
                            } else {
                                GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(INTERSTITIAL);
                                AdMobManager.getInstance().requestAdMobInterstitial(context);
                            }
                        }
                    } else {
                        AdMobManager.getInstance().requestAdMobRewardVideo(context);
                    }

                    break;
            }

            if (GamedockSDK.getInstance(context).isUnitTesting()) {
                if (adType.equals(REWARD_VIDEO)) {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdAvailable(adType);
                } else {
                    GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(adType);
                }
            }
        } catch (NoClassDefFoundError error) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(adType);
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    /**
     * Method used to request more apps from ChartBoost.
     */
    public void requestMoreApps(Context context) {
        GamedockMoreAppsManager.getInstance().requestGamedockMoreApps(context);
    }

    public void requestBannerAd(Context context, String position, String adSize) {
        try {
            if (AdMobManager.getInstance().initialised) {
                AdMobManager.getInstance().requestAdMobBanner(context, position, adSize);
            } else {
                LoggingUtil.e("Failed to request banner ad. Make sure either AdMob is enabled.");
            }

        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    public void showBannerAd(Context context) {
        try {
            if (AdMobManager.getInstance().initialised) {
                AdMobManager.getInstance().showAdMobBanner(context);
            } else {
                LoggingUtil.e("Failed to show banner ad. Make sure either AdMob is enabled.");
            }
        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    public void hideBannerAd(Context context) {
        try {
            if (AdMobManager.getInstance().initialised) {
                AdMobManager.getInstance().hideAdMobBanner(context);
            } else {
                LoggingUtil.e("Failed to hide banner ad. Make sure either AdMob is enabled.");
            }
        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    /**
     * Method used to show a reward video.
     * Can be either AdMob depending on what was initialised and requested.
     */
    public void playVideo(Context context, String location, String rewardType) {
        if (!FeaturesUtil.isFeatureEnabled(AdMobManager.FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(REWARD_VIDEO);
            return;
        }

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo().isVideoEnabled || GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().isVideoEnabled) {
                LoggingUtil.d("Play Video AdMob");
                AdEvents.rewardVideoDidClickPlay(context);
                showAdMob(context,"rewardVideo", location, rewardType);
            } else {
                LoggingUtil.d("Play Video called but AdMob are not enabled.");
            }
        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    /**
     * Method used to show an interstitial ad.
     * Can be either AdMob or ChartBoost depending on what was initialised and requested.
     */
    public void playInterstitial(Context context) {
        if (!FeaturesUtil.isFeatureEnabled(AdMobManager.FEATURE_NAME)) {
            GamedockSDK.getInstance(context).getAdCallbacks().AdNotAvailable(INTERSTITIAL);
            return;
        }

        try {
            if (AdMobManager.getInstance().getAdMobInterstitial().isInterstitialEnabled) {
                LoggingUtil.d("Play Interstitial AdMob");
                showAdMob(context,"interstitial", null, null);
            } else {
                LoggingUtil.d("Play Interstitial called but AdMob interstitials are not enabled.");
            }
        } catch (NoClassDefFoundError error) {
            LoggingUtil.e("One or more modules not included! Check the stacktrace to see which module you are missing!");
        }
    }

    /**
     * Method used to show more apps.
     * Only Gamedock is used.
     */
    public void playMoreApps(Context context) {
        if (GamedockMoreAppsManager.getInstance().isMoreAppsEnabled) {
            GamedockMoreAppsManager.getInstance().showGamedockMoreApps(context);
        } else {
            LoggingUtil.d("Play MoreApps called but Gamedock More Apps not enabled requested yet.");
        }
    }

    /**
     * Method used to show a AdMob ad.
     */
    private void showAdMob(Context context, String adType, String location, String rewardType) {
        if (adType.equals(INTERSTITIAL)) {
            AdMobManager.getInstance().showAdMobInterstitial(context);
        } else {
            if (location != null) {
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoLocation(location);
            }

            if (rewardType != null) {
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoRewardType(rewardType);
            }

            AdMobManager.getInstance().showAdMobRewardVideo(context);
        }
    }

    private boolean priorityAdsContainsGamedockAds(ArrayList<PriorityAdConfig> priorityAdConfigs) {
        boolean containsGamedockAds = false;

        if (priorityAdConfigs == null) {
            return false;
        }

        for (PriorityAdConfig priorityAdConfig : priorityAdConfigs) {
            if (!priorityAdConfig.getNetwork().equals("AdMob") && !priorityAdConfig.getNetwork().equals("AdManager")) {
                containsGamedockAds = true;
                break;
            }
        }

        return containsGamedockAds;
    }

    public boolean isAdAvailable(Context context, String adType) {
        if (!FeaturesUtil.isFeatureEnabled(AdMobManager.FEATURE_NAME)) {
            return false;
        }

        if (AdMobManager.getInstance().initialised || GamedockAdsManager.getInstance().initialised) {
            return AdMobManager.getInstance().isAdAvailable(context, adType) || GamedockAdsManager.getInstance().isAdAvailable(context, adType);
        }

        return false;
    }

    public boolean checkAdIsShowing(String adType) {
        switch (adType) {
            case BANNER:
                return isBannerShowing;
            case INTERSTITIAL:
            case REWARD_VIDEO:
                return isFullScreenAdShowing;
        }

        return false;
    }

    /**
     * Checks if personalised ads have been enabled by the user by checking for the priv value.
     *
     * @param context The context of the Activity.
     * @return If personalised ads have been enabled.
     */
    public boolean checkPersonalisedAdsPermission(Context context) {

        int priv = GamedockSDK.getInstance(context).getStorageUtil().getInt(StorageUtil.Keys.GDPRStatus, -1);

        boolean withPersonalisedAds = false;

        switch (priv) {
            case 2:
                withPersonalisedAds = true;
                break;
            case 3:
                withPersonalisedAds = true;
                break;
        }

        return withPersonalisedAds;
    }

    /**
     * Method that shows a circular progress bar using the standard android layout.
     *
     * @param context The activity context.
     * @return The generated progress bar. {@link ProgressBar}
     */
    public ProgressBar showBasicSpinner(Context context) {
        LoggingUtil.d("Showing basic spinner!!");

        progressBar = new ProgressBar(context, null, android.R.attr.progressBarStyleLarge);
        RelativeLayout.LayoutParams plp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        plp.addRule(RelativeLayout.CENTER_IN_PARENT);
        progressBar.setLayoutParams(plp);
        progressBar.setIndeterminate(true);
        progressBar.setVisibility(View.VISIBLE);

        return progressBar;
    }

    /**
     * Method that hides the generated circular progress bar after the ad has finished showing.
     */
    public void hideSpinner() {
        LoggingUtil.d("Hiding spinner!!");
        if (advancedProgressBar != null) {
            advancedProgressBar.setVisibility(View.GONE);
            advancedProgressBar = null;
        }

        if (progressBar != null) {
            progressBar.setVisibility(View.GONE);
            progressBar = null;
        }
    }

    /**
     * Method that shows a circular progress bar using the custom layout.
     *
     * @param context The activity context.
     * @return The generated progress bar.
     */
    public SpinnerLoading showAdvanceSpinner(Context context) {
        LoggingUtil.d("Showing advanced spinner!!");

        try {
            advancedProgressBar = new SpinnerLoading(context);
            RelativeLayout.LayoutParams plp = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
            plp.addRule(RelativeLayout.CENTER_IN_PARENT);
            advancedProgressBar.setLayoutParams(plp);
            advancedProgressBar.setVisibility(View.VISIBLE);

            return advancedProgressBar;

        } catch (NoClassDefFoundError | NullPointerException e) {
            LoggingUtil.d("Could not show advanced spinner! Falling back to basic spinner!");
            return null;
        }
    }
}
