package io.gamedock.sdk.ads;


import android.content.Context;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.models.ads.internal.MediationInfo;

/**
 * Class that handles all the internal events that are fired in regards to ads.
 * Handles events for interstitials, reward videos and more apps.
 */
public class AdEvents {

    public static final String AdvertisementInitialized = "advertisementInitialized";

    public static final String BannerRequested = "bannerRequested";
    public static final String BannerDidDisplay = "bannerDidDisplay";
    public static final String BannerDidNotDisplay = "bannerDidNotDisplay";
    public static final String BannerDidHide = "bannerDidHide";
    public static final String BannerDidClick = "bannerDidClick";
    public static final String BannerAvailable = "bannerAvailable";
    public static final String BannerNotAvailable = "bannerNotAvailable";
    public static final String BannerDidMonetize = "bannerDidMonetize";

    public static final String InterstitialRequested = "interstitialRequested";
    public static final String InterstitialDidClose = "interstitialDidClose";
    public static final String InterstitialDidDisplay = "interstitialDidDisplay";
    public static final String InterstitialDidClick = "interstitialDidClick";
    public static final String InterstitialAvailable = "interstitialAvailable";
    public static final String InterstitialNotAvailable = "interstitialNotAvailable";
    public static final String InterstitialDidNotDisplay = "interstitialDidNotDisplay";
    public static final String InterstitialDidMonetize = "interstitialDidMonetize";

    public static final String RewardedVideoRequested = "rewardedVideoRequested";
    public static final String RewardVideoDidClose = "rewardedVideoDidClose";
    public static final String RewardVideoDidDisplay = "rewardedVideoDidDisplay";
    public static final String RewardVideoDidDismiss = "rewardedVideoDidDismiss";
    public static final String RewardVideoAvailable = "rewardedVideoAvailable";
    public static final String RewardVideoNotAvailable = "rewardedVideoNotAvailable";
    public static final String RewardVideoDidNotDisplay = "rewardedVideoDidNotDisplay";
    public static final String RewardVideoDidClickPlay = "rewardedVideoDidClickPlay";
    public static final String RewardVideoDidClick = "rewardedVideoDidClick";
    public static final String RewardVideoDidMonetize = "rewardedVideoDidMonetize";

    private static final String MoreAppsDidClose = "moreAppsDidClose";
    private static final String MoreAppsDidDisplay = "moreAppsDidDisplay";
    private static final String MoreAppsDidClick = "moreAppsDidClick";

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void advertisementInitialized(Context context, ArrayList<MediationInfo> mediationInfoList) {
        Event event = new Event(context);
        event.setName(AdvertisementInitialized);

        JSONArray mediationJSONArray = new JSONArray();
        try {
            for (MediationInfo mediationInfo : mediationInfoList) {
                JSONObject mediationInfoJSON = new JSONObject();
                mediationInfoJSON.put("name", mediationInfo.name);
                mediationInfoJSON.put("latency", mediationInfo.latency);
                mediationInfoJSON.put("status", mediationInfo.status);

                mediationJSONArray.put(mediationInfoJSON);
            }
        } catch (JSONException ignore) {
        }

        event.addCustomData("mediations", mediationJSONArray);

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerRequested(Context context, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(BannerRequested);

        try {
            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }

            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerDidDisplay(Context context, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(BannerDidDisplay);

        try {
            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }
            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            if (AdMobManager.getInstance().getBannerAd() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerBannerAd() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerDidNotDisplay(Context context) {
        Event event = new Event(context);
        event.setName(BannerDidNotDisplay);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            if (AdMobManager.getInstance().getBannerAd() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerBannerAd() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerDidHide(Context context) {
        Event event = new Event(context);
        event.setName(BannerDidHide);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerAvailable(Context context) {
        Event event = new Event(context);
        event.setName(BannerAvailable);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            if (AdMobManager.getInstance().getBannerAd() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerBannerAd() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerNotAvailable(Context context) {
        Event event = new Event(context);
        event.setName(BannerNotAvailable);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            if (AdMobManager.getInstance().getBannerAd() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerBannerAd() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an banner has been clicked and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerDidClick(Context context) {
        Event event = new Event(context);
        event.setName(BannerDidClick);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            if (AdMobManager.getInstance().getBannerAd() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerBannerAd() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerBannerAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsBanner().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a banner has monetized and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void bannerDidMonetize(Context context, String currencyCode, int precisionType, long valueMicros) {
        Event event = new Event(context);
        event.setName(BannerDidMonetize);

        try {
            if (AdMobManager.getInstance().getAdMobBanner() != null && AdMobManager.getInstance().getAdMobBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobBanner().getBannerRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsBanner() != null && GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsBanner().getBannerRequestId());
            }

            event.addCustomData("currencyCode", currencyCode);
            event.addCustomData("precisionType", precisionType);
            event.addCustomData("valueMicros", valueMicros);
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialRequested(Context context, int condition, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(InterstitialRequested);

        try {
            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }

            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialDidClose(Context context, int condition) {
        Event event = new Event(context);
        event.setName(InterstitialDidClose);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }


            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial is available and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialAvailable(Context context, int condition) {
        Event event = new Event(context);
        event.setName(InterstitialAvailable);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            if (AdMobManager.getInstance().getInterstitialAd() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerInterstitialAd() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider().toString());
            }

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial is available and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialNotAvailable(Context context, int condition) {
        Event event = new Event(context);
        event.setName(InterstitialNotAvailable);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            if (AdMobManager.getInstance().getInterstitialAd() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerInterstitialAd() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider().toString());
            }

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been displayed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialDidDisplay(Context context, int condition, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(InterstitialDidDisplay);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }
            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getInterstitialAd() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerInterstitialAd() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider().toString());
            }

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has not been displayed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialDidNotDisplay(Context context, int condition) {
        Event event = new Event(context);
        event.setName(InterstitialDidNotDisplay);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            if (AdMobManager.getInstance().getInterstitialAd() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerInterstitialAd() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider().toString());
            }

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been clicked and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialDidClick(Context context, int condition) {
        Event event = new Event(context);
        event.setName(InterstitialDidClick);

        try {
            if (condition != 0) {
                event.addCustomData("conditionId", condition);
            }

            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }

            if (AdMobManager.getInstance().getInterstitialAd() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (AdMobManager.getInstance().getAdManagerInterstitialAd() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo() != null && AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getAdManagerInterstitialAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has monetized and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void interstitialDidMonetize(Context context, String currencyCode, int precisionType, long valueMicros) {
        Event event = new Event(context);
        event.setName(InterstitialDidMonetize);

        try {
            if (AdMobManager.getInstance().getAdMobInterstitial() != null && AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobInterstitial().getInterstitialRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsInterstitial() != null && GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsInterstitial().getInterstitialRequestId());
            }

            event.addCustomData("currencyCode", currencyCode);
            event.addCustomData("precisionType", precisionType);
            event.addCustomData("valueMicros", valueMicros);
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when an interstitial has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardedVideoRequested(Context context, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(RewardedVideoRequested);

        try {
            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }

            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidClose(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoDidClose);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoLocation(null);

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());

                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoRewardType(null);
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video has been displayed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidDisplay(Context context, String adUnitId, boolean isPriority) {
        Event event = new Event(context);
        event.setName(RewardVideoDidDisplay);

        try {
            if (adUnitId != null) {
                event.addCustomData("adUnitId", adUnitId);
            }
            event.addCustomData("isPriority", isPriority);

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoLocation(null);

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());

                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoRewardType(null);
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            if (AdMobManager.getInstance().getRewardedVideoAd() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video has been dismissed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidDismiss(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoDidDismiss);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoLocation(null);


                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());

                }
                AdMobManager.getInstance().getAdMobRewardedVideo().setVideoRewardType(null);
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video is available and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoAvailable(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoAvailable);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());
                }
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            if (AdMobManager.getInstance().getRewardedVideoAd() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video is not available and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoNotAvailable(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoNotAvailable);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());
                }
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            if (AdMobManager.getInstance().getRewardedVideoAd() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video could not be displayed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidNotDisplay(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoDidNotDisplay);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());
                }
            }


            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            if (AdMobManager.getInstance().getRewardedVideoAd() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a reward video play button was clicked and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidClickPlay(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoDidClickPlay);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());
                }
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when the rewarded video ad was clicked/opened and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidClick(Context context) {
        Event event = new Event(context);
        event.setName(RewardVideoDidClick);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null) {
                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation() != null) {
                    event.addCustomData("location", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoLocation());
                }

                if (AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType() != null) {
                    event.addCustomData("rewardType", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoRewardType());
                }
            }

            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            if (AdMobManager.getInstance().getRewardedVideoAd() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo() != null && AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName() != null) {
                event.addCustomData("mediation", AdMobManager.getInstance().getRewardedVideoAd().getResponseInfo().getMediationAdapterClassName());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider() != null) {
                event.addCustomData("mediation", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getCurrentProvider().toString());
            }
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when a rewarded video has monetized and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void rewardVideoDidMonetize(Context context, String currencyCode, int precisionType, long valueMicros) {
        Event event = new Event(context);
        event.setName(RewardVideoDidMonetize);

        try {
            if (AdMobManager.getInstance().getAdMobRewardedVideo() != null && AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId() != null) {
                event.addCustomData("requestId", AdMobManager.getInstance().getAdMobRewardedVideo().getRewardVideoRequestId());
            } else if (GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo() != null && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardedVideoAd() != null) {
                event.addCustomData("requestId", GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().getRewardVideoRequestId());
            }

            event.addCustomData("currencyCode", currencyCode);
            event.addCustomData("precisionType", precisionType);
            event.addCustomData("valueMicros", valueMicros);
        } catch (Exception | Error e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when more apps has been closed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void moreAppsDidClose(Context context) {
        Event event = new Event(context);
        event.setName(MoreAppsDidClose);

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when more apps has been displayed and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void moreAppsDidDisplay(Context context) {
        Event event = new Event(context);
        event.setName(MoreAppsDidDisplay);

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that creates an event for when more apps has been clicked and sends it to the backend.
     *
     * @param context The activity context.
     */
    public static void moreAppsDidClick(Context context) {
        Event event = new Event(context);
        event.setName(MoreAppsDidClick);

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

}