package io.gamedock.sdk.ads;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.ads.admob.AdMobAdActivity;
import io.gamedock.sdk.ads.admob.AdMobManager;
import io.gamedock.sdk.ads.internal.GamedockAdsAdActivity;
import io.gamedock.sdk.ads.internal.GamedockAdsManager;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Ad Networks Callbacks.
 * Sends the different status that the ads going through: Available, Not Available, Started, Finished.
 * Handles the logic to create a circular progress bar before showing an ad.
 */
public class AdCallbacks {

    private OnAdsListener adsListener;

    public AdCallbacks() {
        adsListener = null;
    }

    public AdCallbacks(OnAdsListener adsListener) {
        this.adsListener = adsListener;
    }

    public void AdsInitialized() {
        if (this.adsListener != null) {
            this.adsListener.AdsInitialized();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "AdsInitialized", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called by the ad providers when an ad is available.
     * Fires the callback either to the native implementation or Unity depending on which one implements the SDK.
     *
     * @param type The ad type that is available.
     */
    public void AdAvailable(String type) {
        if (this.adsListener != null) {
            this.adsListener.AdAvailable(type);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "AdAvailable", type);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnAdsListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called by the ad providers when an ad is not available.
     * Fires the callback either to the native implementation or Unity depending on which one implements the SDK.
     *
     * @param type The ad type that is not available.
     */
    public void AdNotAvailable(String type) {
        LoggingUtil.d("AdNotAvailable Type: " + type);
        if (this.adsListener != null) {
            this.adsListener.AdNotAvailable(type);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "AdNotAvailable", type);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnAdsListener in order to receive information or something went wrong with Unity");
            }
        }

        if (type.toLowerCase().trim().equals("interstitial")) {
            AdMobManager.getInstance().getAdMobInterstitial().isInterstitialEnabled = false;
            GamedockAdsManager.getInstance().getGamedockAdsInterstitial().isInterstitialEnabled = false;
        } else if (type.toLowerCase().trim().equals("rewardvideo")) {
            AdMobManager.getInstance().getAdMobRewardedVideo().isVideoEnabled = false;
            GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().isVideoEnabled = false;
        }
    }

    /**
     * Method called by the SDK when an ad has started.
     * Fires the callback either to the native implementation or Unity depending on which one implements the SDK.
     */
    public void AdStart(String type) {
        if (this.adsListener != null) {
            this.adsListener.AdStart(type);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "AdStart", type);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called by the ad providers when an ad is finished.
     * Fires the callback either to the native implementation or Unity depending on which one implements the SDK.
     * Passes any reward gained to the player.
     *
     * @param network The network for witch the ad belongs to. Can be AdMob or Chartboost.
     * @param adType  The ad type that has finished showing.
     * @param reason  The reason with which the ad has finished. Used mostly by Fyber to know if the reward should be given.
     */
    public void AdFinished(String network, String adType, String reason) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("network", network);
            jsonObject.put("type", adType);
            jsonObject.put("reason", reason);

            switch (network) {
                case AdManager.AdMob:
                    if (adType.equals(AdManager.REWARD_VIDEO) && AdMobManager.getInstance().getAdMobRewardedVideo().getVideoReward() != null) {
                        jsonObject.put("reward", AdMobManager.getInstance().getAdMobRewardedVideo().getVideoReward());
                        AdMobManager.getInstance().getAdMobRewardedVideo().setVideoReward(null);
                    }

                    LoggingUtil.d("AdMob ad finished");

                    if (adType.equals(AdManager.INTERSTITIAL)) {
                        AdMobManager.getInstance().getAdMobInterstitial().isInterstitialEnabled = false;
                        AdMobManager.getInstance().setSelectedInterstitialPriorityConfig(null);
                    } else {
                        AdMobManager.getInstance().getAdMobRewardedVideo().isVideoEnabled = false;
                        AdMobManager.getInstance().setSelectedRewardVideoPriorityConfig(null);
                    }
                    AdManager.getInstance().hideSpinner();
                    if (AdMobAdActivity.activity != null) {
                        AdMobAdActivity.activity.finish();
                    }
                    break;
                case AdManager.GamedockAds:
                    if (adType.equals(AdManager.REWARD_VIDEO) && GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().hasEarnedReward()) {
                        jsonObject.put("reward", new JSONObject());
                        GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().setEarnedReward(false);
                    }

                    LoggingUtil.d("AdMob ad finished");

                    if (adType.equals(AdManager.INTERSTITIAL)) {
                        GamedockAdsManager.getInstance().getGamedockAdsInterstitial().isInterstitialEnabled = false;
                        AdMobManager.getInstance().setSelectedInterstitialPriorityConfig(null);
                    } else {
                        GamedockAdsManager.getInstance().getGamedockAdsRewardedVideo().isVideoEnabled = false;
                        AdMobManager.getInstance().setSelectedRewardVideoPriorityConfig(null);
                    }
                    AdManager.getInstance().hideSpinner();
                    if (GamedockAdsAdActivity.activity != null) {
                        GamedockAdsAdActivity.activity.finish();
                    }
                    break;
            }
            if (this.adsListener != null) {
                this.adsListener.AdFinished(jsonObject.toString());
            } else {
                LoggingUtil.d("Sending AdFinished to Unity");
                UnityPlayer.UnitySendMessage("GamedockSDK", "AdFinished", jsonObject.toString());
            }
        } catch (NullPointerException | NoClassDefFoundError | JSONException e) {
            LoggingUtil.w("You need to register the OnAdsListener in order to receive information or something went wrong with Unity");
        }
    }
}
