package ru.foodtechlab.lib.auth.integration.restapi.feign.commons;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.rcore.domain.security.exceptions.AuthorizationErrorReason;
import com.rcore.rest.api.commons.exception.HttpCommunicationException;
import feign.Response;
import feign.RetryableException;
import feign.codec.ErrorDecoder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import io.foodtechlab.exceptionhandler.core.Error;
import io.foodtechlab.exceptionhandler.core.ErrorApiResponse;
import ru.foodtechlab.lib.auth.integration.core.AccessTokenService;

import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;

@Slf4j
@RequiredArgsConstructor
public class AuthorizationErrorDecoder implements ErrorDecoder {

    protected final ObjectMapper mapper;
    protected final AccessTokenService accessTokenService;

    protected List<String> reasonsForRetry = Arrays.asList(
            AuthorizationErrorReason.ACCESS_TOKEN_EXPIRED.name(),
            AuthorizationErrorReason.ACCESS_TOKEN_INACTIVATED.name(),
            AuthorizationErrorReason.ACCESS_TOKEN_REFRESHED.name(),
            AuthorizationErrorReason.ACCESS_TOKEN_NOT_EXIST.name(),
            AuthorizationErrorReason.ACCESS_TOKEN_MODIFIED.name(),
            AuthorizationErrorReason.ACCESS_TOKEN_MALFORMED.name()
    );

    @SneakyThrows
    @Override
    public Exception decode(String s, Response response) {
        Reader reader = response.body().asReader(StandardCharsets.UTF_8);
        String result = IOUtils.toString(reader);
        mapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        ErrorApiResponse<Error> errorResponse = mapper.readValue(result, new TypeReference<>() {
        });
        response.close();
        reader.close();

        //Проверяем ошибки авторизации
        if (response.status() == 401 || response.status() == 403) {
            if (!errorResponse.getErrors().isEmpty()) {
                String reason = errorResponse.getErrors().get(0).getReason();
                if (reasonsForRetry.contains(reason)) {
                    //Дропаем токен
                    accessTokenService.clearToken(response.request().url());
                    //Повторяем запрос
                    throw new RetryableException(response.status(), response.reason(), response.request().httpMethod(), null, response.request());
                }
            }
        }

        return decode(errorResponse);
    }

    public Exception decode(ErrorApiResponse<Error> errorApiResponse) {
        return new HttpCommunicationException(errorApiResponse);
    };
}
